<?php
/**
 * WP Staging | Restore.
 *
 * A standalone script to extract and restore backups.
 * This is a compressed, compiled script based on PHP, JS code, CSS and HTML.
 * If you are a developer who would like to get your hands on the sources of this file, please contact us at support@wp-staging.com.
 *
 * Version      : 1.0.1
 * Build Id     : a106832237f1
 * Build Date   : Jun 10, 2024 15:59:26 UTC
 * Support      : https://wp-staging.com/support/
 */
namespace { !getenv('wpstg-restorer-as-library') && exit; /**@wpstg-restorer-halt**/ }
namespace {
    if (version_compare(PHP_VERSION, '7.0', '<')) {
        exit("WP Staging Restore requires at least PHP version 7.0, current version " . PHP_VERSION . ".\n");
    }
    if (!getenv('wpstg-restorer-as-library') && (defined('ABSPATH') || defined('WPSTG_RESTORER'))) {
        exit("WP Staging Restore should run as a standalone.\n");
    }
    define('WPSTG_RESTORER', true);
    date_default_timezone_set('UTC');
    final class WPStagingRestorer
    {
        const MAX_MEMORY = 268435456;
        const MAX_TIMEOUT = 180;
        const MAX_TIMEOUT_EXTRACT = 60;
        const MAX_TIMEOUT_RESTORE = 60;
        const CHMOD_DIR = 0755;
        const CHMOD_FILE = 0644;
        const KB_IN_BYTES = 1024;
        const MB_IN_BYTES = 1048576;
        const GB_IN_BYTES = 1073741824;
        private $appFile = 'wpstg-restore.php';
        private $buildId = 'a106832237f1';
        private $version = '1.0.1';
        private $backupVersion = '2.0.0';
        private $backupDir = 'wp-staging/backups';
        private $rootPath = null;
        private $uploadPath = null;
        private $backupPath = null;
        private $tmpPath = null;
        private $cachePath = null;
        private $logFile = null;
        private $dataServer = [];
        private $dataCookie = [];
        private $dataPost = [];
        private $dataGet = [];
        private $dataRequest = [];
        private $error = [];
        private $timerStart = null;
        private $maxProcessingTime = 10;
        private $wpCoreHandle = null;
        private $accessHandle = null;
        private $activateHandle = null;
        private $extractorHandle = null;
        private $restorerHandle = null;
        private $cacheHandle = null;
        private $fileHandle = null;
        private $withIdentifierHandle = null;
        private $viewHandle = null;
        private $searchReplaceHandle = null;
        public function __construct()
        {
            $this->timerStart = microtime(true);
            $this->rootPath   = realpath(__DIR__);
            $this->tmpPath    = $this->rootPath . '/wpstg-restore';
            $this->cachePath  = $this->tmpPath . '/cache';
            $this->uploadPath = $this->rootPath . '/wp-content/uploads';
            $this->backupPath = $this->uploadPath . '/' . $this->backupDir;
            $this->logFile    = $this->tmpPath . '/restore.log';
            $this->captureFatalError();
            $this->setMaxResource();
        }
        public function setMeta($key, $value)
        {
            if (property_exists($this, $key)) {
                $this->{$key} = $value;
            }
        }
        public function getMeta()
        {
            return (object)[
                'timerStart'        => $this->timerStart,
                'appFile'           => $this->appFile,
                'buildId'           => $this->buildId,
                'version'           => $this->version,
                'backupVersion'     => $this->backupVersion,
                'backupDir'         => $this->backupDir,
                'rootPath'          => $this->rootPath,
                'uploadPath'        => $this->uploadPath,
                'backupPath'        => $this->backupPath,
                'tmpPath'           => $this->tmpPath,
                'cachePath'         => $this->cachePath,
                'maxProcessingTime' => $this->maxProcessingTime,
                'dataServer'        => $this->dataServer,
                'dataCookie'        => $this->dataCookie,
                'dataPost'          => $this->dataPost,
                'dataGet'           => $this->dataGet,
                'dataRequest'       => $this->dataRequest,
            ];
        }
        public function getHandle(string $caller, $useHandle = null)
        {
            $handles = [
                'access'         => $this->accessHandle,
                'activate'       => $this->activateHandle,
                'cache'          => $this->cacheHandle,
                'file'           => $this->fileHandle,
                'wpcore'         => $this->wpCoreHandle,
                'extractor'      => $this->extractorHandle,
                'restorer'       => $this->restorerHandle,
                'withIdentifier' => $this->withIdentifierHandle,
                'searchReplace'  => $this->searchReplaceHandle
            ];
            $callerKey = strtolower(str_replace('WpstgRestorer\\', '', $caller));
            if (empty($useHandle) && array_key_exists($callerKey, $handles)) {
                unset($handles[$callerKey]);
            }
            $useHandle = array_fill_keys((array)$useHandle, 1);
            if (array_key_exists($callerKey, $handles) && array_key_exists($callerKey, $useHandle)) {
                throw new \LogicException(sprintf('Invalid caller: %s', $caller));
            }
            if ($handleKeys = array_intersect_key($handles, $useHandle)) {
                $handles = $handleKeys;
            }
            foreach ($handles as $name => $object) {
                if ($object === null) {
                    $classHandle    = new \ReflectionClass("WpstgRestorer\\" . $name);
                    $handles[$name] = $classHandle->newInstance($this);
                }
            }
            return (object)$handles;
        }
        private function hasHttps(): bool
        {
            if (!empty($this->dataServer['HTTP_CF_VISITOR'])) {
                $cfVisitorObject = json_decode($this->dataServer['HTTP_CF_VISITOR']);
                if (isset($cfVisitorObject->schema) && $cfVisitorObject->schema === 'https') {
                    return true;
                }
            }
            if (!empty($this->dataServer['HTTP_X_FORWARDED_PROTO']) && $this->dataServer['HTTP_X_FORWARDED_PROTO'] === 'https') {
                return true;
            }
            if (!empty($this->dataServer['HTTPS']) && in_array(strtolower($this->dataServer['HTTPS']), ['on', '1'])) {
                return true;
            }
            if (!empty($this->dataServer['SERVER_PORT']) && (int)$this->dataServer['SERVER_PORT'] === 443) {
                return true;
            }
            return false;
        }
        public function siteUrl(): string
        {
            if (empty($this->dataServer['HTTP_HOST']) || empty($this->dataServer['SCRIPT_FILENAME']) || empty($this->dataServer['PHP_SELF']) || empty($this->dataServer['REQUEST_URI'])) {
                return '';
            }
            $schema            = $this->hasHttps() ? 'https://' : 'http://';
            $url               = $schema . $this->dataServer['HTTP_HOST'];
            $scriptFilenameDir = dirname($this->dataServer['SCRIPT_FILENAME']);
            $path              = '';
            if ($this->rootPath === $scriptFilenameDir . '/') {
                $path = preg_replace('@/[^/]*$@i', '', $this->dataServer['PHP_SELF']);
                return rtrim($url . $path, '/');
            }
            if (strpos($this->rootPath, $scriptFilenameDir) !== false) {
                $subDirectory = substr($this->rootPath, strpos($this->rootPath, $scriptFilenameDir) + strlen($scriptFilenameDir));                $path         = preg_replace('@/[^/]*$@i', '', $this->dataServer['REQUEST_URI']) . $subDirectory;
            } else {
                $path = $this->dataServer['REQUEST_URI'];
            }
            return rtrim($url . $path, '/');
        }
        public function userAgent(): string
        {
            $url = $this->siteUrl();
            if (empty($url)) {
                $url = 'https://wp-staging.com/';
            }
            return 'Mozilla/5.0 (compatible; wpstg-restorer/' . $this->version . '; +' . $url . ')';
        }
        private function requirementCheck(): bool
        {
            if (!is_writable($this->rootPath)) {
                $this->error['rootpath-writable'] = 'Current working directory is not writable.';
            }
            if (!class_exists('ZipArchive')) {
                $this->error['zip-ext'] = 'PHP ZipArchive extension is not available.';
            }
            if (!extension_loaded('curl') || !function_exists('curl_init')) {
                $this->error['curl-ext'] = 'PHP cURL extension is not available.';
            }
            if (!extension_loaded('mysqli') || !class_exists('mysqli')) {
                $this->error['mysqli-ext'] = 'PHP mysqli extension is not available.';
            }
            return empty($this->error);
        }
        public function getBootupError(): array
        {
            return $this->error;
        }
        public function addBootupError(string $key, string $text): array
        {
            $this->error[$key] = $text;
            return $this->error;
        }
        private function createWorkingDir(): bool
        {
            if (!empty($this->error)) {
                return false;
            }
            clearstatcache();
            if (is_dir($this->tmpPath) && is_dir($this->cachePath)) {
                return true;
            }
            if (!$this->mkdir($this->tmpPath)) {
                $this->error['tmp-dir'] = "Can't create working directory";
                return false;
            }
            if (!$this->mkdir($this->cachePath)) {
                $this->error['cache-dir'] = "Can't create cache directory";
                return false;
            }
            $this->fileHandle->preventAccessToDirectory($this->tmpPath);
            $this->fileHandle->preventAccessToDirectory($this->cachePath);
            return true;
        }
        public function setDateTime(\DateTime $dateTime): string
        {
            $defaultDateFormat = 'M j, Y';
            $defaultTimeFormat = 'H:i:s';
            if (!function_exists('get_date_from_gmt') || !function_exists('get_option')) {
                return $dateTime->format($defaultDateFormat . ' ' . $defaultTimeFormat) . ' UTC';
            }
            if (!($dateFormat = get_option('date_format'))) {
                $dateFormat = $defaultDateFormat;
            }
            $dateFormat = str_replace('F', 'M', $dateFormat);
            if (!($timeFormat = get_option('time_format'))) {
                $timeFormat = $defaultTimeFormat;
            }
            return get_date_from_gmt($dateTime->format('Y-m-d H:i:s'), $dateFormat . ' ' . $timeFormat);
        }
        private function isFunctionDisabled(string $name): bool
        {
            static $disableFunctions = [];
            if (empty($disableFunctions)) {
                $disableFunctions = array_map(function ($input) {
                    return trim($input);
                }, explode(',', ini_get('disable_functions')));
            }
            return in_array($name, $disableFunctions);
        }
        private function convertTobytes(string $value): int
        {
            $value = strtolower(trim($value));
            $bytes = (int) $value;
            if (false !== strpos($value, 'g')) {
                $bytes *= self::GB_IN_BYTES;
            } elseif (false !== strpos($value, 'm')) {
                $bytes *= self::MB_IN_BYTES;
            } elseif (false !== strpos($value, 'k')) {
                $bytes *= self::KB_IN_BYTES;
            }
            return min($bytes, PHP_INT_MAX);
        }
        public function maxMemoryLimit(int $bytes = 0): int
        {
            static $memoryLimit;
            if (isset($memoryLimit) && (int)$bytes === 0) {
                return (int)$memoryLimit;
            }
            $memoryLimit = $this->convertTobytes(ini_get('memory_limit'));
            $bytes       = (int)($bytes > 0 ? $bytes : self::MAX_MEMORY);
            if ($bytes > $memoryLimit) {
                if ($bytes < PHP_INT_MAX) {
                    $bytes += self::KB_IN_BYTES;                }
                ini_set('memory_limit', $bytes);
            }
            $memoryLimit = $this->convertTobytes(ini_get('memory_limit'));
            return $memoryLimit;
        }
        public function getMemoryLimit(): int
        {
            return $this->maxMemoryLimit();
        }
        public function maxExecutionTime(int $second = 0): int
        {
            static $maxExecutionTime;
            if (isset($maxExecutionTime) && (int)$second === 0) {
                return $maxExecutionTime;
            }
            $maxExecutionTime = (int)ini_get('max_execution_time');
            $second           = (int)( $second > 0 ? $second : self::MAX_TIMEOUT );
            if ($second > 0 && $maxExecutionTime > 0 && !$this->isFunctionDisabled('set_time_limit')) {
                $second += 1;
                set_time_limit($second);
                $maxExecutionTime = (int)ini_get('max_execution_time');
            }
            if ($maxExecutionTime > 10) {
                $maxExecutionTime -= 1;
            }
            return $maxExecutionTime;
        }
        public function isMaxExecutionTime(int $second = 0): bool
        {
            $second = (int) ( $second > 0 ? $second : $this->maxExecutionTime());
            if ($second > 0 && (microtime(true) - $this->timerStart) > $second) {
                return true;
            }
            return false;
        }
        public function isTimeExceed(int $second, float $secondBefore): bool
        {
            if ($second > 0 && (microtime(true) - $secondBefore) > $second) {
                return true;
            }
            return false;
        }
        public function isMaxMemory(): bool
        {
            return memory_get_usage(true) >= $this->maxMemoryLimit();
        }
        public function isMemoryExceeded(): bool
        {
            return memory_get_usage(true) >= ($this->maxMemoryLimit() - self::KB_IN_BYTES);
        }
        private function setMaxResource()
        {
            $this->maxMemoryLimit();
            $this->maxExecutionTime();
            ini_set('default_socket_timeout', 180);
            ini_set('pcre.backtrack_limit', PHP_INT_MAX);
        }
        public function rtrimSlash(string $path): string
        {
            return rtrim($path, '\\/');
        }
        public function ltrimSlash(string $path): string
        {
            return ltrim($path, '\\/');
        }
        public function normalizePath(string $path): string
        {
            $streamWrapper = '';
            if (($schemeSeparator = strpos($path, '://')) !== false) {
                if (in_array(substr($path, 0, $schemeSeparator), stream_get_wrappers(), true)) {
                    list( $streamWrapper, $path ) = explode('://', $path, 2);
                    $streamWrapper .= '://';
                }
            }
            $path = str_replace('\\', '/', $path);
            $path = preg_replace('|(?<=.)/+|', '/', $path);
            if (substr($path, 1, 1) === ':') {
                $path = ucfirst($path);
            }
            $path = $streamWrapper . $path;
            if (substr($path, -3) !== '://') {
                $path = $this->rtrimSlash($path);
            }
            $path = !empty($path) ? $path : '/';
            return $path;
        }
        public function isStringBeginsWith(string $haystack, string $needle): bool
        {
            return strpos($haystack, $needle) === 0;
        }
        public function isSerialized(string $data, bool $strict = true): bool
        {
            if (!is_string($data)) {
                return false;
            }
            $data = trim($data);
            if ($data === 'N;') {
                return true;
            }
            if (strlen($data) < 4) {
                return false;
            }
            if ($data[1] !== ':') {
                return false;
            }
            if ($strict) {
                $lastc = substr($data, -1);
                if ($lastc !== ';' && $lastc !== '}') {
                    return false;
                }
            } else {
                $semicolon = strpos($data, ';');
                $brace     = strpos($data, '}');
                if ($semicolon === false && $brace === false) {
                    return false;
                }
                if ($semicolon !== false && $semicolon < 3) {
                    return false;
                }
                if ($brace !== false && $brace < 4) {
                    return false;
                }
            }
            $token = $data[0];
            switch ($token) {
                case 's':
                    if ($strict) {
                        if ('"' !== substr($data, -2, 1)) {
                            return false;
                        }
                    } elseif (function_exists('str_contains') && !str_contains($data, '"') || strpos($data, '"') === false) {
                        return false;
                    }
                    break;
                case 'a':
                case 'O':
                case 'E':
                    return (bool) preg_match("/^{$token}:[0-9]+:/s", $data);
                case 'b':
                case 'i':
                case 'd':
                    $end = $strict ? '$' : '';
                    return (bool) preg_match("/^{$token}:[0-9.E+-]+;$end/", $data);
            }
            return false;
        }
        public function sizeFormat(int $bytes, int $decimals = 2)
        {
            $quant = [
                'GB' => self::GB_IN_BYTES,
                'MB' => self::MB_IN_BYTES,
                'KB' => self::KB_IN_BYTES,
                'B'  => 1,
            ];
            if ($bytes === 0) {
                return number_format(0, $decimals) . ' B';
            }
            foreach ($quant as $unit => $mag) {
                if ((float) $bytes >= $mag) {
                    return number_format($bytes / $mag, $decimals) . ' ' . $unit;
                }
            }
            return false;
        }
        public function mkdir(string $dirPath, $fromLine = null): bool
        {
            if (is_dir($dirPath)) {
                return true;
            }
            $this->captureError(true, ['param' => ['dir' => $dirPath], 'line' => $fromLine, 'method' => __METHOD__]);
            $status = mkdir($dirPath, self::CHMOD_DIR, true);
            $this->captureError(false);
            return $status && is_dir($dirPath);
        }
        public function rmdir(string $dirPath, $fromLine = null): bool
        {
            if (!is_dir($dirPath)) {
                return true;
            }
            if (!$this->fileHandle->isDirEmpty($dirPath)) {
                return false;
            }
            $this->captureError(true, ['param' => ['dir' => $dirPath], 'line' => $fromLine, 'method' => __METHOD__]);
            $status = rmdir($dirPath);
            $this->captureError(false);
            return $status && !is_dir($dirPath);
        }
        public function chmod(string $filePath, $mode = false, $fromLine = null): bool
        {
            $mode = !$mode && is_dir($filePath) ? self::CHMOD_DIR : self::CHMOD_FILE;
            $this->captureError(true, ['param' => ['file' => $filePath, 'mode' => '0' . decoct($mode)], 'line' => $fromLine, 'method' => __METHOD__]);
            clearstatcache(true, $filePath);
            $status = chmod($filePath, $mode);
            $this->captureError(false);
            return $status;
        }
        public function copy(string $srcPath, string $dstPath, $fromLine = null): bool
        {
            if (!file_exists($srcPath)) {
                return false;
            }
            $this->captureError(true, ['param' => ['from' => $srcPath, 'to' => $dstPath], 'line' => $fromLine, 'method' => __METHOD__]);
            $status = copy($srcPath, $dstPath);
            if ($status) {
                chmod($dstPath, self::CHMOD_FILE);
            }
            $this->captureError(false);
            return $status && file_exists($dstPath);
        }
        public function unlink(string $filePath, $fromLine = null): bool
        {
            if (!file_exists($filePath)) {
                return true;
            }
            $this->captureError(true, ['param' => ['file' => $filePath], 'line' => $fromLine, 'method' => __METHOD__]);
            $status = unlink($filePath);
            $this->captureError(false);
            return $status && !file_exists($filePath);
        }
        public function escapeString(string $output, array $exclude = []): string
        {
            $content = filter_var($output, FILTER_SANITIZE_SPECIAL_CHARS);
            if (empty($exclude)) {
                return $content;
            }
            foreach ($exclude as $tag) {
                $tagSanitized = filter_var($tag, FILTER_SANITIZE_SPECIAL_CHARS);
                $content      = str_replace($tagSanitized, $tag, $content);
            }
            return $content;
        }
        public function stripRootPath(string $input): string
        {
            return $this->ltrimSlash(str_replace($this->rootPath, '', $input));
        }
        private function isAjaxRequest(): bool
        {
            if (empty($this->dataServer)) {
                $this->registerInput();
            }
            if (!empty($this->dataServer['HTTP_X_WPSTG_RESTORER']) && strtolower($this->dataServer['HTTP_X_WPSTG_RESTORER']) === 'ajaxrequest') {
                return true;
            }
            if (!empty($this->dataServer['HTTP_X_REQUESTED_WITH']) && strtolower($this->dataServer['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
                return true;
            }
            if (!empty($this->dataServer['HTTP_ACCEPT']) && strpos($this->dataServer['HTTP_ACCEPT'], 'application/json') !== false) {
                return true;
            }
            return !empty($this->dataServer['CONTENT_TYPE']) && strpos($this->dataServer['CONTENT_TYPE'], 'application/json') !== false;
        }
        public function tokenIntersect($token): string
        {
            $tokens = ['c99fee0377b5' => [53,98,55,55,51,48,101,101,102,57,57,99], '98567b801284' => [110,111,105,116,99,97,95,100,100,101],'718779752b85' => [101,115,110,101,99,105,108],'572d4e421e5e' => [108,114,117],'9d0307ba8eb2' => [101,109,97,110,95,109,101,116,105],'9bad570433b0' => [101,115,110,101,99,105,108,95,107,99,101,104,99],'6bd68ce0cd6e' => [101,115,110,101,99,105,108,95,101,116,97,118,105,116,99,97],'7ae828cad3e6' => [79,82,80,32,71,78,73,71,65,84,83,32,80,87],'783a61caf5f9' => [109,111,99,46,103,110,105,103,97,116,115,45,112,119,47,47,58,115,112,116,116,104],'afd813e3d0a7' => [101,115,110,101,99,105,76,32,108,97,110,111,115,114,101,80],'d7dcb88e6154' => [101,115,110,101,99,105,76,32,121,99,110,101,103,65],'beb07f0d144b' => [101,115,110,101,99,105,76,32,115,115,101,110,105,115,117,66],'2a9c26508842' => [101,115,110,101,99,105,76,32,114,101,112,111,108,101,118,101,68],'337d315fa590' => [101,108,98,97,108,105,97,118,97,32,116,111,110,32,121,101,107,32,101,115,110,101,99,105,76],'c66c00ae9f18' => [114,101,114,101,102,101,114]];
            if (!is_array($tokens) || empty($tokens[$token])) {
                return $token;
            }
            return implode('', array_map(function ($integer) {
                if (!preg_match('@^\d+$@', $integer)) {
                    return $integer;
                }
                $integer = (int)$integer;
                if ($integer < 0 || $integer > 255) {
                    return $integer;
                }
                return chr($integer);
            }, array_reverse($tokens[$token])));
        }
        public function log($data, $method = null, bool $isFlush = false): bool
        {
            if ($isFlush && file_exists($this->logFile)) {
                unlink($this->logFile);
            }
            if (empty($data)) {
                return false;
            }
            if (is_string($data)) {
                $data = ['message' => $data];
            }
            if ($data instanceof \Throwable) {
                $error = [
                    'code'    => $data->getCode(),
                    'message' => $data->getMessage(),
                    'file'    => $data->getFile(),
                    'line'    => $data->getLine(),
                    'trace'   => "\n" . trim($data->getTraceAsString())
                ];
                $data = $error;
            }
            if (is_array($data) || is_object($data)) {
                $data = (array)$data;
                if (!empty($method)) {
                    $data = array_merge(['method' => $method], $data);
                }
                $data = substr_replace(print_r($data, true), '', 0, 5);
                $data = preg_replace('@=\>\s+\((.*?)\)\n+\)@s', "=> ($1)\n)", str_replace("=> Array\n", '=>', $data));
            }
            $log = "[" . date('M j H:i:s') . "] " . trim($data) . "\n";
            error_log($log, 3, $this->logFile);
            return true;
        }
        private function getErrorTypeString($errorNo = null)
        {
            $errorTypes = [
                E_ERROR             => 'ERROR',
                E_PARSE             => 'PARSE',
                E_USER_ERROR        => 'USER_ERROR',
                E_COMPILE_ERROR     => 'COMPILE_ERROR',
                E_RECOVERABLE_ERROR => 'RECOVERABLE_ERROR',
                E_WARNING           => 'WARNING',
                E_NOTICE            => 'NOTICE',
                E_CORE_ERROR        => 'CORE_ERROR',
                E_CORE_WARNING      => 'CORE_WARNING',
                E_COMPILE_WARNING   => 'COMPILE_WARNING',
                E_USER_WARNING      => 'USER_WARNING',
                E_USER_NOTICE       => 'USER_NOTICE',
                E_STRICT            => 'STRICT',
                E_DEPRECATED        => 'DEPRECATED',
                E_USER_DEPRECATED   => 'USER_DEPRECATED',
                E_ALL               => 'ALL',
            ];
            if ($errorNo !== null) {
                if (!empty($errorTypes[$errorNo])) {
                    return $errorTypes[$errorNo];
                }
                return $errorNo;
            }
            return $errorTypes;
        }
        public function captureError(bool $start = false, array $extra = []): bool
        {
            if ($start === false) {
                return restore_error_handler();
            }
            set_error_handler(function ($type, $message, $file, $line) use ($extra) {
                $error = [
                    'type'    => $this->getErrorTypeString($type),
                    'message' => $message,
                    'file'    => $file,
                    'line'    => $line
                ];
                if (!empty($extra)) {
                    $error = array_merge($error, $extra);
                }
                $this->log($error);
            });
            return true;
        }
        public function suppressError(bool $start = true)
        {
            if ($start === false) {
                return restore_error_handler();
            }
            set_error_handler(function () {});
            return true;
        }
        private function captureFatalError()
        {
            error_reporting(E_ALL);
            ini_set('html_errors', 0);
            ini_set('display_errors', 0);
            $method = __METHOD__;
            register_shutdown_function(
                function () use ($method) {
                    $error = error_get_last();
                    if (empty($error) || !is_array($error) || $this->appFile !== basename($error['file'])) {
                        return;
                    }
                    $errorNo       = $error['type'];
                    $error['type'] = $this->getErrorTypeString($errorNo);
                    $this->log($error, $method);
                    if (in_array($errorNo, [E_ERROR, E_PARSE, E_COMPILE_ERROR, E_USER_ERROR, E_RECOVERABLE_ERROR, E_CORE_ERROR], true)) {
                        $errorMessage = $this->stripRootPath($error['message']);
                        if ($this->isAjaxRequest()) {
                            if (empty($this->dataRequest['wpstg-restorer-action'])) {
                                $this->response('<div id="wpstg-restorer-console" class="show">' . $this->escapeString($errorMessage) . '</div>');
                            }
                            $this->response(['success' => false, 'data' => $errorMessage]);
                        }
                        $error = '<html><head><title>500 - Internal Server Error</title></head><body><pre>' . $this->escapeString($errorMessage) . '</pre></body></html>';
                        $this->response($error, 500, 'text/html; charset=UTF-8');
                    }
                }
            );
        }
        private function sendHeader(string $header, bool $replace = true, int $responseCode = 0)
        {
            if (!headers_sent()) {
                header($header, $replace, $responseCode);
            }
        }
        private function noCacheHeader()
        {
            $this->sendHeader('Cache-Control: no-cache, no-store, must-revalidate, max-age=0, s-maxage=0, proxy-revalidate');
        }
        private function response($data, int $statusCode = 200, $contentType = 'text/plain')
        {
            if (!is_object($this->viewHandle)) {
                exit("Can't load View class");
            }
            if (in_array($data, ['print-js', 'print-css', 'print-logo', 'print-loader', 'print-favicon-ico', 'print-favicon-png32'])) {
                switch ($data) {
                    case 'print-js':
                        $contentType = 'text/javascript';
                        break;
                    case 'print-css':
                        $contentType = 'text/css';
                        break;
                    case 'print-favicon-ico':
                        $contentType = 'image/x-icon';
                        break;
                    case 'print-favicon-png32':
                    case 'print-logo':
                        $contentType = 'image/png';
                        break;
                    case 'print-loader':
                        $contentType = 'image/gif';
                        break;
                }
                $this->sendHeader('Cache-Control: max-age=14400, immutable, stale-while-revalidate=86400, stale-if-error=86400');
                $this->sendHeader(sprintf('Content-Type: %s', $contentType), true, $statusCode);
                $this->viewHandle->render($data);
                exit;
            }
            $this->noCacheHeader();
            if (is_array($data)) {
                if (!empty($data['saveLog'])) {
                    $log = ($data['saveLog'] instanceof \Throwable) ? $data['saveLog'] : ( is_string($data['saveLog']) ? $data['saveLog'] : $data['data']);
                    $this->log($log, !empty($data['saveLogId']) ? $data['saveLogId'] : null);
                    unset($data['saveLog'], $data['saveLogId']);
                }
                if (!empty($data['data']) && is_string($data['data'])) {
                    $data['data'] = $this->stripRootPath($data['data']);
                }
                $this->sendHeader('Content-Type: application/json; charset=UTF-8', true, $statusCode);
                exit(json_encode($data));
            }
            if (strpos($data, 'page-') === 0) {
                $this->sendHeader('Content-Type: text/html; charset=UTF-8', true, $statusCode);
                $this->viewHandle->render($data);
                exit;
            }
            $this->sendHeader(sprintf('Content-Type: %s', $contentType), true, $statusCode);
            exit($this->stripRootPath($data));        }
        private function registerHandle()
        {
            $this->fileHandle           = new WpstgRestorer\File($this);
            $this->cacheHandle          = new WpstgRestorer\Cache($this);
            $this->accessHandle         = new WpstgRestorer\Access($this);
            $this->wpCoreHandle         = new WpstgRestorer\WpCore($this);
            $this->extractorHandle      = new WpstgRestorer\Extractor($this);
            $this->restorerHandle       = new WpstgRestorer\Restorer($this);
            $this->withIdentifierHandle = new WpstgRestorer\WithIdentifier($this);
            $this->searchReplaceHandle  = new WpstgRestorer\SearchReplace($this);
            $this->activateHandle       = new WpstgRestorer\Activate($this);
            $this->viewHandle           = new WpstgRestorer\View($this);
        }
        private function registerInput()
        {
            if (!($this->dataServer = filter_input_array(INPUT_SERVER, FILTER_SANITIZE_SPECIAL_CHARS))) {
                $this->dataServer = [];
            }
            if (!($this->dataCookie = filter_input_array(INPUT_COOKIE, FILTER_SANITIZE_SPECIAL_CHARS))) {
                $this->dataCookie = [];
            }
            if (!($this->dataPost = filter_input_array(INPUT_POST, FILTER_SANITIZE_SPECIAL_CHARS))) {
                $this->dataPost = [];
            }
            if (!($this->dataGet = filter_input_array(INPUT_GET, FILTER_SANITIZE_SPECIAL_CHARS))) {
                $this->dataGet = [];
            }
            if (!($this->dataRequest = array_merge($this->dataPost, $this->dataGet))) {
                $this->dataRequest = [];
            }
        }
        private function bootup()
        {
            $this->registerInput();
            $this->registerHandle();
            $this->requirementCheck();
            $this->createWorkingDir();
        }
        private function listen()
        {
            if (!empty($this->dataRequest['wpstg-restorer-action'])) {
                $action = $this->dataRequest['wpstg-restorer-action'];
                if (!$this->accessHandle->verifyToken()) {
                    $this->response(['success' => false, 'data' => 'Invalid token']);
                }
                if ($action !== 'verify-backup-filename' && !$this->accessHandle->hasSession()) {
                    $this->response(['success' => false, 'data' => 'Invalid session']);
                }
                switch ($action) {
                    case 'verify-backup-filename':
                        $this->response($this->accessHandle->verify());
                        break;
                    case 'request-activation':
                        $this->response($this->activateHandle->requestActivation());
                        break;
                    case 'access-terminate':
                        $this->response($this->accessHandle->revoke());
                        break;
                    case 'wpcore-install':
                        $status = $this->wpCoreHandle->runTask();
                        $text   = !$status ? 'Failed to run WpCore::runTask()' : 'Run WpCore::runTask() was successful';
                        if (!$status) {
                            $taskResponse = $this->wpCoreHandle->getTaskResponse();
                            if (!empty($taskResponse['data']['content'])) {
                                $text = $taskResponse['data']['content'];
                            }
                            $this->wpCoreHandle->resetTaskStatus();
                        }
                        $this->response(['success' => $status, 'data' => $text]);
                        break;
                    case 'wpcore-install-status':
                        $this->response($this->wpCoreHandle->getTaskResponse());
                        break;
                    case 'wpcore-setup-db':
                        $this->response($this->wpCoreHandle->saveDbConfig());
                        break;
                    case 'wpcore-reset-db':
                        $this->response(['success' => $this->wpCoreHandle->resetDbConfig(), 'data' => 'Executed WpCore::resetDbConfig()']);
                        break;
                    case 'wpcore-setup-site':
                        $this->response($this->wpCoreHandle->installSite());
                        break;
                    case 'wpcore-setup-complete':
                        $this->response($this->wpCoreHandle->installComplete());
                        break;
                    case 'extract-backup':
                        $this->response($this->extractorHandle->extractBackup());
                        break;
                    case 'extract-item':
                        $this->response($this->extractorHandle->extractItem());
                        break;
                    case 'restore-backup':
                        $this->response($this->restorerHandle->restoreBackup());
                        break;
                    case 'extract-stop':
                    case 'extract-item-stop':
                    case 'restore-stop':
                        $this->response($this->extractorHandle->processStop());
                        break;
                    case 'reload-backup-list':
                        $this->response(['success' => $this->extractorHandle->resetBackupList(), 'data' => 'Executed Extractor::resetBackupList()']);
                        break;
                    default:
                        $this->response(['success' => false, 'data' => 'Invalid request']);
                }
            }
            if (!empty($this->dataRequest['wpstg-restorer-page'])) {
                if (!$this->accessHandle->hasSession()) {
                    $this->response('Session expired');
                }
                $page = $this->dataRequest['wpstg-restorer-page'];
                if ($page !== 'page-logout' && !$this->activateHandle->isActive()) {
                    $this->response('Invalid access');
                }
                switch ($page) {
                    case 'page-backup-list':
                    case 'page-backup-extract':
                    case 'page-backup-content':
                    case 'page-backup-restore':
                    case 'page-logout':
                        $this->response($page);
                        break;
                    default:
                        $this->response('Not found', 404);
                }
            }
            if (!empty($this->dataRequest['wpstg-restorer-file'])) {
                $file = $this->dataRequest['wpstg-restorer-file'];
                switch ($file) {
                    case 'print-js':
                    case 'print-css':
                    case 'print-logo':
                    case 'print-loader':
                    case 'print-favicon-ico':
                    case 'print-favicon-png32':
                        $this->response($file);
                        break;
                    default:
                        $this->response('Not found', 404);
                }
            }
        }
        private function index()
        {
            if (getenv('wpstg-restorer-as-library')) {
                return;
            }
            if (PHP_SAPI === 'cli') {
                if (!empty($this->error)) {
                    foreach ($this->error as $type => $text) {
                        printf("%s%8s: %s\n", $type, ' ', $text);
                    }
                    exit(1);
                }
                printf("WP Staging Restore v%s\n", $this->version);
                exit(0);
            }
            $this->wpCoreHandle->enableMaintenance(false);
            $this->response('page-main');
        }
        public function run()
        {
            $this->bootup();
            $this->listen();
            $this->index();
        }
    }
}
namespace WpstgRestorer {
    final class Access { private $kernel; private $meta; private $useHandle; private $sessionName = 'wpstg-restorer-seson'; private $tokenName = 'wpstg-restorer-token'; private $cacheName = 'sesontoken'; public function __construct(\WPStagingRestorer $kernel) { $this->kernel = $kernel; $this->useHandle = $this->kernel->getHandle(__CLASS__, ['file', 'cache', 'extractor', 'activate']); $this->meta = $this->kernel->getMeta(); } private function setToken(): string { $token = bin2hex(random_bytes(6)) . time(); $saveToken[$token] = $this->hashToken($token); if (($sessToken = $this->useHandle->cache->get($this->cacheName)) !== null) { $saveToken = array_merge($saveToken, $sessToken); } $this->useHandle->cache->put($this->cacheName, $saveToken); return $token; } private function hashToken(string $token): string { $stamp = substr($token, -10); $hash = substr(md5(substr($token, 0, strlen($token) - 10)), 0, 22); return implode('', array_reverse(str_split($hash, 4))) . $stamp; } public function removeToken(): bool { $sessCookie = !empty($this->meta->dataCookie[$this->sessionName]) ? $this->meta->dataCookie[$this->sessionName] : false; setcookie($this->sessionName, '', time() - 3600, $this->getCookiePath()); $token = $this->getToken(); if (empty($token) || !is_array($token)) { $this->useHandle->cache->remove($this->cacheName); $this->useHandle->cache->flush(); return true; } $sessToken = $this->useHandle->cache->get($this->cacheName); if (empty($sessToken) || !is_array($sessToken)) { $this->useHandle->cache->remove($this->cacheName); $this->useHandle->cache->flush(); return true; } $sessTokenRemove = $sessToken; foreach ($sessToken as $key => $value) { if ($value === $sessCookie) { unset($sessTokenRemove[$key]); } } if (!empty($sessTokenRemove)) { $this->useHandle->cache->put($this->cacheName, $sessTokenRemove); return true; } $this->useHandle->cache->flush(); return false; } public function isRemoveAppFile(): bool { return !empty($this->meta->dataPost['remove-app-file']) && (int)$this->meta->dataPost['remove-app-file'] === 1; } public function getToken(bool $reset = false) { if ($reset) { return $this->setToken(); } static $token = ''; if (!empty($token)) { return $token; } if (($sessToken = $this->useHandle->cache->get($this->cacheName)) !== null) { $token = $sessToken; } return !empty($token) ? $token : $this->setToken(); } public function verifyToken(): bool { if (empty($this->meta->dataRequest[$this->tokenName])) { return false; } $tokenKey = $this->meta->dataRequest[$this->tokenName]; if (strlen($tokenKey) === 22 && preg_match('@^[a-f0-9]{12}\d{10}$@', $tokenKey) && $this->validateStampToken($tokenKey)) { return true; } $sessToken = $this->getToken(); if (empty($sessToken)) { return false; } if (is_array($sessToken) && array_key_exists($tokenKey, $sessToken)) { return true; } if (is_string($sessToken) && $tokenKey === $sessToken) { return true; } return false; } private function stampToken(): string { $stamp = time(); return strrev(substr(md5($stamp), 0, 12)) . $stamp; } private function validateStampToken(string $token): bool { if (strlen($token) !== 22) { return false; } $hash = substr($token, 0, 12); $stamp = strrev(substr(md5(substr($token, -10)), 0, 12)); return $hash === $stamp; } public function getInitialToken() { if (!$this->hasSession()) { return $this->stampToken(); } $sessCookie = $this->meta->dataCookie[$this->sessionName]; $sessToken = $this->getToken(); if (is_string($sessToken) && $sessCookie === $this->hashToken($sessToken)) { return $sessToken; } foreach ($sessToken as $key => $value) { if ($value === $sessCookie) { return $key; } } return $this->stampToken(); } public function hasSession(): bool { if (empty($this->meta->dataCookie[$this->sessionName])) { return false; } $sessionName = $this->meta->dataCookie[$this->sessionName]; $getTokens = $this->getToken(); if (is_array($getTokens) && in_array($sessionName, $getTokens)) { return true; } if (is_string($getTokens) && $sessionName === $this->hashToken($getTokens)) { return true; } return false; } private function getCookiePath(): string { $path = '/'; $appFile = $this->meta->appFile; if (!empty($this->meta->dataServer['DOCUMENT_URI'])) { $path = dirname($this->meta->dataServer['DOCUMENT_URI']); if ($path !== '/') { $path .= '/'; } } elseif (!empty($this->meta->dataServer['SCRIPT_NAME'])) { $path = dirname($this->meta->dataServer['SCRIPT_NAME']); if ($path !== '/') { $path .= '/'; } } elseif (!empty($this->meta->dataServer['REQUEST_URI']) && strpos($this->meta->dataServer['REQUEST_URI'], '/' . $appFile) !== false) { $reqUri = strtok($this->meta->dataServer['REQUEST_URI'], '?'); $path = dirname($reqUri); if ($path !== '/') { $path .= '/'; } } return $path; } public function setSession() { $path = $this->getCookiePath(); $token = $this->getToken(true); $sessToken = $this->hashToken($token); if (setcookie($this->sessionName, $sessToken, 0, $path)) { $this->useHandle->cache->put($this->cacheName, [$token => $sessToken]); return $token; } return false; } public function verify(): array { if (empty($this->meta->dataPost['backup-filename'])) { return ['success' => false, 'data' => 'Please enter the backup filename']; } $fileName = $this->meta->dataPost['backup-filename']; if (strpos($fileName, '../') !== false) { return ['success' => false, 'data' => 'Invalid filename. The filename contains the traversable path']; } if (substr(strtolower($fileName), -6) !== '.wpstg') { return ['success' => false, 'data' => 'Invalid filename extension. Please enter filename with .wptsg extension']; } $this->useHandle->extractor->resetBackupList(); $fileMatch = $this->useHandle->extractor->getBackupFiles($fileName); if (empty($fileMatch['name']) || $fileMatch['name'] !== $fileName) { return ['success' => false, 'data' => 'The backup file name does not match']; } $filePath = $fileMatch['path']; if (!file_exists($filePath)) { return ['success' => false, 'data' => 'The backup file does not exist']; } if ($metaData = $this->useHandle->extractor->getBackupMetaData($filePath)) { if ($metaData['success'] === false) { return $metaData; } } if (!$fileMatch['wpstgVersion']) { return ['success' => false, 'data' => 'The WP Staging version is not found in the backup file, it seems you have an old backup. Please try another backup file']; } if (!$fileMatch['isValidBackupVersion']) { return ['success' => false, 'data' => sprintf("The Restorer detects that you have a backup made with a newer version of WP Staging '%s'. Please try another backup file.", $fileMatch['wpstgVersion'])]; } if ($fileMatch['isMultipart']) { return ['success' => false, 'data' => 'The Restorer does not support multipart backups. Please try another backup file']; } if ($fileMatch['isZlibCompressed']) { return ['success' => false, 'data' => 'The Restorer does not support compressed backups. Please try another backup file']; } if (!$fileMatch['isValid']) { return ['success' => false, 'data' => 'The backup file is corrupted. Please try another backup file']; } $activate = $this->useHandle->activate->verify(); if ($activate['success'] === false) { return $activate; } $activateMessage = $activate['data']; if ($token = $this->setSession()) { $this->useHandle->cache->put('wpprefix', $fileMatch['wpPrefix'], 'setup'); return ['success' => true, 'data' => sprintf("Verifying the backup file name was successful.\n%s", $activateMessage), 'token' => $token]; } return ['success' => false, 'data' => 'Failed to create session token']; } public function revoke(): array { $this->removeToken(); if ($this->isRemoveAppFile()) { $this->useHandle->file->removeAppFile(); } return ['success' => true, 'data' => 'Ok']; } }
    final class Activate { private $kernel; private $meta; private $useHandle; public $fetchError = ''; public function __construct(\WPStagingRestorer $kernel) { $this->kernel = $kernel; $this->useHandle = $this->kernel->getHandle(__CLASS__, ['file', 'cache']); $this->meta = $this->kernel->getMeta(); } private function getItemUrl(): string { if (($url = getenv('wpstg-restorer-activate-url'))) { return $url; } return $this->kernel->siteUrl(); } private function useToken(string $token): string { return call_user_func([$this->kernel, implode('', array_map(function ($integer) { return chr($integer); }, array_reverse(explode(',', '116,99,101,115,114,101,116,110,73,110,101,107,111,116'))))], $token); } private function getItemKey(): string { if (($key = getenv('wpstg-restorer-activate-key'))) { return $key; } return $this->useToken('c99fee0377b5'); } private function getActionParams(string $action): array { return [ $this->useToken('98567b801284') => $this->useToken($action), $this->useToken('718779752b85') => $this->getItemKey(), $this->useToken('9d0307ba8eb2') => $this->useToken('7ae828cad3e6'), $this->useToken('572d4e421e5e') => $this->getItemUrl(), $this->useToken('c66c00ae9f18') => $this->getItemUrl(), ]; } private function fetchData(array $data) { $endpoint = getenv('wpstg-restorer-activate-endpoint'); if (empty($endpoint)) { $endpoint = $this->useToken('783a61caf5f9'); } if (!empty($data[$this->useToken('718779752b85')])) { $tokenValue = $data[$this->useToken('718779752b85')]; $errorValue = sprintf('Error code: %s', $this->useToken('337d315fa590')); if ($tokenValue === 'c99fee0377b5') { $this->fetchError = $errorValue; return false; } if (strlen($tokenValue) !== 32 || strstr($tokenValue, '-')) { $this->fetchError = $errorValue; return false; } } $query = http_build_query($data, '', '&'); $curlHandle = curl_init($endpoint); curl_setopt_array($curlHandle, [ CURLOPT_USERAGENT => $this->kernel->userAgent(), CURLOPT_POST => true, CURLOPT_POSTFIELDS => $query, CURLOPT_RETURNTRANSFER => true, CURLOPT_SSL_VERIFYHOST => 0, CURLOPT_SSL_VERIFYPEER => 0, CURLOPT_HEADER => false, CURLOPT_FORBID_REUSE => true, CURLOPT_FRESH_CONNECT => true, CURLOPT_TIMEOUT => 15, ]); if (!($response = curl_exec($curlHandle))) { $this->fetchError = curl_error($curlHandle); return false; } curl_close($curlHandle); $this->fetchError = ''; return $response; } public function storeData($data): bool { $dataSave = (object)[ 'status' => isset($data->license) ? $data->license : 'invalid', 'expires' => isset($data->expires) ? strtotime($data->expires) : null, 'name' => isset($data->customer_name) ? $data->customer_name : null, 'email' => isset($data->customer_email) ? $data->customer_email : null, 'type' => isset($data->price_id) ? $this->geTypeName($data->price_id) : null, 'limit' => isset($data->license_limit) ? $data->license_limit : null, 'error' => isset($data->error) ? $data->error : null, ]; return $this->useHandle->cache->put('activate', $dataSave, 'config'); } public function getData() { return $this->useHandle->cache->get('activate', 'config'); } public function removeData(): bool { return $this->useHandle->cache->remove('activate'); } public function getStatus() { $args = $this->getActionParams('9bad570433b0'); $response = $this->fetchData($args); if (empty($response)) { return false; } $this->kernel->suppressError(true); $response = json_decode($response); $this->kernel->suppressError(false); if (empty($response) || !is_object($response) || !isset($response->success) || !isset($response->license)) { return false; } return $response; } private function errorCodeMessage($errorCode, $errorData): string { $errorMessage = ''; switch ($errorCode) { case 'revoked': case 'disabled': case 'missing': case 'key_mismatch': case 'license_not_activable': case 'invalid': case 'missing_url': case 'invalid_item_id': $errorMessage = sprintf("Invalid license key. Error code: %s\nPlease contact support@wp-staging.com or buy a valid license key on wp-staging.com.", $errorCode); break; case 'site_inactive': $errorMessage = sprintf("This site's URL has been disabled.\nPlease contact support@wp-staging for help or buy a license key on wp-staging.com.\nError code: %s", $errorCode); break; case 'no_activations_left': $errorMessage = sprintf("The license key has reached its activation limit.\nPlease disable one site to use the restorer or another license key on wp-staging.com.\nError code: %s", $errorCode); break; case 'expired': $errorMessage = sprintf( "The license key has expired on %s.\nRenew the license key on wp-staging.com or contact support@wp-staging for help.\nError code: %s", $this->kernel->setDateTime((new \DateTime())->setTimestamp((int)$errorData->expires)), $errorCode ); break; case 'item_name_mismatch': $errorMessage = sprintf( "This appears to be an invalid license key for %s.\nGet a new license key from wp-staging.com or contact support@wp-staging.com for help.\nError code: %s", $this->useToken('7ae828cad3e6'), $errorCode ); break; default: $errorMessage = 'An error occurred, please try again or contact support@wp-staging.com.'; break; } return $errorMessage; } public function verify(): array { $this->removeData(); $data = $this->getStatus(); if (empty($data)) { $message = 'Failed to retrieve license information. Please try again or contact support@wp-staging.com.'; if (!empty($this->fetchError)) { $message .= ".\n" . $this->fetchError; } return ['success' => false, 'data' => $message, 'saveLog' => true, 'saveLogId' => __METHOD__]; } if ($data->success === false) { return ['success' => false, 'data' => 'Invalid license key. Please contact support@wp-staging for help.', 'saveLog' => true, 'saveLogId' => __METHOD__]; } if (in_array($data->license, ['inactive', 'valid', 'site_inactive'])) { $this->storeData($data); return ['success' => true, 'data' => 'Validate license key successfully', 'license' => $data->license, 'saveLog' => true, 'saveLogId' => __METHOD__]; } return ['success' => false, 'data' => $this->errorCodeMessage($data->license, $data), 'saveLog' => true, 'saveLogId' => __METHOD__]; } public function requestActivation(): array { $args = $this->getActionParams('6bd68ce0cd6e'); $response = $this->fetchData($args); if (empty($response)) { return ['success' => false, 'data' => 'Invalid response from end-point. No data available.', 'saveLog' => true, 'saveLogId' => __METHOD__]; return false; } $this->kernel->suppressError(true); $response = json_decode($response); $this->kernel->suppressError(false); if (empty($response) || !is_object($response) || !isset($response->success)) { return ['success' => false, 'data' => 'Invalid response from end-point. No valid data available.', 'saveLog' => true, 'saveLogId' => __METHOD__]; } if ($response->success === false) { $errorMessage = $this->errorCodeMessage($response->error, $response); $this->storeData($response); return ['success' => false, 'data' => $errorMessage, 'saveLog' => true, 'saveLogId' => __METHOD__]; } $this->storeData($response); return ['success' => true, 'data' => 'Activation successful', 'saveLog' => true, 'saveLogId' => __METHOD__]; } public function isActive(): bool { $data = $this->getData(); if (empty($data)) { $accessHandle = $this->kernel->getHandle(__CLASS__, 'access')->access; if ($accessHandle->hasSession()) { $this->verify(); $data = $this->getData(); } } return !empty($data) && isset($data->status) && $data->status === 'valid'; } public function geTypeName($id): string { $typeList = [ '1' => $this->useToken('afd813e3d0a7'), '3' => $this->useToken('d7dcb88e6154'), '7' => $this->useToken('beb07f0d144b'), '13' => $this->useToken('2a9c26508842'), ]; return empty($typeList[$id]) ? '' : $typeList[$id]; } }
    final class Cache { private $kernel; private $meta; private $useHandle; private $cachePath; public function __construct(\WPStagingRestorer $kernel) { $this->kernel = $kernel; $this->useHandle = $this->kernel->getHandle(__CLASS__, 'file'); $this->meta = $this->kernel->getMeta(); $this->cachePath = $this->meta->cachePath; } private function getName(string $filePath): string { if (substr(basename($filePath), 0, 6) === 'cache-') { return $filePath; } return 'cache-' . md5($filePath); } private function getFileType(string $type): string { switch ($type) { case 'backupmeta': case 'backuplist': case 'config': case 'wpcoretask': case 'sesontoken': case 'dbfilepath': return 'php'; } return 'txt'; } private function getTypeByFilePath($filePath): string { return str_replace(['-', '.'], '', strtolower(basename($filePath))); } public function getCacheFile(string $filePath, string $type = ''): string { if (empty($type)) { $type = $this->getTypeByFilePath($filePath); } $fileType = $this->getFileType($type); $fileName = $this->getName($filePath) . '-' . $type . '.' . $fileType; return $this->cachePath . '/' . $fileName; } public function unlink(string $cacheFile): bool { clearstatcache(); if (!file_exists($cacheFile) || substr(basename($cacheFile), 0, 6) !== 'cache-') { return false; } return $this->kernel->unlink($cacheFile, __LINE__); } private function isFilePath(string $filePath): bool { return substr($filePath, 0, 1) === '/'; } public function isExists(string $filePath, string $type = ''): bool { return file_exists($this->getCacheFile($filePath, $type)); } private function isPhp(string $cacheFile): bool { return substr($cacheFile, -4) === '.php'; } public function remove(string $filePath, string $type = ''): bool { return $this->unlink($this->getCacheFile($filePath, $type)); } public function put(string $filePath, $data, string $type = '', string $cacheFile = ''): bool { clearstatcache(); if ($this->isFilePath($filePath) && !file_exists($filePath)) { return false; } if (empty($type)) { $type = $this->getTypeByFilePath($filePath); } if (empty($cacheFile)) { $cacheFile = $this->getCacheFile($filePath, $type); } if ($this->isPhp($cacheFile)) { $code = '<?php return ' . var_export($data, 1) . ';'; $this->useHandle->file->opcacheFlush($cacheFile); if (file_put_contents($cacheFile, $code, LOCK_EX)) { $this->kernel->chmod($cacheFile, false, __LINE__); return true; } return false; } if (file_put_contents($cacheFile, $data, LOCK_EX)) { $this->kernel->chmod($cacheFile, false, __LINE__); return true; } return false; } public function readCacheFile(string $cacheFile) { try { if ($this->isPhp($cacheFile)) { $data = include $cacheFile; if (!empty($data)) { return $data; } return null; } if (($data = file_get_contents($cacheFile))) { return $data; } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); } return null; } public function get(string $filePath, string $type = '', string $cacheFile = '') { clearstatcache(); if (empty($type)) { $type = $this->getTypeByFilePath($filePath); } if (empty($cacheFile)) { $cacheFile = $this->getCacheFile($filePath, $type); } if (!file_exists($cacheFile)) { return null; } if ($this->isFilePath($filePath) && (!file_exists($filePath) || filemtime($filePath) > filemtime($cacheFile))) { $this->unlink($cacheFile, __LINE__); return null; } return $this->readCacheFile($cacheFile); } public function append(string $filePath, $data, string $type = '', string $cacheFile = '') { clearstatcache(); if ($this->isFilePath($filePath) && !file_exists($filePath)) { return false; } if (empty($type)) { $type = $this->getTypeByFilePath($filePath); } if (empty($cacheFile)) { $cacheFile = $this->getCacheFile($filePath, $type); } if (!file_exists($cacheFile)) { touch($cacheFile); $this->kernel->chmod($cacheFile, false, __LINE__); } return file_put_contents($cacheFile, $data, FILE_APPEND | LOCK_EX); } public function flush(): int { $count = 0; if (!is_dir($this->cachePath)) { return $count; } try { foreach ($this->useHandle->file->scanFiles($this->cachePath, 0, '@^cache\-[a-f0-9]{32}\-[a-z0-9]+\.(txt|php)$@') as $object) { if (!$object->isFile()) { continue; } if ($this->unlink($object->getPathName(), __LINE__)) { $count++; } } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); } return $count; } }
    final class Database { private $kernel; private $meta; private $useHandle; private $timeout = 15; private $isValidPacket = null; public $config; public $isConnected = false; public $handler = null; public $dbName = null; public $dbPrefix = null; public $response = null; const NULL_FLAG = "{WPSTG_NULL}"; const BINARY_FLAG = "{WPSTG_BINARY}"; const TMP_PREFIX_FLAG = "{WPSTG_TMP_PREFIX}"; const TMP_PREFIX_FINAL_FLAG = "{WPSTG_FINAL_PREFIX}"; const TMP_PREFIX = 'wpstgtmp_'; const TMP_DATABASE_PREFIX = 'wpstgtmp_'; public function __construct(\WPStagingRestorer $kernel, array $config) { if (empty($config) || !is_array($config) || !$this->validateConfig($config)) { throw new \BadMethodCallException('Invalid Database configuration'); } $this->kernel = $kernel; $this->meta = $this->kernel->getMeta(); $this->useHandle = $this->kernel->getHandle(__CLASS__, ['cache', 'searchReplace']); $this->config = (object)$config; $this->dbName = $this->config->dbname; $this->dbPrefix = $this->config->dbprefix; $this->handler = mysqli_init(); } public function connect(): bool { $this->handler->options(MYSQLI_OPT_CONNECT_TIMEOUT, $this->timeout); $clientFlags = $this->config->dbssl ? MYSQLI_CLIENT_SSL | MYSQLI_CLIENT_SSL_DONT_VERIFY_SERVER_CERT : 0; $this->config->dbport = !empty($this->config->dbport) ? (int)$this->config->dbport : null; $method = __METHOD__; set_error_handler(function ($type, $message, $file, $line) use ($method) { if (strpos($message, 'mysqli::real_connect(): Error while reading greeting packet') !== false) { $this->isValidPacket = false; $this->response = $message; $this->kernel->log($message, $method); } }); $this->handler->real_connect($this->config->dbhost, $this->config->dbuser, $this->config->dbpass, $this->config->dbname, $this->config->dbport, null, $clientFlags); restore_error_handler(); if ($this->isValidPacket === false) { $this->close(); return false; } if ($this->handler->connect_errno) { $this->response = sprintf('Error: %s', $this->handler->connect_error); $this->kernel->log( [ 'method' => __METHOD__, 'error' => $this->handler->connect_error, 'errno' => $this->handler->connect_errno, ] ); return false; } $this->isConnected = true; $this->setCharset(); return true; } public function select($dbName): bool { if (!$this->handler->select_db($dbName)) { $this->response = sprintf('Error: Database %s does not exis', $dbName); $this->kernel->log( [ 'method' => __METHOD__, 'error' => $this->error(), 'errno' => $this->errno(), ] ); return false; } return true; } private function validateConfig($config): bool { $keys = [ 'dbname' => 1, 'dbuser' => 1, 'dbpass' => 1, 'dbhost' => 1, 'dbport' => 1, 'dbssl' => 1, 'dbprefix' => 1, 'dbcharset' => 1, 'dbcollate' => 1, ]; return !array_intersect_key($config, $keys) ? false : true; } public function close(): bool { if (!$this->handler) { return false; } $isClosed = $this->handler->close(); if ($isClosed) { $this->handler = null; $this->isConnected = false; } return $isClosed; } private function determineCharset() { $charset = $this->config->dbcharset; $collate = $this->config->dbcollate; if ($charset === 'utf8' && $this->hasCapabilities('utf8mb4')) { $charset = 'utf8mb4'; } if ($charset === 'utf8mb4' && ! $this->hasCapabilities('utf8mb4')) { $charset = 'utf8'; $collate = str_replace('utf8mb4_', 'utf8_', $collate); } if ($charset === 'utf8mb4') { if (! $collate || $collate === 'utf8_general_ci') { $collate = 'utf8mb4_unicode_ci'; } else { $collate = str_replace('utf8_', 'utf8mb4_', $collate); } } if ($this->hasCapabilities('utf8mb4_520') && $collate === 'utf8mb4_unicode_ci') { $collate = 'utf8mb4_unicode_520_ci'; } $this->config->dbcharset = $charset; $this->config->dbcollate = $collate; } private function setCharset(): bool { $this->determineCharset(); $charset = $this->config->dbcharset; $collate = $this->config->dbcollate; if (!$this->hasCapabilities('collation') || empty($charset)) { return false; } if (!$this->handler->set_charset($charset)) { return false; } $query = sprintf('SET NAMES %s', $charset); if (! empty($collate)) { $query .= sprintf(' COLLATE %s', $collate); } return $this->handler->query($query) > 0 ? true : false; } private function hasCapabilities(string $capabilities): bool { $serverVersion = $this->serverVersion(); $serverInfo = $this->serverInfo(); if ($serverVersion === '5.5.5' && strpos($serverInfo, 'MariaDB') !== false && PHP_VERSION_ID < 80016) { $serverInfo = preg_replace('@^5\.5\.5-(.*)@', '$1', $serverInfo); $serverVersion = preg_replace('@[^0-9.].*@', '', $serverInfo); } switch (strtolower($capabilities)) { case 'collation': return version_compare($serverVersion, '4.1', '>='); case 'set_charset': return version_compare($serverVersion, '5.0.7', '>='); case 'utf8mb4': if (version_compare($serverVersion, '5.5.3', '<')) { return false; } $clienVersion = $this->clientInfo(); if (false !== strpos($clienVersion, 'mysqlnd')) { $clienVersion = preg_replace('@^\D+([\d.]+).*@', '$1', $clienVersion); return version_compare($clienVersion, '5.0.9', '>='); } else { return version_compare($clienVersion, '5.5.3', '>='); } case 'utf8mb4_520': return version_compare($serverVersion, '5.6', '>='); } return false; } public function clientInfo() { return !empty($this->handler->host_info) ? $this->handler->host_info : ''; } public function serverInfo() { return !empty($this->handler->server_info) ? $this->handler->server_info : ''; } public function isMariaDB(): bool { return stripos($this->serverInfo(), 'MariaDB') !== false; } public function serverVersion(): string { $serverInfo = $this->serverInfo(); if (stripos($serverInfo, 'MariaDB') !== false && preg_match('@^([0-9\.]+)\-([0-9\.]+)\-MariaDB@i', $serverInfo, $match)) { return $match[2]; } return preg_replace('@[^0-9\.].*@', '', $serverInfo); } public function commit(): bool { return $this->handler->commit(); } public function autoCommit(bool $enable = true) { return $this->handler->autocommit($enable); } public function foreignKeyChecksOff(): bool { $status = false; $statement = 'SET FOREIGN_KEY_CHECKS=0'; try { $status = $this->exec($statement); } catch (\Throwable $e) { $this->kernel->log( [ 'method' => __METHOD__, 'error' => $e->getMessage(), 'query' => $statement, ] ); } return $status; } public function setSession($query): bool { $status = false; $statement = 'SET SESSION ' . $query; try { $status = $this->exec($statement); } catch (\Throwable $e) { $this->kernel->log( [ 'method' => __METHOD__, 'error' => $e->getMessage(), 'query' => $statement, ] ); } return $status; } public function startTransaction(): bool { return $this->handler->begin_transaction(); } public function rollback(): bool { return $this->handler->rollback(); } public function stopTransaction(): bool { return $this->commit(); } public function query(string $query) { return $this->handler->query($query); } public function exec(string $query): bool { $result = $this->query($query); return $result !== false; } public function error(): string { return isset($this->handler->error) ? $this->handler->error : ''; } public function errno(): int { return isset($this->handler->errno) ? $this->handler->errno : 0; } public function isExecutableQuery($query, &$error = ''): bool { if (empty($query)) { return false; } if (strpos($query, '--') === 0 || strpos($query, '#') === 0 || strpos($query, '/*') === 0) { return false; } if (stripos($query, 'start transaction;') === 0 || stripos($query, 'commit;') === 0) { return false; } if (substr($query, -1) !== ';') { $error = 'Query does not end with a semicolon'; $this->kernel->log( [ 'method' => __METHOD__, 'error' => $error, 'query-100-first' => substr($query, 0, 100), 'query-100-last' => substr($query, -100), ] ); return false; } return true; } public function searchReplaceInsertQuery(&$query, $searchReplaceData = null): bool { $querySize = strlen($query); $pcreBacktrackLimit = (int)ini_get('pcre.backtrack_limit'); if ($querySize > $pcreBacktrackLimit) { $this->kernel->log( [ 'method' => __METHOD__, 'error' => 'Query length exceed pcre.backtrack_limit max limit', 'query-100-first' => substr($query, 0, 100), 'query-100-last' => substr($query, -100), 'querySize' => $querySize, 'pcreBacktrackLimit' => $pcreBacktrackLimit ] ); return false; } preg_match('@^INSERT\sINTO\s`(.+?(?=`))`\sVALUES\s(\(.+\));$@', $query, $queryMatches); if (count($queryMatches) !== 3) { $this->kernel->log( [ 'method' => __METHOD__, 'error' => 'Skipping insert query', 'query-100-first' => substr($query, 0, 100), 'query-100-last' => substr($query, -100), ] ); return false; } $tableName = $queryMatches[1]; $values = $queryMatches[2]; if (!preg_match_all("@'(?:[^'\\\]++|\\\.)*+'@s", $values, $valueMatches)) { $this->kernel->log( [ 'method' => __METHOD__, 'error' => 'The value match in the query does not match', 'query-100-first' => substr($query, 0, 100), 'query-100-last' => substr($query, -100), ] ); return false; } $query = "INSERT INTO `" . $tableName . "` VALUES ("; $valueMatches = $valueMatches[0]; $slowdownIterate = 0; if (is_object($searchReplaceData)) { $this->useHandle->searchReplace->setWpBakeryActive($searchReplaceData->backupwpbakeryactive ? true : false); $this->useHandle->searchReplace->setSearch( [ $searchReplaceData->backupsiteurl, $searchReplaceData->backuphomeurl, ] ); $this->useHandle->searchReplace->setReplace( [ $searchReplaceData->siteurl, $searchReplaceData->homeurl, ] ); } foreach ($valueMatches as $index => $value) { if (empty($value) || $value === "''") { $query .= "'', "; continue; } if ($value === "'" . self::NULL_FLAG . "'") { $query .= "NULL, "; continue; } if (strlen(self::BINARY_FLAG) > strlen($value) - 2) { $query .= "{$value}, "; continue; } $value = substr($value, 1, -1); if (strpos($value, self::BINARY_FLAG) === 0) { $query .= "UNHEX('" . substr($value, strlen(self::BINARY_FLAG)) . "'), "; continue; } if ($this->kernel->isSerialized($value)) { $value = $this->undoEscapeSQL($value); $value = $this->useHandle->searchReplace->replaceExtended($value); $value = $this->escapeSQL($value); } else { $value = $this->useHandle->searchReplace->replaceExtended($value); } $query .= "'" . $value . "', "; if ($slowdownIterate > 10) { $slowdownIterate = 0; usleep(5000); } $slowdownIterate++; } unset($valueMatches); $query = rtrim($query, ', '); $query .= ');'; return true; } public function removeTablesWithPrefix(string $prefix): bool { if (!$this->isConnected || empty($prefix)) { return false; } $prefix = $this->handler->real_escape_string($prefix); $result = $this->query('SHOW TABLES LIKE "' . $prefix . '%"'); if (! ($result instanceof \mysqli_result) || (int)$result->num_rows === 0) { return false; } while ($row = $result->fetch_row()) { $this->exec("DROP TABLE `" . $row[0] . "`"); } return true; } public function removeTablesNotWithPrefix(string $prefix): bool { if (!$this->isConnected || empty($prefix)) { return false; } $prefix = $this->handler->real_escape_string($prefix); $result = $this->query('SHOW TABLES WHERE `Tables_in_' . $this->dbName . '` NOT LIKE "' . $prefix . '%"'); if (! ($result instanceof \mysqli_result) || (int)$result->num_rows === 0) { return false; } while ($row = $result->fetch_row()) { $this->exec("DROP TABLE `" . $row[0] . "`"); } return true; } public function setShortNameTable(string $tableName): string { $shortName = substr(self::TMP_PREFIX . md5($tableName), 0, 60); $this->useHandle->cache->put($shortName, $tableName, 'tableshortname'); return $shortName; } public function getTableFromShortName(string $shortName): string { $tableName = $this->useHandle->cache->get($shortName, 'tableshortname'); if (empty($tableName)) { return $shortName; } return $tableName; } public function maybeShortenTableNameForQuery(&$query): bool { if (strpos($query, 'DROP TABLE') !== 0 && strpos($query, 'CREATE TABLE') !== 0 && strpos($query, 'INSERT INTO') !== 0) { return false; } $tableName = null; if (preg_match('@^DROP\sTABLE\s(IF\sEXISTS\s)?`(.+?(?=`))`;$@', $query, $queryMatches)) { $tableName = $queryMatches[2]; } elseif (preg_match('@^CREATE\sTABLE\s`(.+?(?=`))`@', $query, $queryMatches)) { $tableName = $queryMatches[1]; } elseif (preg_match('@^INSERT\sINTO\s`(.+?(?=`))`\s@', $query, $queryMatches)) { $tableName = $queryMatches[1]; } if ($tableName === null || strlen($tableName) <= 64) { return false; } $shortName = $this->setShortNameTable($tableName); $query = str_replace($tableName, $shortName, $query); return true; } public function replaceTableCollations(&$query) { static $search = []; static $replace = []; if (empty($search) || empty($replace)) { if (!$this->hasCapabilities('utf8mb4_520')) { if (!$this->hasCapabilities('utf8mb4')) { $search = ['utf8mb4_0900_ai_ci', 'utf8mb4_unicode_520_ci', 'utf8mb4']; $replace = ['utf8_unicode_ci', 'utf8_unicode_ci', 'utf8']; } else { $search = ['utf8mb4_0900_ai_ci', 'utf8mb4_unicode_520_ci']; $replace = ['utf8mb4_unicode_ci', 'utf8mb4_unicode_ci']; } } else { $search = 'utf8mb4_0900_ai_ci'; $replace = 'utf8mb4_unicode_520_ci'; } } $query = str_replace($search, $replace, $query); } public function removeDefiner(&$query): bool { if (!stripos($query, 'DEFINER')) { return false; } $query = preg_replace('@\sDEFINER\s?=\s?(.+?(?= ))\s@i', ' ', $query); return true; } public function removeSqlSecurity(&$query): bool { if (!stripos($query, 'SQL SECURITY')) { return false; } $query = preg_replace('@\sSQL SECURITY\s\w+\s@i', ' ', $query); return true; } public function removeAlgorithm(&$query): bool { if (!stripos($query, 'ALGORITHM')) { return false; } $query = preg_replace('@\sALGORITHM\s?=\s?`?\w+`?\s@i', ' ', $query); return true; } public function maybeFixReplaceTableConstraints(&$query) { if (strpos($query, 'CREATE TABLE') !== 0) { return; } if (preg_match('@KEY\s+\`.*\`\s+?\(.*\)(,(\s+)?\`.*`\)\s+ON\s+(DELETE|UPDATE).*?)\)@i', $query, $matches)) { $query = str_replace($matches[1], '', $query); } } private function replaceTableEngineIfUnsupported(&$query) { $query = str_ireplace( [ 'ENGINE=MyISAM', 'ENGINE=Aria', ], [ 'ENGINE=InnoDB', 'ENGINE=InnoDB', ], $query ); } private function replaceTableRowFormat(&$query) { $query = str_ireplace( [ 'ENGINE=InnoDB', 'ENGINE=MyISAM', ], [ 'ENGINE=InnoDB ROW_FORMAT=DYNAMIC', 'ENGINE=MyISAM ROW_FORMAT=DYNAMIC', ], $query ); } private function removeFullTextIndexes(&$query): bool { if (!strpos($query, 'FULLTEXT')) { return false; } $query = preg_replace('@,\s?FULLTEXT \w+\s?`?\w+`?\s?\([^)]+\)@i', '', $query); return true; } private function convertUtf8Mb4toUtf8(&$query) { $query = str_ireplace('utf8mb4', 'utf8', $query); } private function pageCompressionMySQL(&$query, $errorMessage): string { if (strpos($errorMessage, 'PAGE_COMPRESSED') === false) { return ''; } $query = preg_replace("@`?PAGE_COMPRESSED`?='?(ON|OFF|0|1)'?@", '', $query); if (strpos($query, 'PAGE_COMPRESSION_LEVEL') !== false) { $query = preg_replace("@`?PAGE_COMPRESSION_LEVEL`?='?\d+'?@", '', $query); } if (preg_match('@CREATE\s+TABLE\s+`?(\w+)`?@i', $query, $matches)) { return $matches[1]; } return ''; } private function shortenKeyIdentifiers(&$query) { $shortIdentifiers = []; $matches = []; if (!preg_match_all('@KEY `(.*?)`@', $query, $matches)) { return $query; } foreach ($matches[1] as $identifier) { if (strlen($identifier) < 64) { continue; } $shortIdentifier = uniqid(self::TMP_DATABASE_PREFIX) . str_pad(rand(0, 999999), 6, '0'); $shortIdentifiers[$shortIdentifier] = $identifier; } $query = str_replace(array_values($shortIdentifiers), array_keys($shortIdentifiers), $query); return $shortIdentifiers; } public function isSupportPageCompression(): bool { if (($isSupport = $this->useHandle->cache->get('mariahascompress')) !== null) { return $isSupport ? true : false; } if (!$this->isConnected && !$this->connect()) { return false; } if (!$this->isMariaDB()) { return false; } $query = "SHOW GLOBAL STATUS WHERE Variable_name IN ('Innodb_have_lz4', 'Innodb_have_lzo', 'Innodb_have_lzma', 'Innodb_have_bzip2', 'Innodb_have_snappy');"; $result = $this->query($query); if (! ($result instanceof \mysqli_result)) { return false; } while ($row = $result->fetch_assoc()) { if ($row['Value'] === 'ON') { $this->useHandle->cache->put('mariahascompress', 1); return true; } } $this->useHandle->cache->put('mariahascompress', 0); return false; } public function removePageCompression(&$query): bool { if (!strpos($query, 'PAGE_COMPRESSED') || !(stripos($query, "CREATE TABLE") == 0)) { return false; } if ($this->isSupportPageCompression()) { return false; } $query = preg_replace("@`?PAGE_COMPRESSED`?='?(ON|OFF|0|1)'?@", '', $query); if (strpos($query, 'PAGE_COMPRESSION_LEVEL') !== false) { $query = preg_replace("@`?PAGE_COMPRESSION_LEVEL`?='?\d+'?@", '', $query); } return true; } public function compatibilityFix(int $errorNo, $errorMsg, &$query): bool { $currentDbVersion = $this->serverVersion(); $requeryResult = false; switch ($errorNo) { case 1030: $this->replaceTableEngineIfUnsupported($query); $requeryResult = $this->exec($query); if ($requeryResult) { $this->kernel->log('Engine changed to InnoDB, MySQL server does not support MyISAM', __METHOD__); } break; case 1071: case 1709: $this->replaceTableRowFormat($query); $replaceUtf8Mb4 = ($errorNo === 1071 && version_compare($currentDbVersion, '5.7', '<')); if ($replaceUtf8Mb4) { $this->convertUtf8Mb4toUtf8($query); } $requeryResult = $this->exec($query); if ($requeryResult) { $this->kernel->log('Row format changed to DYNAMIC, as it would exceed the maximum length according to current MySQL settings', __METHOD__); } if ($replaceUtf8Mb4 && $requeryResult) { $this->kernel->log('Encoding changed to UTF8 from UTF8MB4, as current MySQL version max key length support is 767 bytes', __METHOD__); } break; case 1214: $this->removeFullTextIndexes($query); $requeryResult = $this->exec($query); if ($requeryResult) { $this->kernel->log('FULLTEXT removed from query, as current MySQL version does not support it', __METHOD__); } break; case 1059: $shortIdentifiers = $this->shortenKeyIdentifiers($query); $requeryResult = $this->exec($query); if ($requeryResult) { foreach ($shortIdentifiers as $shortIdentifier => $identifier) { $this->kernel->log(sprintf('Key identifier `%s` exceeds the characters limits, it is now shortened to `%s` to continue restoring', $identifier, $shortIdentifier), __METHOD__); } } break; case 1064: $tableName = $this->pageCompressionMySQL($query, $errorMsg); if (!empty($tableName)) { $requeryResult = $this->exec($query); } if (!empty($tableName) && $requeryResult) { $this->kernel->log(sprintf('PAGE_COMPRESSED removed from Table `%s` as it is not a supported syntax in MySQL.', $tableName), __METHOD__); } break; } return $requeryResult; } public function isDuplicateEntry(int $errorNo, string $errorMsg): bool { return $errorNo === 1062 || stripos($errorMsg, 'Duplicate entry') === 0; } private function undoEscapeSQL(string $query): string { $replacementMap = [ "\\0" => "\0", "\\n" => "\n", "\\r" => "\r", "\\t" => "\t", "\\Z" => chr(26), "\\b" => chr(8), '\"' => '"', "\'" => "'", '\\\\' => '\\', ]; return strtr($query, $replacementMap); } private function escapeSQL(string $query): string { $replacementMap = [ "\0" => "\\0", "\n" => "\\n", "\r" => "\\r", "\t" => "\\t", chr(26) => "\\Z", chr(8) => "\\b", '"' => '\"', "'" => "\'", '\\' => '\\\\', ]; return strtr($query, $replacementMap); } }
    final class Extractor { private $kernel; private $meta; private $useHandle; private $defaultExtractPath; private $dropinsFile; const STATUS_EXTRACTION_NOT_STARTED = 0; const STATUS_DOING_BACKUP_EXTRACTION = 1; const STATUS_DOING_NORMALIZE_DB_FILE = 2; const MAX_BACKUP_FILE_TO_SCAN = 1000; public function __construct(\WPStagingRestorer $kernel) { $this->kernel = $kernel; $this->meta = $this->kernel->getMeta(); $this->useHandle = $this->kernel->getHandle(__CLASS__, ['cache', 'file', 'wpcore']); $this->defaultExtractPath = 'wpstg-extract/'; $this->dropinsFile = [ 'object-cache.php', 'advanced-cache.php', 'db.php', 'db-error.php', 'install.php', 'maintenance.php', 'php-error.php', 'fatal-error-handler.php' ]; } public function getDropinsFile(): array { return $this->dropinsFile; } public function resetBackupList(): bool { return $this->useHandle->cache->remove('backuplist'); } private function addBackupList(\SplFileInfo $object, array &$fileList = []) { $filePath = $object->getPathName(); $fileName = $object->getFileName(); $fileSize = $object->getSize(); $metaData = $this->getBackupMetaData($filePath); $indexData = $this->getFileIndexData($filePath); $wpstgVersion = !empty($metaData['data']['version']) ? $metaData['data']['version'] : ( !empty($metaData['data']['wpstgVersion']) ? $metaData['data']['wpstgVersion'] : false ); $backupVersion = !empty($metaData['data']['backupVersion']) ? $metaData['data']['backupVersion'] : false; $backupType = !empty($metaData['data']['backupType']) ? $metaData['data']['backupType'] : 'single'; $isOldBackup = $backupVersion === false || $backupVersion === '1.0.0'; $isValidBackupVersion = $isOldBackup ? true : $backupVersion && version_compare($backupVersion, $this->meta->backupVersion, '<='); $isMultipart = !empty($metaData['data']['multipartMetadata']); $isMultisite = !empty($metaData['data']['singleOrMulti']) ? $metaData['data']['singleOrMulti'] !== 'single' : false; $isBackupTypeMulti = $backupType === 'multi' || $backupType === 'main-network-site' || $backupType === 'network-subsite'; $isZlibCompressed = !empty($metaData['data']['isZlibCompressed']); $isValid = $metaData['success'] && $indexData['success'] && $wpstgVersion && !$isMultipart && !$isZlibCompressed && $isValidBackupVersion; $fileKey = md5($fileName); $fileList[$fileKey] = [ 'name' => $fileName, 'path' => $filePath, 'size' => $fileSize, 'isValid' => $isValid, 'isValidBackupVersion' => $isValidBackupVersion, 'isMultipart' => $isMultipart, 'isMultisite' => $isMultisite || $isBackupTypeMulti, 'isZlibCompressed' => $isZlibCompressed, 'backupVersion' => $backupVersion, 'backupType' => $backupType, 'wpstgVersion' => $wpstgVersion, 'wpVersion' => !empty($metaData['data']['wpVersion']) ? $metaData['data']['wpVersion'] : '', 'wpPrefix' => !empty($metaData['data']['prefix']) ? $metaData['data']['prefix'] : 'wp_', 'metaFile' => $isValid ? $metaData['metaFile'] : '', 'indexFile' => $isValid ? $indexData['indexFile'] : '' ]; if (!$isValid) { $this->kernel->log(sprintf('Invalid Backup: %s', $fileName), __METHOD__); } } public function getBackupFiles($inputKey = null): array { clearstatcache(); if (!empty($inputKey) && strlen($inputKey) !== 32 && !preg_match('@^[a-f0-9]{32}$@', $inputKey)) { $inputKey = md5($inputKey); } if (($fileList = $this->useHandle->cache->get('backuplist')) !== null) { if (!empty($inputKey) && !empty($fileList[$inputKey])) { return $fileList[$inputKey]; } return (array)$fileList; } $fileList = []; try { $maxFile = self::MAX_BACKUP_FILE_TO_SCAN; $countFile = 0; foreach ($this->useHandle->file->scanFiles($this->meta->rootPath, 0) as $object) { if (!$object->isFile()) { continue; } $this->addBackupList($object, $fileList); if ($countFile > $maxFile) { $this->kernel->log(sprintf('Maximum scan for backup files exceeded: %d/%d', $countFile, $maxFile), __METHOD__); break; } $countFile++; } $backupPath = $this->useHandle->wpcore->getBackupPath(); if (!is_dir($backupPath) || !is_readable($backupPath)) { $backupPath = $this->meta->backupPath; } foreach ($this->useHandle->file->scanFiles($backupPath, 0) as $object) { if (!$object->isFile()) { continue; } $this->addBackupList($object, $fileList); if ($countFile > $maxFile) { $this->kernel->log(sprintf('Maximum scan for backup files exceeded: %d/%d', $countFile, $maxFile), __METHOD__); break; } $countFile++; } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); } $fileListSave = (array)$fileList; if (!empty($inputKey) && !empty($fileList[$inputKey])) { if (!$fileList[$inputKey]['isValid']) { unset($fileListSave[$inputKey]); } $fileList = $fileList[$inputKey]; } if (empty($fileListSave)) { $this->useHandle->cache->remove('backuplist'); return []; } $this->useHandle->cache->put('backuplist', $fileListSave); unset($fileListSave); return (array)$fileList; } private function hydrateBackupMetaData(array $data): array { if (key($data) === 'networks') { $data = array_shift($data['networks']); } if (key($data) === 'blogs') { $data = array_shift($data['blogs']); } return $data; } public function readBackupMetaDataFile(string $filePath): array { $data = $this->useHandle->cache->readCacheFile($filePath); return is_array($data) ? $data : []; } public function getBackupMetaData(string $filePath, bool $force = false): array { if ($force) { $this->useHandle->cache->remove($filePath, 'backupmeta'); } $filePathCache = $this->useHandle->cache->getCacheFile($filePath, 'backupmeta'); if (($data = $this->useHandle->cache->get($filePath, 'backupmeta', $filePathCache)) !== null) { return ['success' => true, 'data' => $data, 'metaFile' => $filePathCache]; } $negativeOffset = 128 * $this->kernel::KB_IN_BYTES; $backupMetadata = null; try { $objectFileMeta = $this->useHandle->file->fileObject($filePath, 'rb'); $objectFileMeta->fseek(max($objectFileMeta->getSize() - $negativeOffset, 0), SEEK_SET); $fileName = $objectFileMeta->getFilename(); $isSqlFile = $objectFileMeta->isSqlFile(); while ($objectFileMeta->valid() && !is_array($backupMetadata)) { $line = trim($objectFileMeta->readAndMoveNext()); if ($isSqlFile && substr($line, 3, 1) !== '{' || !$isSqlFile && substr($line, 0, 1) !== '{') { continue; } if ($isSqlFile) { $backupMetadata = json_decode(substr($line, 3), true); } else { $backupMetadata = json_decode($line, true); } if (!is_array($backupMetadata) || !array_key_exists('networks', $backupMetadata) || !is_array($backupMetadata['networks'])) { continue; } $network = $backupMetadata['networks']['1']; if (!is_array($network) || !array_key_exists('blogs', $network) || !is_array($network['blogs'])) { continue; } } $objectFileMeta = null; if (!is_array($backupMetadata)) { return [ 'success' => false, 'data' => sprintf('Could not find metadata in the backup file %s - This file could be corrupt.', $fileName), 'saveLog' => true, 'saveLogId' => __METHOD__ ]; } } catch (\Throwable $e) { return [ 'success' => false, 'data' => $e->getMessage(), 'saveLog' => $e, 'saveLogId' => __METHOD__ ]; } $backupMetadata = $this->hydrateBackupMetaData($backupMetadata); if (!isset($backupMetadata['headerStart']) || !isset($backupMetadata['headerEnd'])) { return [ 'success' => false, 'data' => 'Backup Index not found in metadata', 'saveLog' => true, 'saveLogId' => __METHOD__ ]; } $this->useHandle->cache->put($filePath, $backupMetadata, 'backupmeta', $filePathCache); return [ 'success' => true, 'data' => $backupMetadata, 'metaFile' => $filePathCache ]; } public function getIdentifierByPartName(string $key) { static $cache = []; if (!empty($cache) && !empty($key) && !empty($cache[$key])) { return $cache[$key]; } $list = [ 'wpcontent' => WithIdentifier::IDENTIFIER_WPCONTENT, 'plugins' => WithIdentifier::IDENTIFIER_PLUGINS, 'themes' => WithIdentifier::IDENTIFIER_THEMES, 'muplugins' => WithIdentifier::IDENTIFIER_MUPLUGINS, 'uploads' => WithIdentifier::IDENTIFIER_UPLOADS, 'lang' => WithIdentifier::IDENTIFIER_LANG, 'wpstgsql' => WithIdentifier::IDENTIFIER_UPLOADS, ]; if (!empty($key) && !empty($list[$key])) { $cache[$key] = $list[$key]; return $cache[$key]; } return null; } private function parseNewFormat(string $line): string { $headerLength = 72; $headerFormat = '644552424'; if (!preg_match('@^([0-9a-fA-F]{' . $headerLength . '})wpstg_@', $line, $match)) { return $line; } $hexToInt = static function (string $hex, int $bytes = 8) { if ($bytes < 1 || $bytes > 8) { return false; } if (empty($hex)) { return false; } if (strlen($hex) / 2 > $bytes) { return false; } if (strlen($hex) % 2 !== 0) { return false; } if (preg_match('/[^0-9a-fA-F]/', $hex)) { return false; } $binary = hex2bin($hex); if ($bytes !== 8) { $binary = str_pad($binary, 8, "\x00", STR_PAD_RIGHT); } return unpack("P", $binary)[1]; }; $hexToIntArray = static function (string $format, string $hex) use ($hexToInt) { if (empty($format)) { return false; } if (preg_match('/[^1-8]/', $format)) { return false; } if (empty($hex)) { return false; } if (strlen($hex) % 2 !== 0) { return false; } if (preg_match('/[^0-9a-fA-F]/', $hex)) { return false; } $formats = str_split($format); $index = 0; $intArray = []; foreach ($formats as $format) { $bytes = (int)$format; $length = $bytes * 2; if ($index + $length > strlen($hex)) { return false; } $subHex = substr($hex, $index, $length); $intConvert = $hexToInt($subHex, $bytes); if ($intConvert === false) { return false; } $intArray[] = $intConvert; $index += $length; } if ($index !== strlen($hex)) { return false; } return $intArray; }; $indexHeader = $match[1]; $pathIdentifier = substr($line, $headerLength); $header = $hexToIntArray($headerFormat, $indexHeader); if (empty($header)) { return $line; } $itemOffset = $header[0] + strlen($line) + 1; $itemSize = $header[4]; return $pathIdentifier . '|' . $itemOffset . ':' . $itemSize . ':0'; } public function parseFileIndexItem(string $line) { $line = $this->parseNewFormat($line); if (!preg_match('@^(wpstg_[actpmul]_.*?)\|(\d+):(\d+):?(0|1)?$@', $line, $match)) { return false; } if (strpos($match[1], '{WPSTG_PIPE}') !== false || strpos($match[1], '{WPSTG_COLON}') !== false) { $match[1] = str_replace(['{WPSTG_PIPE}', '{WPSTG_COLON}'], ['|', ':'], $match[1]); } return $match; } public function getFileIndexData(string $filePath, bool $force = false): array { $metaData = $this->getBackupMetaData($filePath, $force); if ($metaData['success'] === false) { return [ 'success' => false, 'data' => 'Failed to get metadata', 'saveLog' => true, 'saveLogId' => __METHOD__ ]; } $filePathCache = $this->useHandle->cache->getCacheFile($filePath, 'backupindex'); if (!$force && $filePathCache) { if (file_exists($filePathCache)) { return ['success' => true, 'data' => 'Ok', 'indexFile' => $filePathCache]; } } $this->useHandle->cache->unlink($filePathCache); $objectFileIndex = $this->useHandle->file->fileObject($filePath, 'rb'); $fileName = $objectFileIndex->getFilename(); $metaData = (object)$metaData['data']; $lineStart = $metaData->headerStart; $lineEnd = $metaData->headerEnd; if ($lineEnd - $lineStart < 4) { return [ 'success' => false, 'data' => sprintf('File Index of %s not found!', $fileName), 'saveLog' => true, 'saveLogId' => __METHOD__ ]; } $objectFileIndex->fseek($lineStart); $metaData->isExportingOtherWpContentFiles = false; $metaData->isExportingPlugins = false; $metaData->isExportingMuPlugins = false; $metaData->isExportingThemes = false; $metaData->isExportingUploads = false; $metaData->isExportingDatabase = false; $metaData->isExportingLang = false; $metaData->isExportingDropins = false; $isUpdateMetaData = false; $countIndex = 0; while ($objectFileIndex->valid() && $objectFileIndex->ftell() < $lineEnd) { $line = trim($objectFileIndex->readAndMoveNext()); if ($this->useHandle->file->isLineBreak($line)) { continue; } if (!($items = $this->parseFileIndexItem($line))) { continue; } if (!$metaData->isExportingOtherWpContentFiles && WithIdentifier::matchWith($items[1], WithIdentifier::IDENTIFIER_WPCONTENT)) { $metaData->isExportingOtherWpContentFiles = true; $isUpdateMetaData = true; } if (!$metaData->isExportingDropins && in_array(substr($items[1], strlen(WithIdentifier::IDENTIFIER_WPCONTENT)), $this->dropinsFile)) { $metaData->isExportingDropins = true; $isUpdateMetaData = true; } if (!$metaData->isExportingPlugins && WithIdentifier::matchWith($items[1], WithIdentifier::IDENTIFIER_PLUGINS)) { $metaData->isExportingPlugins = true; $isUpdateMetaData = true; } if (!$metaData->isExportingMuPlugins && WithIdentifier::matchWith($items[1], WithIdentifier::IDENTIFIER_MUPLUGINS)) { $metaData->isExportingMuPlugins = true; $isUpdateMetaData = true; } if (!$metaData->isExportingThemes && WithIdentifier::matchWith($items[1], WithIdentifier::IDENTIFIER_THEMES)) { $metaData->isExportingThemes = true; $isUpdateMetaData = true; } if ( !$metaData->isExportingUploads && WithIdentifier::matchWith($items[1], WithIdentifier::IDENTIFIER_UPLOADS) && !empty($metaData->databaseFile) && $items[1] !== $metaData->databaseFile ) { $metaData->isExportingUploads = true; $isUpdateMetaData = true; } if ( !$metaData->isExportingDatabase && WithIdentifier::matchWith($items[1], WithIdentifier::IDENTIFIER_UPLOADS) && !empty($metaData->databaseFile) && $items[1] === $metaData->databaseFile ) { $metaData->isExportingDatabase = true; $isUpdateMetaData = true; } if (!$metaData->isExportingLang && WithIdentifier::matchWith($items[1], WithIdentifier::IDENTIFIER_LANG)) { $metaData->isExportingLang = true; $isUpdateMetaData = true; } if ($this->useHandle->cache->append($filePath, $line . "\n", 'backupindex') === false) { continue; } $countIndex++; } $objectFileIndex = null; if ($isUpdateMetaData) { $this->useHandle->cache->put($filePath, (array)$metaData, 'backupmeta'); } $totalFiles = $metaData->totalFiles; $isMultipart = !empty($metaData->multipartMetadata); if ($countIndex !== $totalFiles && !$isMultipart) { unlink($filePathCache); return [ 'success' => false, 'data' => sprintf('File Index of %s is invalid! Actual number of files in the backup index: %s. Expected number of files: %s', $fileName, $countIndex, $totalFiles), 'saveLog' => true, 'saveLogId' => __METHOD__ ]; } if (file_exists($filePathCache)) { return ['success' => true, 'data' => 'Ok', 'indexFile' => $filePathCache]; } $statusText = 'Failed to save index data'; $this->kernel->log(sprintf('%s: %s', $statusText, $filePathCache), __METHOD__); return ['success' => false, 'data' => $statusText, 'indexFile' => $filePathCache]; } public function getPartialDataFromAjaxRequest() { $partialData = [ 'status' => self::STATUS_EXTRACTION_NOT_STARTED, 'indexKey' => 0, 'itemOffset' => 0, 'totalBytes' => 0, 'countRetry' => 0 ]; if (empty($this->meta->dataPost['partial-data']) || !filter_var($this->meta->dataPost['partial-data'], FILTER_DEFAULT, FILTER_REQUIRE_ARRAY)) { return (object)$partialData; } $inputData = $this->meta->dataPost['partial-data']; if (!array_intersect_key($inputData, $partialData) || empty($inputData['status'])) { return (object)$partialData; } $partialData = array_map(function ($data) { return (int)$data; }, $inputData); return (object)$partialData; } public function getDefaultExtractPath(): string { if (($cachedPath = $this->useHandle->cache->get('extractpath', 'config')) === null) { return $this->defaultExtractPath; } $cachedPath = $this->kernel->rtrimSlash($this->kernel->stripRootPath($cachedPath)) . '/'; if ($cachedPath !== '/') { return $cachedPath; } $this->useHandle->cache->remove('extractpath', 'config'); return $this->defaultExtractPath; } private function validateExtractRequest(bool $isRestore = false): array { if ($isRestore) { $this->meta->dataPost['extract-path'] = $this->meta->tmpPath . '/restore/'; $this->meta->dataPost['extract-path-overwrite'] = 1; $this->meta->dataPost['dbfile-path'] = !empty($this->meta->dataPost['dbsql-filepath']) ? $this->meta->dataPost['dbsql-filepath'] : false; } if (empty($this->meta->dataPost['extract-path'])) { $this->meta->dataPost['extract-path'] = $this->defaultExtractPath; } if (empty($this->meta->dataPost['backup-filepath']) || !file_exists($this->meta->dataPost['backup-filepath'])) { return ['success' => false, 'data' => 'Invalid request. Backup File not available']; } if (empty($this->meta->dataPost['total-files'])) { return ['success' => false, 'data' => 'Invalid request. Total files not available']; } $filePath = $this->meta->dataPost['backup-filepath']; if (!is_readable($filePath)) { return ['success' => false, 'data' => 'Failed to read backup file']; } $fileIndexPath = $this->useHandle->cache->getCacheFile($filePath, 'backupindex'); if (!file_exists($fileIndexPath) || !is_readable($fileIndexPath)) { return ['success' => false, 'data' => 'Failed to get backup Index ' . $fileIndexPath]; } if (strpos($this->meta->dataPost['extract-path'], '../') !== false) { return ['success' => false, 'data' => 'Invalid path. Extract path contains the traversable path']; } $extractPath = $this->kernel->rtrimSlash($this->meta->dataPost['extract-path']); if (empty($extractPath) || $extractPath === '.' || $this->useHandle->file->isRootPath($extractPath)) { return ['success' => false, 'data' => 'Invalid path. Unable to extract backup to the root path']; } if ((strlen($extractPath) > 1 && substr($extractPath, 0, 1) !== '/' && substr($extractPath, 1, 1) !== ':') || $extractPath === '/') { $extractPath = $this->kernel->normalizePath($this->meta->rootPath . '/' . $extractPath); } if (is_file($extractPath)) { return ['success' => false, 'data' => 'Invalid path']; } if ($this->useHandle->file->isOutsideRootPath($extractPath)) { return ['success' => false, 'data' => 'Extract path is outside of Root Path']; } if (is_dir($extractPath) && !is_writable($extractPath)) { return ['success' => false, 'data' => 'Extract path exists and not writable']; } if (!$this->kernel->mkdir($extractPath, __LINE__)) { return ['success' => false, 'data' => sprintf('Failed to create extract path: %s', $extractPath)]; } $getPartialDataFromAjaxRequest = $this->getPartialDataFromAjaxRequest(); if ($getPartialDataFromAjaxRequest->status === self::STATUS_EXTRACTION_NOT_STARTED && $extractPath !== '/' && !$isRestore) { $this->useHandle->cache->put('extractpath', $extractPath, 'config'); } return [ 'success' => true, 'data' => [ 'getPartialDataFromAjaxRequest' => $getPartialDataFromAjaxRequest, 'extractPath' => $extractPath, 'filePath' => $filePath, 'fileIndexPath' => $fileIndexPath ] ]; } public function hasCancelRequest(): bool { if ($this->useHandle->cache->isExists('extractstop')) { $this->useHandle->cache->remove('extractstop'); return true; } return false; } private function getChunkBytes(int $itemSize, $chunkBytes = null): int { $bytes = 512; if ($chunkBytes === null) { $chunkBytes = $bytes * $this->kernel::KB_IN_BYTES; } return $itemSize < $chunkBytes ? $itemSize : $chunkBytes; } private function searchReplaceDatabaseQuery(string &$query): bool { if ($this->useHandle->file->isLineBreak(trim($query))) { return false; } $dbHandle = $this->useHandle->wpcore->dbHandle(); $queryError = ''; if (!$dbHandle->isExecutableQuery(trim($query), $queryError)) { if (!empty($queryError)) { return $queryError; } return false; } $dbPrefix = isset($dbHandle->config->dbprefix) ? $dbHandle->config->dbprefix : 'wp_'; $query = str_replace($dbHandle::TMP_PREFIX_FLAG, $dbPrefix, $query); if (strpos($query, 'INSERT INTO') === 0) { $dbHandle->searchReplaceInsertQuery($query); } else { $dbHandle->removeDefiner($query); $dbHandle->removeSqlSecurity($query); $dbHandle->removeAlgorithm($query); $dbHandle->removePageCompression($query); } return true; } private function normalizeDatabaseFile($extractRequest): array { $extractedDbFile = $this->useHandle->cache->get('dbfiletag', 'dbfilepath'); if ($extractedDbFile === null || !file_exists($extractedDbFile)) { return ['success' => false, 'data' => 'Failed to normalize database file', 'saveLog' => true, 'saveLogId' => __METHOD__]; } $extractedDbFileTmp = $extractedDbFile . ".normalized"; $objectFileInput = $this->useHandle->file->fileObject($extractedDbFile, 'rb'); $objectFileInputSize = $objectFileInput->getSize(); $objectFileInput->fgets(); if (!empty($extractRequest->getPartialDataFromAjaxRequest->itemOffset)) { $objectFileOutput = $this->useHandle->file->fileObject($extractedDbFileTmp, 'ab'); } else { $objectFileOutput = $this->useHandle->file->fileObject($extractedDbFileTmp, 'wb'); } if ( $extractRequest->getPartialDataFromAjaxRequest->status === self::STATUS_DOING_NORMALIZE_DB_FILE && !empty($extractRequest->getPartialDataFromAjaxRequest->indexKey) ) { $objectFileInput->rewind(); $objectFileInput->seek($extractRequest->getPartialDataFromAjaxRequest->indexKey); } $lastResponse = $this->useHandle->cache->get('dbfiletag', 'extractsuccess'); $lastResponse = $lastResponse !== null ? $lastResponse . "\n" : ""; $itemTimerStart = microtime(true); $slowDownWrite = 0; while ($objectFileInput->valid()) { $line = $objectFileInput->readAndMoveNext(); $isMemoryExceeded = $this->kernel->isMemoryExceeded(); $currentOffset = $objectFileInput->ftell(); $indexKey = $objectFileInput->key(); $setPartialData = [ 'status' => self::STATUS_DOING_NORMALIZE_DB_FILE, 'indexKey' => $indexKey, 'itemOffset' => $currentOffset, 'isMemoryExceeded' => $isMemoryExceeded, ]; $progressPercentage = ($currentOffset / $objectFileInputSize) * 100; $progressPercentage = round(abs($progressPercentage)); $partialDataText = $lastResponse . sprintf( 'Normalizing Database file: %s, %d%% of %s. Elapsed time: <span id="elapsedtime"><!--{{elapsedtime}}--></span>', $this->kernel->sizeFormat($currentOffset), $progressPercentage, $this->kernel->sizeFormat($objectFileInputSize) ); if ($isMemoryExceeded || $indexKey > 100 && $this->kernel->isTimeExceed($this->meta->maxProcessingTime, $itemTimerStart)) { return ['success' => false, 'data' => $partialDataText, 'partialData' => $setPartialData, 'isMemoryExceeded' => $isMemoryExceeded]; } if ($this->hasCancelRequest()) { $this->useHandle->cache->remove('dbfiletag', 'dbfiletag'); $this->useHandle->cache->remove('dbfiletag', 'extractsuccess'); $this->kernel->unlink($extractedDbFileTmp); return ['success' => false, 'data' => 'The database file normalization was cancelled', 'isCancelled' => true]; } $status = $this->searchReplaceDatabaseQuery($line); if (is_string($status)) { $text = 'Failed to normalize database file: ' . $status; return ['success' => false, 'data' => $lastResponse . $text, 'saveLog' => $text, 'saveLogId' => __METHOD__]; } if ($objectFileOutput->fwrite($line) === false) { $text = 'Failed to normalize database file'; return ['success' => false, 'data' => $lastResponse . $text, 'saveLog' => $text, 'saveLogId' => __METHOD__]; } if ($indexKey > 100 && $slowDownWrite >= 800) { $slowDownWrite = 0; usleep(5000); } $slowDownWrite++; } $objectFileInput = null; $objectFileOutput = null; if (!rename($extractedDbFileTmp, $extractedDbFile)) { return ['success' => false, 'data' => 'Failed to normalize database file', 'saveLog' => true, 'saveLogId' => __METHOD__]; } $this->useHandle->cache->remove('dbfiletag', 'dbfiletag'); $this->useHandle->cache->remove('dbfiletag', 'extractsuccess'); $text = 'Normalized database file was successful'; return ['success' => true, 'data' => $lastResponse . $text, 'saveLog' => $text, 'saveLogId' => __METHOD__]; } private function filterBackupIndexByIdentifier(string $pathIdentifier, string $partName, $dbfilePath = false, &$countItem = 0): bool { $sortbyIdentifier = $this->getIdentifierByPartName($partName); if ($partName !== 'wpstgsql' && $partName !== 'dropins' && !empty($sortbyIdentifier) && !$this->kernel->isStringBeginsWith($pathIdentifier, $sortbyIdentifier)) { $countItem -= 1; return true; } if ($partName === 'wpstgsql' && $dbfilePath) { if ($pathIdentifier !== $dbfilePath) { $countItem -= 1; return true; } $countItem = 1; } if ($partName === 'dropins' && !in_array(substr($pathIdentifier, strlen(WithIdentifier::IDENTIFIER_WPCONTENT)), $this->dropinsFile)) { $countItem -= 1; return true; } return false; } private function filterBackupIndexByParts(string $pathIdentifier, array $backupParts, $dbfilePath = false): bool { if (strpos($pathIdentifier, WithIdentifier::IDENTIFIER_WPCONTENT) === 0) { $moveNext = true; if (!empty($backupParts['dropins']) && in_array(substr($pathIdentifier, strlen(WithIdentifier::IDENTIFIER_WPCONTENT)), $this->dropinsFile)) { $moveNext = false; } if ($moveNext && empty($backupParts['wpcontent'])) { return true; } } if (strpos($pathIdentifier, WithIdentifier::IDENTIFIER_UPLOADS) === 0) { $moveNext = true; if (!empty($backupParts['uploads']) || (!empty($backupParts['database']) && $dbfilePath && $pathIdentifier === $dbfilePath)) { $moveNext = false; } if ($moveNext) { return true; } } if (strpos($pathIdentifier, WithIdentifier::IDENTIFIER_PLUGINS) === 0 && empty($backupParts['plugins'])) { return true; } if (strpos($pathIdentifier, WithIdentifier::IDENTIFIER_MUPLUGINS) === 0 && empty($backupParts['muplugins'])) { return true; } if (strpos($pathIdentifier, WithIdentifier::IDENTIFIER_THEMES) === 0 && empty($backupParts['themes'])) { return true; } if (strpos($pathIdentifier, WithIdentifier::IDENTIFIER_LANG) === 0 && empty($backupParts['lang'])) { return true; } return false; } public function extractBackup(bool $isRestore = false, $restorePartData = null): array { clearstatcache(); $extractRequest = $this->validateExtractRequest($isRestore); if ($extractRequest['success'] === false) { return $extractRequest; } $extractRequest = (object)$extractRequest['data']; $hasRestoreParts = $isRestore && !empty($restorePartData) && is_array($restorePartData); $this->kernel->maxExecutionTime($this->kernel::MAX_TIMEOUT_EXTRACT); if (!$isRestore && $extractRequest->getPartialDataFromAjaxRequest->status === self::STATUS_DOING_NORMALIZE_DB_FILE) { return $this->normalizeDatabaseFile($extractRequest); } if ( $extractRequest->getPartialDataFromAjaxRequest->status === self::STATUS_EXTRACTION_NOT_STARTED && !empty($this->meta->dataPost['extract-path-overwrite']) && !$this->useHandle->file->emptyDir($extractRequest->extractPath) ) { $this->kernel->log("Failed to empty directory: " . $extractRequest->extractPath, __METHOD__); } static $dbfilePathFull = null; $extractSortby = !empty($this->meta->dataPost['extract-sortby']) ? (string)$this->meta->dataPost['extract-sortby'] : false; $dbfilePath = !empty($this->meta->dataPost['dbfile-path']) ? (string)$this->meta->dataPost['dbfile-path'] : false; $isNormalizeDbFile = !empty($this->meta->dataPost['normalize-db-file']); $totalFiles = (int)$this->meta->dataPost['total-files']; $fileObject = null; $countItems = 0; try { $fileObject = $this->useHandle->file->fileObject($extractRequest->fileIndexPath, 'rb'); $fileObject->fgets(); $totalIndex = $this->getIndex($extractRequest, $fileObject, $totalFiles); $objectFileInput = null; $objectFileOutput = null; $slowDownRead = 0; $slowDownWrite = 0; if ($extractRequest->getPartialDataFromAjaxRequest->status) { $fileObject->rewind(); $fileObject->seek($extractRequest->getPartialDataFromAjaxRequest->indexKey); if (($extractSortby || $hasRestoreParts) && !empty($extractRequest->getPartialDataFromAjaxRequest->countItem)) { $countItems = $extractRequest->getPartialDataFromAjaxRequest->countItem; } } while ($fileObject->valid()) { $line = trim($fileObject->readAndMoveNext()); $indexKey = $fileObject->key(); if (!$extractSortby && !$hasRestoreParts) { $countItems = $indexKey > $totalIndex ? $totalIndex : $indexKey; } if (empty($line)) { continue; } if (!($match = $this->parseFileIndexItem($line))) { continue; } $pathIdentifier = $match[1]; $itemFile = WithIdentifier::replaceIdentifierPath($pathIdentifier); $itemOffset = (int)$match[2]; $itemSize = (int)$match[3]; if ($extractSortby) { $countItems++; if ($this->filterBackupIndexByIdentifier($pathIdentifier, $extractSortby, $dbfilePath, $countItems)) { continue; } } if ($hasRestoreParts) { if ($this->filterBackupIndexByParts($pathIdentifier, $restorePartData, $dbfilePath)) { continue; } $countItems++; } $totalBytes = $itemOffset + $itemSize; $itemChunkBytes = $this->getChunkBytes($itemSize); $itemExtractBytes = 0; $itemExtractBytesTotal = 0; $isWriteAppend = false; if ( $extractRequest->getPartialDataFromAjaxRequest->status && !empty($extractRequest->getPartialDataFromAjaxRequest->itemOffset) && $extractRequest->getPartialDataFromAjaxRequest->itemOffset > $itemOffset ) { $itemOffset = $extractRequest->getPartialDataFromAjaxRequest->itemOffset; $isWriteAppend = true; } $itemDir = $extractRequest->extractPath . '/' . dirname($itemFile); $itemDir = $this->kernel->normalizePath($itemDir); if (!$this->kernel->mkdir($itemDir, __LINE__)) { return ['success' => false, 'data' => sprintf('Failed to extract: %s', $itemFile), 'saveLog' => true, 'saveLogId' => __METHOD__]; } $extractFile = $extractRequest->extractPath . '/' . $itemFile; $extractFile = $this->kernel->normalizePath($extractFile); $objectFileInputSize = 0; if ($objectFileInput === null) { $objectFileInput = $this->useHandle->file->fileObject($extractRequest->filePath, 'rb'); $objectFileInputSize = $objectFileInput->getSize(); } $objectFileInput->fseek($itemOffset); if ($objectFileOutput === null) { if ($isWriteAppend) { $objectFileOutput = $this->useHandle->file->fileObject($extractFile, 'ab'); $itemExtractBytesTotal = $objectFileOutput->getSize(); } else { $objectFileOutput = $this->useHandle->file->fileObject($extractFile, 'wb'); } } $isSqlFile = $objectFileOutput->isSqlFile() && $match[1] === $dbfilePath; if (!$isRestore && $isNormalizeDbFile && $isSqlFile && empty($dbfilePathFull)) { $dbfilePathFull = $extractFile; } $isLargeItem = $itemSize > ( 10 * $this->kernel::MB_IN_BYTES); $itemTimerStart = microtime(true); $setPartialData = []; $partialDataText = null; while ($objectFileInput->valid() && $objectFileInput->ftell() < $totalBytes && $itemExtractBytesTotal < $itemSize) { $isMemoryExceeded = $this->kernel->isMemoryExceeded(); if ($this->hasCancelRequest()) { unlink($extractFile); return ['success' => false, 'data' => 'The backup extraction was cancelled', 'isCancelled' => true]; } $itemExtractBytesLeft = $itemSize - $itemExtractBytesTotal; $itemChunkBytes = $this->getChunkBytes($itemExtractBytesLeft, $itemChunkBytes); $itemExtractContent = $objectFileInput->fread($itemChunkBytes); $itemExtractBytes = $objectFileOutput->fwrite($itemExtractContent); if ($itemExtractBytes === false || $itemExtractBytes < 0) { return ['success' => false, 'data' => sprintf('Failed to extract: %s', $itemFile), 'isCancelled' => true]; } $itemExtractBytesTotal += $itemExtractBytes; $isBreakExtractBytes = false; if ($isMemoryExceeded || ( (int)$itemExtractBytesTotal > 0 && $this->kernel->isTimeExceed($this->meta->maxProcessingTime, $itemTimerStart))) { $isBreakExtractBytes = true; if ($countItems > 1) { $countItems--; } } $setPartialData = [ 'status' => self::STATUS_DOING_BACKUP_EXTRACTION, 'totalIndex' => $totalIndex, 'indexKey' => $indexKey, 'itemOffset' => $objectFileInput->ftell(), 'totalBytes' => $totalBytes, 'isLargeItem' => $isLargeItem, 'isMemoryExceeded' => $isMemoryExceeded, 'countItem' => $countItems, 'countRetry' => 0 ]; $partialDataText = $this->getDataText($extractSortby, $hasRestoreParts, $countItems, $itemExtractBytesTotal, $totalBytes, $itemSize, $totalIndex, $objectFileInputSize); $partialDataText .= "\n" . sprintf("> %s\n> %s/%s", $itemFile, $this->kernel->sizeFormat($itemExtractBytesTotal), $this->kernel->sizeFormat($itemSize)); if ($isBreakExtractBytes) { return ['success' => false, 'data' => $partialDataText, 'partialData' => $setPartialData, 'isMemoryExceeded' => $isMemoryExceeded]; } $slowDownWrite = $this->getSlowDownWrite($itemExtractBytesTotal, $slowDownWrite); $slowDownWrite++; } $objectFileInput = null; $objectFileOutput = null; $this->kernel->chmod($extractFile, false, __LINE__); $isMemoryExceeded = $this->kernel->isMemoryExceeded(); $maxProcessingTimeOffset = $this->meta->maxProcessingTime - 2; if (!empty($setPartialData['status']) && $partialDataText !== null && ( $isMemoryExceeded || $itemExtractBytesTotal > 0 && $this->kernel->isTimeExceed($maxProcessingTimeOffset, $itemTimerStart))) { return ['success' => false, 'data' => $partialDataText, 'partialData' => $setPartialData, 'isMemoryExceeded' => $isMemoryExceeded]; } if ($itemExtractBytesTotal > 0 && $slowDownRead >= 800) { $slowDownRead = 0; usleep(5000); } $slowDownRead++; } } catch (\Throwable $e) { return ['success' => false, 'data' => $e->getMessage(), 'saveLog' => $e, 'saveLogId' => __METHOD__]; } $fileObject = null; $responseText = sprintf('Extracting files was successful: %d files extracted', $countItems); if (!$isRestore && $isNormalizeDbFile && !empty($dbfilePathFull)) { $setPartialData = [ 'status' => self::STATUS_DOING_NORMALIZE_DB_FILE, ]; $text = $responseText; $this->kernel->log($text, __METHOD__); $this->useHandle->cache->put('dbfiletag', $dbfilePathFull, 'dbfilepath'); $this->useHandle->cache->put('dbfiletag', $text, 'extractsuccess'); $text .= "\nNormalizing database file in progress"; return ['success' => false, 'data' => $text, 'partialData' => $setPartialData]; } return ['success' => true, 'data' => $responseText, 'saveLog' => true, 'saveLogId' => __METHOD__, 'isCompleted' => true]; } public function extractItem(): array { clearstatcache(); $extractRequest = $this->validateExtractRequest(); if ($extractRequest['success'] === false) { return $this->validateExtractRequest(); } $extractRequest = (object)$extractRequest['data']; if ( empty($this->meta->dataPost['offset-data']) || !filter_var($this->meta->dataPost['offset-data'], FILTER_DEFAULT, FILTER_REQUIRE_ARRAY) ) { return ['success' => false, 'data' => 'Invalid offset data', 'saveLog' => true, 'saveLogId' => __METHOD__]; } $offsetData = array_map(function ($data) { return (int)$data; }, $this->meta->dataPost['offset-data']); $offsetDataPartial = $offsetData; $this->kernel->maxExecutionTime($this->kernel::MAX_TIMEOUT_EXTRACT); if ($extractRequest->getPartialDataFromAjaxRequest->status === self::STATUS_DOING_NORMALIZE_DB_FILE) { return $this->normalizeDatabaseFile($extractRequest); } if ( $extractRequest->getPartialDataFromAjaxRequest->status === self::STATUS_EXTRACTION_NOT_STARTED && !empty($this->meta->dataPost['extract-path-overwrite']) ) { $this->useHandle->file->emptyDir($extractRequest->extractPath); } $normalizeDbFile = !empty($this->meta->dataPost['normalize-db-file']); static $dbfilePathFull = null; $objectFileIndex = $this->useHandle->file->fileObject($extractRequest->fileIndexPath, 'rb'); $totalIndex = count($offsetData); $line = ''; $objectFileInput = null; $objectFileOutput = null; $slowDownWrite = 0; $countItem = 0; foreach ($offsetData as $num => $index) { $objectFileIndex->rewind(); $objectFileIndex->seekUseParent($index); $line = trim($objectFileIndex->fgetsUseParent()); if (empty($line)) { if (!$objectFileIndex->eof()) { continue; } $objectFileIndex->seekUseParent($index - 1); $line = trim($objectFileIndex->fgetsUseParent()); if (empty($line)) { continue; } } if (!($match = $this->parseFileIndexItem($line))) { continue; } $itemFile = WithIdentifier::replaceIdentifierPath($match[1]); $itemOffset = (int)$match[2]; $itemSize = (int)$match[3]; $totalBytes = $itemOffset + $itemSize; $itemChunkBytes = $this->getChunkBytes($itemSize); $itemExtractBytes = 0; $itemExtractBytesTotal = 0; $isWriteAppend = false; if ( $extractRequest->getPartialDataFromAjaxRequest->status && $extractRequest->getPartialDataFromAjaxRequest->itemOffset > $itemOffset ) { $itemOffset = $extractRequest->getPartialDataFromAjaxRequest->itemOffset; $isWriteAppend = true; } $itemDir = $extractRequest->extractPath . '/' . dirname($itemFile); $itemDir = $this->kernel->normalizePath($itemDir); if (!$this->kernel->mkdir($itemDir, __LINE__)) { return ['success' => false, 'data' => sprintf('Failed to extract: %s', $itemFile),'saveLog' => true, 'saveLogId' => __METHOD__]; } $extractFile = $extractRequest->extractPath . '/' . $itemFile; $extractFile = $this->kernel->normalizePath($extractFile); if ($objectFileInput === null) { $objectFileInput = $this->useHandle->file->fileObject($extractRequest->filePath, 'rb'); } $objectFileInput->fseek($itemOffset); if ($objectFileOutput === null) { if ($isWriteAppend) { $objectFileOutput = $this->useHandle->file->fileObject($extractFile, 'ab'); $itemExtractBytesTotal = $objectFileOutput->getSize(); } else { $objectFileOutput = $this->useHandle->file->fileObject($extractFile, 'wb'); } } $isLargeItem = $itemSize > ( 10 * $this->kernel::MB_IN_BYTES); $itemTimerStart = microtime(true); $setPartialData = []; $partialDataText = null; $isSqlFile = $objectFileOutput->isSqlFile(); if ($isSqlFile && $normalizeDbFile) { $dbfilePathFull = $extractFile; } while ($objectFileInput->valid() && $objectFileInput->ftell() < $totalBytes && $itemExtractBytesTotal < $itemSize) { $isMemoryExceeded = $this->kernel->isMemoryExceeded(); $currentOffset = $objectFileInput->ftell(); $setPartialData = [ 'status' => self::STATUS_DOING_BACKUP_EXTRACTION, 'indexKey' => $index, 'itemOffset' => $currentOffset, 'totalBytes' => $totalBytes, 'isLargeItem' => $isLargeItem, 'isMemoryExceeded' => $isMemoryExceeded, 'countItem' => $countItem ]; $countItemText = ( $countItem === 0 ? 1 : $countItem) . ($countItem > 1 ? ' files' : ' file'); $progressSize = $itemExtractBytesTotal; $partialDataText = sprintf( 'Extracting %s size of %s. Elapsed time: <span id="elapsedtime"><!--{{elapsedtime}}--></span>', $this->kernel->sizeFormat($progressSize), $countItemText ); $partialDataText .= "\n"; $partialDataText .= sprintf( "> %s\n> %s/%s", $itemFile, $this->kernel->sizeFormat($itemExtractBytesTotal), $this->kernel->sizeFormat($itemSize) ); if ($isMemoryExceeded || $this->kernel->isTimeExceed($this->meta->maxProcessingTime, $itemTimerStart)) { return ['success' => false, 'data' => $partialDataText, 'partialData' => $setPartialData, 'offsetData' => $offsetDataPartial, 'isMemoryExceeded' => $isMemoryExceeded]; } if ($this->hasCancelRequest()) { unlink($extractFile); return ['success' => false, 'data' => 'The backup extraction was cancelled', 'isCancelled' => true]; } $itemExtractBytesLeft = $itemSize - $itemExtractBytesTotal; $itemChunkBytes = $this->getChunkBytes($itemExtractBytesLeft, $itemChunkBytes); $itemExtractContent = $objectFileInput->fread($itemChunkBytes); $itemExtractBytes = $objectFileOutput->fwrite($itemExtractContent, $this->kernel->getMemoryLimit()); if ($itemExtractBytes === false || $itemExtractBytes < 0) { return ['success' => false, 'data' => sprintf('Failed to extract: %s', $itemFile), 'saveLog' => true, 'saveLogId' => __METHOD__]; } $itemExtractBytesTotal += $itemExtractBytes; $slowDownWrite = $this->getSlowDownWrite($itemExtractBytesTotal, $slowDownWrite); $slowDownWrite++; } $objectFileInput = null; $objectFileOutput = null; $this->kernel->chmod($extractFile, false, __LINE__); if (isset($offsetDataPartial[$num])) { unset($offsetDataPartial[$num]); } $countItem++; } $objectFileIndex = null; if ($normalizeDbFile && !empty($dbfilePathFull)) { $setPartialData = [ 'status' => self::STATUS_DOING_NORMALIZE_DB_FILE, ]; $text = 'Extracted ' . $totalIndex . ' files was successful'; $this->useHandle->cache->put('dbfiletag', $dbfilePathFull, 'dbfilepath'); $this->useHandle->cache->put('dbfiletag', $text, 'extractsuccess'); $text .= "\nNormalizing database file in progress"; return ['success' => false, 'data' => $text, 'partialData' => $setPartialData]; } return ['success' => true, 'data' => 'Extracted ' . $totalIndex . ' files was successful', 'saveLog' => true, 'saveLogId' => __METHOD__, 'isCompleted' => true]; } public function processStop(): array { if ($this->useHandle->cache->put('extractstop', time())) { return ['success' => true, 'data' => 'Send signal to stop the process', 'isCancelled' => true]; } return ['success' => false, 'data' => 'Failed to stop the process', 'isCancelled' => true]; } private function getDataText($extractSortby, bool $hasRestoreParts, int $countItem, int $itemExtractBytesTotal, int $totalBytes, int $itemSize, int $totalIndex, int $objectFileInputSize): string { if ($extractSortby || $hasRestoreParts) { $countItemText = $countItem . ($countItem > 1 ? ' files' : ' file'); $progressSize = $countItem === 1 ? $itemExtractBytesTotal : ($totalBytes - $itemSize) + $itemExtractBytesTotal; $partialDataText = sprintf( 'Extracting %s size of %s. Elapsed time: <span id="elapsedtime"><!--{{elapsedtime}}--></span>', $this->kernel->sizeFormat($progressSize), $countItemText ); } else { $countItemText = $countItem . '/' . $totalIndex . ($countItem > 1 ? ' files' : ' file'); $progressSize = ($totalBytes - $itemSize) + $itemExtractBytesTotal; $progressPercentage = round(abs((($progressSize / $objectFileInputSize) * 100))); $partialDataText = sprintf( 'Extracting %s/%s, %s%% size of %s. Elapsed time: <span id="elapsedtime"><!--{{elapsedtime}}--></span>', $this->kernel->sizeFormat($progressSize), $this->kernel->sizeFormat($objectFileInputSize), $progressPercentage, $countItemText ); } return $partialDataText; } private function getIndex($extractRequest, $objectFileIndex, int $totalFiles): int { if (!empty($extractRequest->getPartialDataFromAjaxRequest->totalIndex)) { $totalIndex = $extractRequest->getPartialDataFromAjaxRequest->totalIndex; } else { $totalIndex = $objectFileIndex->totalLines(); if ($totalIndex !== $totalFiles) { $this->kernel->log(sprintf('FileObject totalLines() "%d" does not match with backupMeta totalFiles "%d"', $totalIndex, $totalFiles), __METHOD__); $totalIndex = $totalFiles; } } return (int)$totalIndex; } private function getSlowDownWrite(int $itemExtractBytesTotal, int $slowDownWrite): int { if ((int)$itemExtractBytesTotal > 0 && $slowDownWrite >= 800) { $slowDownWrite = 0; usleep(5000); } return $slowDownWrite; } }
    final class File { private $kernel; private $meta; const SCAN_CURRENT_DIR_ONLY = 0; const SCAN_UP_TO_ONE_DIR = 1; const SCAN_ALL_DIR = -1; public function __construct(\WPStagingRestorer $kernel) { $this->kernel = $kernel; $this->meta = $this->kernel->getMeta(); } public function fileObject(string $filePath, string $mode = 'rb'): FileObject { return new FileObject($filePath, $mode); } public function isLineBreak($string): bool { return empty($string) || in_array($string, ["\r", "\n", "\r\n", "\n\r", chr(13), chr(10), PHP_EOL]) || preg_match('@^\s+' . chr(10) . '$@', $string); } public function isDirEmpty(string $dirPath): bool { if (!is_dir($dirPath)) { return true; } return !(new \FilesystemIterator($dirPath))->valid(); } public function isOutsideRootPath(string $dirPath): bool { $dirPath = $this->kernel->normalizePath($dirPath); $rootPath = $this->kernel->normalizePath($this->meta->rootPath); return $rootPath !== substr($dirPath, 0, strlen($rootPath)); } public function isRootPath(string $dirPath): bool { $dirPath = $this->kernel->normalizePath($dirPath); $rootPath = $this->kernel->normalizePath($this->meta->rootPath); return $dirPath === $rootPath; } private function isPathExclude($path, $exclusion): bool { if (empty($exclusion) || !is_array($exclusion)) { return false; } foreach ($exclusion as $item) { if (strpos($path, $item) !== false) { return true; } } return false; } public function moveDir(string $srcPath, string $dstPath, array $exclude = [], bool $allowOutsideRootPath = false) { if (!is_dir($srcPath)) { return false; } if ($this->isDirEmpty($srcPath)) { return false; } if (!$allowOutsideRootPath && $this->isOutsideRootPath($dstPath)) { return false; } if (!$allowOutsideRootPath && $this->isOutsideRootPath($dstPath)) { return false; } $this->kernel->mkdir($dstPath, __LINE__); $countFile = 0; try { $dirIterator = new \RecursiveIteratorIterator( new \RecursiveDirectoryIterator($srcPath, \RecursiveDirectoryIterator::SKIP_DOTS), \RecursiveIteratorIterator::SELF_FIRST ); foreach ($dirIterator as $item) { $filePath = $this->kernel->normalizePath($dstPath . '/' . $dirIterator->getSubPathname()); if ($item->isDir()) { $this->kernel->mkdir($filePath, __LINE__); } else { $itemCopy = $this->kernel->normalizePath($item->getPathname()); if ($this->isPathExclude($itemCopy, $exclude)) { continue; } $this->kernel->mkdir(dirname($filePath), __LINE__); if (rename($itemCopy, $filePath)) { $countFile++; } } } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); return false; } $this->kernel->rmdir($srcPath, __LINE__); return $countFile; } public function removeDir(string $dirPath, array $exclude = [], bool $removeEmpty = true): bool { if (!is_dir($dirPath)) { return true; } if ($this->isRootPath($dirPath) || $this->isOutsideRootPath($dirPath)) { return false; } if (!is_writable($dirPath) || $dirPath === '/' || substr($dirPath, 0, 2) === '..') { return false; } try { if ($removeEmpty && $this->isDirEmpty($dirPath)) { return $this->kernel->rmdir($dirPath, __LINE__); } $dirIterator = new \RecursiveIteratorIterator( new \RecursiveDirectoryIterator($dirPath, \RecursiveDirectoryIterator::SKIP_DOTS), \RecursiveIteratorIterator::CHILD_FIRST ); foreach ($dirIterator as $item) { $itemPath = $this->kernel->normalizePath($item->getPathname()); if ($this->isPathExclude($itemPath, $exclude)) { continue; } if ($item->isDir()) { $this->kernel->rmdir($itemPath, __LINE__); } else { $this->kernel->unlink($itemPath, __LINE__); } } if ($removeEmpty && $this->isDirEmpty($dirPath)) { $this->kernel->rmdir($dirPath, __LINE__); } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); return false; } return true; } public function emptyDir(string $srcDir): bool { $srcDir = $this->kernel->normalizePath($srcDir); if (!is_dir($srcDir) || $this->isOutsideRootPath($srcDir) || $this->isRootPath($srcDir)) { return false; } return $this->removeDir($srcDir); } public function removeAppFile() { $this->removeDir($this->meta->tmpPath); $this->kernel->unlink($this->meta->rootPath . '/' . $this->meta->appFile, __LINE__); return true; } public function opcacheFlush(string $filePath, bool $force = true): bool { static $canInvalidate = null; if ( $canInvalidate === null && function_exists('opcache_invalidate') && ( !ini_get('opcache.restrict_api') || !empty($this->meta->dataServer['SCRIPT_FILENAME']) && stripos(realpath($this->meta->dataServer['SCRIPT_FILENAME']), ini_get('opcache.restrict_api')) === 0 ) ) { $canInvalidate = true; } if (!$canInvalidate || strtolower(substr($filePath, -4)) !== '.php') { return false; } return opcache_invalidate($filePath, $force); } public function opcacheFlushDir(string $dirPath): bool { $dirPath = realpath($dirPath); if (empty($dirPath) || !is_dir($dirPath) || !is_readable($dirPath) || $this->isDirEmpty($dirPath)) { return false; } try { foreach ($this->scanFiles($dirPath, -1, '@\.php$@') as $file) { $this->opcacheFlush($file, true); } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); return false; } return true; } public function scanFiles(string $dirPath, int $maxDepth = 0, $pattern = null) { $dirPath = realpath($dirPath); if ($dirPath === false || !is_dir($dirPath) || !is_readable($dirPath)) { return []; } $pattern = !empty($pattern) ? $pattern : '@\.wpstg$@'; $recursiveDirectoryIteratorFlags = \FilesystemIterator::SKIP_DOTS | \RecursiveDirectoryIterator::KEY_AS_FILENAME | \RecursiveDirectoryIterator::CURRENT_AS_FILEINFO; $recursiveDirectoryIterator = new \RecursiveDirectoryIterator($dirPath, $recursiveDirectoryIteratorFlags); $recursiveIteratorIterator = new \RecursiveIteratorIterator($recursiveDirectoryIterator); $recursiveIteratorIterator->setMaxDepth($maxDepth); $regexIterator = new \RegexIterator($recursiveIteratorIterator, $pattern, \RegexIterator::MATCH, \RegexIterator::USE_KEY); return $regexIterator; } public function preventAccessToDirectory(string $path) { $path = $this->kernel->normalizePath($path); if (!file_exists($path . '/index.html')) { file_put_contents($path . '/index.html', '<!-- ' . time() . ' -->'); } if (!file_exists($path . '/index.php')) { file_put_contents($path . '/index.php', '<?php // ' . time()); } if (empty($this->meta->dataServer['SERVER_SOFTWARE'])) { return; } if ( (stripos($this->meta->dataServer['SERVER_SOFTWARE'], 'Apache') !== false || stripos($this->meta->dataServer['SERVER_SOFTWARE'], 'LiteSpeed') !== false) && !file_exists($path . '/.htaccess') ) { file_put_contents($path . '/.htaccess', 'Deny from all', LOCK_EX); } if (stripos(PHP_OS, 'WIN') === 0 && !file_exists($path . '/web.config')) { $xml = '<?xml version="1.0"?>' . PHP_EOL; $xml .= '<configuration>' . PHP_EOL; $xml .= '   <system.web>' . PHP_EOL; $xml .= '       <authorization>' . PHP_EOL; $xml .= '           <deny users="*" />' . PHP_EOL; $xml .= '       </authorization>' . PHP_EOL; $xml .= '   </system.web>' . PHP_EOL; $xml .= '</configuration>' . PHP_EOL; file_put_contents($path . '/web.config', $xml, LOCK_EX); } } }
    final class FileObject extends \SplFileObject
    {
        private $totalLines = null;
        private $fgetsUsedOnKey0 = false;
        private $fseekUsed = false;
        public function __construct(string $fullPath, string $openMode)
        {
            try {
                parent::__construct($fullPath, $openMode);
            } catch (\Throwable $e) {
                throw $e;
            }
        }
        public function totalLines($useParent = false): int
        {
            if ($this->totalLines !== null) {
                return $this->totalLines;
            }
            if ($useParent) {
                $currentKey = $this->keyUseParent();
                $this->seekUseParent(PHP_INT_MAX);
                $this->totalLines = $this->keyUseParent();
                if ($currentKey < 0) {
                    $currentKey = 0;
                }
                $this->seekUseParent($currentKey);
            } else {
                $currentKey = $this->key();
                if ($currentKey < 0) {
                    $currentKey = 0;
                }
                $this->seek(PHP_INT_MAX);
                $this->totalLines = $this->key();
                $this->seek($currentKey);
            }
            if ($this->totalLines > 0) {
                if (PHP_VERSION === '8.2.0RC3' || version_compare(PHP_VERSION, '8.2.0', '>=')) {
                    $this->totalLines += 1;
                }
                if (version_compare(PHP_VERSION, '8.1', '>') && version_compare(PHP_VERSION, '8.1.11', '<=')) {
                    $this->totalLines += 1;
                }
            }
            return $this->totalLines;
        }
        #[\ReturnTypeWillChange]
        public function seek($offset)
        {
            if ($offset < 0) {
                throw new \Exception("Can't seek file: " . $this->getPathname() . " to negative offset: $offset");
            }
            $this->fseekUsed       = false;
            $this->fgetsUsedOnKey0 = false;
            if ($offset === 0 || version_compare(PHP_VERSION, '8.0.1', '<')) {
                parent::seek($offset);
                return;
            }
            $offset -= 1;
            if ($this->totalLines !== null && $offset >= $this->totalLines) {
                $offset += 1;
            }
            $originalFlags = $this->getFlags();
            $newFlags      = $originalFlags & ~self::READ_AHEAD;
            $this->setFlags($newFlags);
            parent::seek($offset);
            if ($this->eof()) {
                $this->current();
                $this->totalLines = $this->key();
                return;
            }
            $this->current();
            $this->next();
            $this->current();
            $this->setFlags($originalFlags);
        }
        public function fgets(): string
        {
            if ($this->key() === 0 || version_compare(PHP_VERSION, '8.0.1', '<')) {
                $this->fgetsUsedOnKey0 = true;
                return parent::fgets();
            }
            $originalFlags = $this->getFlags();
            $newFlags      = $originalFlags & ~self::READ_AHEAD;
            $this->setFlags($newFlags);
            $line = $this->current();
            $this->next();
            if (version_compare(PHP_VERSION, '8.0.19', '<')) {
                $line = $this->current();
            }
            if (version_compare(PHP_VERSION, '8.1', '>') && version_compare(PHP_VERSION, '8.1.6', '<')) {
                $line = $this->current();
            }
            if (!$this->fseekUsed) {
                $line = $this->current();
            }
            $this->setFlags($originalFlags);
            return $line;
        }
        #[\ReturnTypeWillChange]
        public function key()
        {
            if ($this->fgetsUsedOnKey0 || version_compare(PHP_VERSION, '8.0.19', '<')) {
                return parent::key();
            }
            if (version_compare(PHP_VERSION, '8.1', '>') && version_compare(PHP_VERSION, '8.1.6', '<')) {
                return parent::key();
            }
            return parent::key() - 1;
        }
        #[\ReturnTypeWillChange]
        public function fseek($offset, $whence = SEEK_SET)
        {
            if (version_compare(PHP_VERSION, '8.0.19', '<')) {
                return parent::fseek($offset, $whence);
            }
            if (version_compare(PHP_VERSION, '8.1', '>') && version_compare(PHP_VERSION, '8.1.6', '<')) {
                return parent::fseek($offset, $whence);
            }
            for ($i = 0; $i < 3; $i++) {
                parent::fseek(0);
                $this->fgets();
            }
            $this->fseekUsed = true;
            return parent::fseek($offset, $whence);
        }
        public function readAndMoveNext($useFgets = false): string
        {
            if ($useFgets && version_compare(PHP_VERSION, '8.0.1', '<')) {
                return parent::fgets();
            }
            $originalFlags = $this->getFlags();
            $newFlags      = $originalFlags & ~self::READ_AHEAD;
            $this->setFlags($newFlags);
            $line = $this->current();
            $this->next();
            $this->setFlags($originalFlags);
            return $line;
        }
        public function isSqlFile(): bool
        {
            return $this->getExtension() === 'sql';
        }
        public function fgetsUseParent(): string
        {
            return parent::fgets();
        }
        public function keyUseParent(): int
        {
            return parent::key();
        }
        public function seekUseParent($offset)
        {
            parent::seek($offset);
        }
    }
    final class Restorer { private $kernel; private $meta; private $useHandle; private $partialData; private $extractPath; private $statusFile; private $hasRestoreParts; private $isOverwriteParts; const RESTORE_PART_UPLOADS = 1; const RESTORE_PART_PLUGINS = 2; const RESTORE_PART_THEMES = 3; const RESTORE_PART_LANG = 4; const RESTORE_PART_WPCONTENT = 5; const RESTORE_PART_DELAY_DATABASE = 6; const RESTORE_PART_DATABASE = 7; const RESTORE_PART_RENAME_TABLES = 8; const RESTORE_PART_DROPINS = 9; const RESTORE_PART_MU_PLUGINS = 10; const RESTORE_PART_DONE = 11; const RESTORE_PART_FALSE = 0; const NO_RESTORATION_PROCESS_YET = 0; const STATUS_DOING_RESTORATION = 2; public function __construct(\WPStagingRestorer $kernel) { $this->kernel = $kernel; $this->meta = $this->kernel->getMeta(); $this->useHandle = $this->kernel->getHandle(__CLASS__, ['cache', 'file', 'extractor', 'wpcore', 'withIdentifier']); $this->extractPath = $this->meta->tmpPath . '/restore/'; } private function getPath(string $identifier) { $srcPath = $this->useHandle->withIdentifier::getRelativePath($identifier); $dstPath = $this->useHandle->withIdentifier->getAbsolutePath($identifier); return (object)['src' => $srcPath, 'dst' => $dstPath]; } private function restoreUploads(): array { $this->partialData->restoreNextPart = self::RESTORE_PART_PLUGINS; if ($this->hasRestoreParts->uploads === self::RESTORE_PART_FALSE && $this->hasRestoreParts->database === self::RESTORE_PART_FALSE) { return ['sucess' => false, 'data' => '', 'partialData' => $this->partialData]; } $identifier = WithIdentifier::IDENTIFIER_UPLOADS; $getPath = $this->getPath($identifier); if ($getPath->src === $identifier || $getPath->dst === $identifier) { return ['sucess' => false, 'data' => 'Failed to restore Media Library: Could not get a valid path', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $getPath->src = $this->kernel->normalizePath($this->extractPath . '/' . $getPath->src); $getPath->dst = $this->kernel->normalizePath($getPath->dst); if (!empty($this->isOverwriteParts->uploads) && !$this->useHandle->file->removeDir($getPath->dst, ['wp-staging/backups', 'wp-staging/cache'])) { return ['sucess' => false, 'data' => 'Failed to restore Media Library: Unable to overwrite directory', 'saveLogId' => __METHOD__, 'isAborted' => true]; } if (($countFile = $this->useHandle->file->moveDir($getPath->src, $getPath->dst)) === false) { $countFile = 0; } $text = ''; if ($countFile > 0) { $text = sprintf('Restoring Media files: %d files restored', $countFile); $this->kernel->log($text, __METHOD__); } return ['sucess' => false, 'data' => $text, 'partialData' => $this->partialData]; } private function restorePlugins(): array { $this->partialData->restoreNextPart = self::RESTORE_PART_THEMES; if ($this->hasRestoreParts->plugins === self::RESTORE_PART_FALSE) { return ['sucess' => false, 'data' => '', 'partialData' => $this->partialData]; } $identifier = WithIdentifier::IDENTIFIER_PLUGINS; $getPath = $this->getPath($identifier); if ($getPath->src === $identifier || $getPath->dst === $identifier) { return ['sucess' => false, 'data' => 'Failed to restore Plugins: Could not get a valid path', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $getPath->src = $this->kernel->normalizePath($this->extractPath . '/' . $getPath->src); $getPath->dst = $this->kernel->normalizePath($getPath->dst); $exclude = [ 'wp-staging-dev/', 'wp-staging-pro/', 'wp-staging/' ]; if (!empty($this->isOverwriteParts->plugins) && !$this->useHandle->file->removeDir($getPath->dst, $exclude)) { return ['sucess' => false, 'data' => 'Failed to restore Plugins: Unable to overwrite directory', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $exclude = []; foreach (['wp-staging-dev/', 'wp-staging-pro/', 'wp-staging/'] as $dir) { if (file_exists($getPath->dst . '/' . $dir)) { $exclude[] = $dir; } } if (($countFile = $this->useHandle->file->moveDir($getPath->src, $getPath->dst, $exclude)) === false) { $countFile = 0; } $text = ''; if ($countFile > 0) { $text = sprintf('Restoring Plugins: %d files restored', $countFile); $this->kernel->log($text, __METHOD__); } return ['sucess' => false, 'data' => $text, 'partialData' => $this->partialData]; } private function restoreThemes(): array { $this->partialData->restoreNextPart = self::RESTORE_PART_LANG; if ($this->hasRestoreParts->themes === self::RESTORE_PART_FALSE) { return ['sucess' => false, 'data' => '', 'partialData' => $this->partialData]; } $identifier = WithIdentifier::IDENTIFIER_THEMES; $getPath = $this->getPath($identifier); if ($getPath->src === $identifier || $getPath->dst === $identifier) { return ['sucess' => false, 'data' => 'Failed to restore Themes: Could not get a valid path', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $getPath->src = $this->kernel->normalizePath($this->extractPath . '/' . $getPath->src); $getPath->dst = $this->kernel->normalizePath($getPath->dst); if (!empty($this->isOverwriteParts->themes) && !$this->useHandle->file->removeDir($getPath->dst)) { return ['sucess' => false, 'data' => 'Failed to restore Themes: Unable to overwrite directory', 'saveLogId' => __METHOD__, 'isAborted' => true]; } if (($countFile = $this->useHandle->file->moveDir($getPath->src, $getPath->dst)) === false) { $countFile = 0; } $text = ''; if ($countFile > 0) { $text = sprintf('Restoring Themes: %d files restored', $countFile); $this->kernel->log($text, __METHOD__); } return ['sucess' => false, 'data' => $text, 'partialData' => $this->partialData]; } private function restoreLang(): array { $this->partialData->restoreNextPart = self::RESTORE_PART_WPCONTENT; if ($this->hasRestoreParts->lang === self::RESTORE_PART_FALSE) { return ['sucess' => false, 'data' => '', 'partialData' => $this->partialData]; } $identifier = WithIdentifier::IDENTIFIER_LANG; $getPath = $this->getPath($identifier); if ($getPath->src === $identifier || $getPath->dst === $identifier) { return ['sucess' => false, 'data' => 'Failed to restore Language files: Could not get a valid path', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $getPath->src = $this->kernel->normalizePath($this->extractPath . '/' . $getPath->src); $getPath->dst = $this->kernel->normalizePath($getPath->dst); if (!empty($this->isOverwriteParts->lang) && !$this->useHandle->file->removeDir($getPath->dst)) { return ['sucess' => false, 'data' => 'Failed to restore Language files: Unable to overwrite directory', 'saveLogId' => __METHOD__, 'isAborted' => true]; } if (($countFile = $this->useHandle->file->moveDir($getPath->src, $getPath->dst)) === false) { $countFile = 0; } $text = ''; if ($countFile > 0) { $text = sprintf('Restoring Language files: %d files restored', $countFile); $this->kernel->log($text, __METHOD__); } return ['sucess' => false, 'data' => $text, 'partialData' => $this->partialData]; } private function restoreWpContent(): array { $this->partialData->restoreNextPart = self::RESTORE_PART_DELAY_DATABASE; if ($this->hasRestoreParts->wpcontent === self::RESTORE_PART_FALSE) { return ['sucess' => false, 'data' => '', 'partialData' => $this->partialData]; } $identifier = WithIdentifier::IDENTIFIER_WPCONTENT; $getPath = $this->getPath($identifier); if ($getPath->src === $identifier || $getPath->dst === $identifier) { return ['sucess' => false, 'data' => 'Failed to restore other files in wp-content: Could not get a valid path', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $getPath->src = $this->kernel->normalizePath($this->extractPath . '/' . $getPath->src); $getPath->dst = $this->kernel->normalizePath($getPath->dst); $exclude = [ $this->useHandle->withIdentifier::getRelativePath(WithIdentifier::IDENTIFIER_UPLOADS), $this->useHandle->withIdentifier::getRelativePath(WithIdentifier::IDENTIFIER_THEMES), $this->useHandle->withIdentifier::getRelativePath(WithIdentifier::IDENTIFIER_PLUGINS), $this->useHandle->withIdentifier::getRelativePath(WithIdentifier::IDENTIFIER_MUPLUGINS), $this->useHandle->withIdentifier::getRelativePath(WithIdentifier::IDENTIFIER_LANG), ]; if (!empty($this->isOverwriteParts->wpcontent) && !$this->useHandle->file->removeDir($getPath->dst, $exclude)) { return ['sucess' => false, 'data' => 'Failed to restore other files in wp-content: Unable to overwrite directory', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $excludeCopy = array_merge($exclude, $this->useHandle->extractor->getDropinsFile()); if (($countFile = $this->useHandle->file->moveDir($getPath->src, $getPath->dst, $excludeCopy)) === false) { $countFile = 0; } $text = ''; if ($countFile > 0) { $text = sprintf('Restoring other files in wp-content: %d files restored', $countFile); $this->kernel->log($text, __METHOD__); } return ['sucess' => false, 'data' => $text, 'partialData' => $this->partialData]; } private function restoreDatabase(): array { $this->partialData->restoreNextPart = self::RESTORE_PART_RENAME_TABLES; if ($this->hasRestoreParts->database === self::RESTORE_PART_FALSE) { return ['sucess' => false, 'data' => '', 'partialData' => $this->partialData]; } if (empty($this->meta->dataPost['dbsql-filepath'])) { return ['success' => false, 'data' => 'Invalid request. Database File not available', 'saveLogId' => __METHOD__, 'isAborted' => true]; } if (empty($this->meta->dataPost['search-replace-data']) || !filter_var($this->meta->dataPost['search-replace-data'], FILTER_DEFAULT, FILTER_REQUIRE_ARRAY)) { return ['success' => false, 'data' => 'Invalid request. Search Replace data not available', 'saveLogId' => __METHOD__, 'isAborted' => true]; } if (!array_intersect_key($this->meta->dataPost['search-replace-data'], ['backupsiteurl' => 1, 'backuphomeurl' => 1, 'backupwpbakeryactive' => 1, 'siteurl' => 1, 'homeurl' => 1])) { return ['success' => false, 'data' => 'Invalid request. Invalid Search Replace data', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $searchReplaceData = (object)$this->meta->dataPost['search-replace-data']; $isReplaceSite = $searchReplaceData->backupsiteurl !== $searchReplaceData->siteurl || $searchReplaceData->backuphomeurl !== $searchReplaceData->homeurl; $dbSqlFile = $this->kernel->normalizePath($this->meta->rootPath . '/' . WithIdentifier::replaceIdentifierPath($this->meta->dataPost['dbsql-filepath'])); if (!file_exists($dbSqlFile)) { return ['success' => false, 'data' => 'Failed to restore Database: File not available', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $this->kernel->maxExecutionTime($this->kernel::MAX_TIMEOUT_RESTORE); $dbHandle = $this->useHandle->wpcore->dbHandle(); if ($dbHandle->connect() === false) { return ['success' => false, 'data' => sprintf('Failed to restore Database: %s', $dbHandle->response), 'saveLogId' => __METHOD__, 'isAborted' => true]; } $dbPrefix = isset($dbHandle->config->dbprefix) ? $dbHandle->config->dbprefix : 'wp_'; $dbTmpPrefix = $dbHandle::TMP_PREFIX; $objectFileDb = null; $chunkQuery = 0; $totalQuery = 0; try { $objectFileDb = $this->useHandle->file->fileObject($dbSqlFile, 'rb'); $objectFileDb->fgets(); if (!empty($this->partialData->totalQuery)) { $totalQuery = $this->partialData->totalQuery; } else { if (!empty($this->partialData->countRetry) && $this->partialData->countRetry === 1) { $totalQuery = 0; $this->kernel->log('FileObject::totalLines() randomly failing. Response status without total query', __METHOD__); } else { $totalQuery = $objectFileDb->totalLines(); } } $fileSize = $objectFileDb->getSize(); $isLargeQuery = $fileSize > $this->kernel::GB_IN_BYTES; $itemTimerStart = microtime(true); $setPartialData = []; $partialDataText = null; if ($this->partialData->status === self::STATUS_DOING_RESTORATION) { if (empty($this->partialData->isRestoreDb)) { $dbHandle->removeTablesWithPrefix($dbTmpPrefix); } else { $objectFileDb->rewind(); $objectFileDb->seekUseParent($this->partialData->indexKey); } } $emptyQuery = 0; if (!empty($this->partialData->emptyQuery)) { $emptyQuery = $this->partialData->emptyQuery; } $dbHandle->setSession("sql_mode = 'NO_AUTO_VALUE_ON_ZERO'"); $dbHandle->foreignKeyChecksOff(); $isTransaction = false; while ($objectFileDb->valid()) { $query = trim($objectFileDb->fgets()); $indexKey = $objectFileDb->keyUseParent(); if ($this->useHandle->extractor->hasCancelRequest()) { $this->useHandle->wpcore->enableMaintenance(false); $dbHandle->commit(); $dbHandle->close(); return ['success' => false, 'data' => 'The backup restoration was cancelled', 'isCancelled' => true]; } if (empty($query)) { $emptyQuery++; continue; } if ($this->useHandle->file->isLineBreak($query)) { continue; } $queryError = ''; if (!$dbHandle->isExecutableQuery($query, $queryError)) { if (!empty($queryError)) { $dbHandle->commit(); return ['success' => false, 'data' => sprintf('Error: %s. Line %d', $queryError, $objectFileDb->key()), 'saveLogId' => __METHOD__, 'isAborted' => true]; } continue; } $isMemoryExceeded = $this->kernel->isMemoryExceeded(); $currentOffset = $objectFileDb->ftell(); $setPartialData = [ 'status' => self::STATUS_DOING_RESTORATION, 'totalQuery' => $totalQuery, 'indexKey' => $indexKey, 'itemOffset' => $currentOffset, 'isLargeItem' => $isLargeQuery, 'isMemoryExceeded' => $isMemoryExceeded, 'isRestoreDb' => 1, 'restoreNextPart' => self::RESTORE_PART_DATABASE, 'emptyQuery' => $emptyQuery, 'countRetry' => 0 ]; $indexKeyBefore = isset($this->partialData->indexKey) ? $this->partialData->indexKey : 0; $queriesPerSecond = ($indexKey - $indexKeyBefore) / (microtime(true) - $itemTimerStart); $queriesPerSecond = abs($queriesPerSecond); $progressPercentage = null; $progressText = '<!--{{saveResponseTag}}-->Restoring Database: Elapsed time: <span id="elapsedtime"><!--{{elapsedtime}}--></span>' . "\n"; $executedText = sprintf( 'Restoring Database: Executed %s queries (%s queries per second)', number_format($indexKey), number_format($queriesPerSecond) ); if ($totalQuery > 0) { $executedText = sprintf( 'Restoring Database: Executed %s/%s queries (%s queries per second)', number_format($indexKey), number_format($totalQuery), number_format($queriesPerSecond) ); $progressPercentage = ceil(($indexKey / $totalQuery) * 100); if ($progressPercentage === 100 && $indexKey < $totalQuery) { $progressPercentage = 99; } $progressText = sprintf('<!--{{saveResponseTag}}-->Restoring Database: Progress %d%% - Elapsed time: <span id="elapsedtime"><!--{{elapsedtime}}--></span>', $progressPercentage) . "\n"; } $partialDataText = $progressText . $executedText; if ($isMemoryExceeded || $this->kernel->isTimeExceed($this->meta->maxProcessingTime, $itemTimerStart)) { $dbHandle->commit(); return ['success' => false, 'data' => $partialDataText, 'partialData' => $setPartialData, 'isMemoryExceeded' => $isMemoryExceeded]; } $query = str_replace($dbHandle::TMP_PREFIX_FLAG, $dbTmpPrefix, $query); $dbHandle->maybeShortenTableNameForQuery($query); $dbHandle->maybeFixReplaceTableConstraints($query); $dbHandle->replaceTableCollations($query); if (strpos($query, 'INSERT INTO') === 0) { if (!$isTransaction) { $isTransaction = $dbHandle->startTransaction(); } if (strpos($query, $dbHandle::BINARY_FLAG) !== false || strpos($query, $dbHandle::NULL_FLAG) !== false || $isReplaceSite) { $dbHandle->searchReplaceInsertQuery($query, $searchReplaceData); } if (strpos($query, $dbHandle::TMP_PREFIX_FINAL_FLAG) !== false) { $query = str_replace($dbHandle::TMP_PREFIX_FINAL_FLAG, $dbPrefix, $query); } $result = $dbHandle->query($query); if ($result === false && !empty($dbHandle->error())) { $this->kernel->log( [ 'method' => __METHOD__, 'error' => $dbHandle->error(), 'errno' => $dbHandle->errno(), ] ); } if ($totalQuery > 10000 && $chunkQuery > 1000 && ( $isMemoryExceeded || $this->kernel->isTimeExceed($this->meta->maxProcessingTime, $itemTimerStart))) { $dbHandle->commit(); usleep(5000); return ['success' => false, 'data' => $partialDataText, 'partialData' => $setPartialData, 'isMemoryExceeded' => $isMemoryExceeded]; } $chunkQuery++; } else { $dbHandle->commit(); $isTransaction = false; $dbHandle->removeDefiner($query); $dbHandle->removeSqlSecurity($query); $dbHandle->removeAlgorithm($query); $dbHandle->removePageCompression($query); $result = $dbHandle->exec($query); } if ($result === false) { $errorNo = $dbHandle->errno(); $errorMsg = $dbHandle->error(); $requeryResult = $dbHandle->compatibilityFix($errorNo, $errorMsg, $query); if ($requeryResult) { return [ 'sucess' => false, 'data' => sprintf('Restoring Database: Compatibility fixes %d', $errorNo), 'saveLog' => true, 'saveLogId' => __METHOD__, 'partialData' => $this->partialData, 'isAppendResponse' => true ]; } $errorNo = $dbHandle->errno(); $errorMsg = $dbHandle->error(); $dbHandle->commit(); $dbHandle->close(); $this->kernel->log( [ 'method' => __METHOD__, 'error' => $errorMsg, 'errno' => $errorNo, 'query-index' => $indexKey, 'query-100-first' => substr($query, 0, 100), 'query-100-last' => substr($query, -100), ] ); if ($dbHandle->isDuplicateEntry($errorNo, $errorMsg) && !empty($partialDataText) && !empty($setPartialData)) { return ['success' => false, 'data' => $partialDataText, 'partialData' => $setPartialData, 'isMemoryExceeded' => false]; } return ['success' => false, 'data' => sprintf('Error: (%d) %s', $errorNo, $errorMsg), 'saveLogId' => __METHOD__, 'isAborted' => true]; } } $objectFileDb = null; } catch (\Throwable $e) { $dbHandle->close(); $errorMsg = $e->getMessage(); $errorNo = $e->getCode(); if ($dbHandle->isDuplicateEntry($errorNo, $errorMsg) && !empty($partialDataText) && !empty($setPartialData)) { return ['success' => false, 'data' => $partialDataText, 'partialData' => $setPartialData, 'isMemoryExceeded' => false]; } return ['success' => false, 'data' => 'Error: ' . $errorMsg, 'saveLog' => $e, 'saveLogId' => __METHOD__, 'isAborted' => true]; } $dbHandle->commit(); $dbHandle->close(); $countQuery = $indexKey - $emptyQuery; return [ 'sucess' => false, 'data' => sprintf('Restoring Database was successful: Executed %d queries', $indexKey), 'saveLog' => true, 'saveLogId' => __METHOD__, 'partialData' => $this->partialData, 'isAppendResponse' => true ]; } private function renameTables(): array { $this->partialData->restoreNextPart = self::RESTORE_PART_DROPINS; if ($this->hasRestoreParts->database === self::RESTORE_PART_FALSE) { return ['sucess' => false, 'data' => '', 'partialData' => $this->partialData]; } $this->kernel->maxExecutionTime($this->kernel::MAX_TIMEOUT_RESTORE); $dbHandle = $this->useHandle->wpcore->dbHandle(); if ($dbHandle->connect() === false) { return ['success' => false, 'data' => sprintf('Failed to rename Tables: %s', $dbHandle->response), 'saveLogId' => __METHOD__, 'isAborted' => true]; } $dbPrefix = isset($dbHandle->config->dbprefix) ? $dbHandle->config->dbprefix : 'wp_'; $dbTmpPrefix = $dbHandle::TMP_PREFIX; $result = $dbHandle->query('SHOW TABLES LIKE "' . $dbTmpPrefix . '%"'); if (! ($result instanceof \mysqli_result) || (int)$result->num_rows === 0) { return ['sucess' => false, 'data' => 'No tables found to rename', 'partialData' => $this->partialData]; } $countRenamed = 0; $tableCleanup = []; $itemTimerStart = microtime(true); $setPartialData = []; $totalRows = (int)$result->num_rows; try { $dbHandle->foreignKeyChecksOff(); $dbHandle->autocommit(false); $dbHandle->startTransaction(); while ($row = $result->fetch_row()) { if ($this->useHandle->extractor->hasCancelRequest()) { $this->useHandle->wpcore->enableMaintenance(false); $dbHandle->commit(); $dbHandle->autocommit(true); $dbHandle->close(); return ['success' => false, 'data' => 'The backup restoration was cancelled', 'saveLogId' => __METHOD__, 'isCancelled' => true]; } $isMemoryExceeded = $this->kernel->isMemoryExceeded(); $setPartialData = [ 'status' => self::STATUS_DOING_RESTORATION, 'isMemoryExceeded' => $isMemoryExceeded, 'isRestoreDb' => 1, 'restoreNextPart' => self::RESTORE_PART_RENAME_TABLES ]; if ($isMemoryExceeded || $this->kernel->isTimeExceed($this->meta->maxProcessingTime, $itemTimerStart)) { $dbHandle->commit(); return [ 'success' => false, 'data' => sprintf("Renaming Database tables: %d/%d", $countRenamed, $totalRows), 'saveLog' => true, 'saveLogId' => __METHOD__, 'partialData' => $setPartialData, 'isMemoryExceeded' => $isMemoryExceeded ]; } $tableTmp = $row[0]; $tableOld = str_replace($dbTmpPrefix, $dbPrefix, $dbHandle->getTableFromShortName($tableTmp)); $tableCleanup[$tableOld] = 1; if ($dbHandle->exec("DROP TABLE IF EXISTS `" . $tableOld . "`") && $dbHandle->exec("RENAME TABLE `" . $tableTmp . "` to `" . $tableOld . "`")) { $countRenamed++; } } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); $dbHandle->rollback(); $dbHandle->autocommit(true); $dbHandle->close(); return ['sucess' => false, 'data' => 'Renaming Database tables: failed to rename Database Tables', 'saveLogId' => __METHOD__, 'isAborted' => true]; } try { $totalOldTables = count($tableCleanup); $countRemoved = 0; if ($totalOldTables > 0) { $result = $dbHandle->query('SHOW TABLES LIKE "' . $dbPrefix . '%"'); if (($result instanceof \mysqli_result) && (int)$result->num_rows > 0) { while ($row = $result->fetch_row()) { $isMemoryExceeded = $this->kernel->isMemoryExceeded(); $setPartialData = [ 'status' => self::STATUS_DOING_RESTORATION, 'isMemoryExceeded' => $isMemoryExceeded, 'isRestoreDb' => 1, 'restoreNextPart' => self::RESTORE_PART_RENAME_TABLES ]; if ($isMemoryExceeded || $this->kernel->isTimeExceed($this->meta->maxProcessingTime, $itemTimerStart)) { $dbHandle->commit(); return [ 'success' => false, 'data' => sprintf("Removing Tables: %d/%d", $countRemoved, $totalOldTables), 'partialData' => $setPartialData, 'isMemoryExceeded' => $isMemoryExceeded ]; } if (!array_key_exists($row[0], $tableCleanup)) { if ($dbHandle->exec("DROP TABLE `" . $row[0] . "`")) { $countRemoved++; } } } } } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); $dbHandle->rollback(); $dbHandle->autocommit(true); $dbHandle->close(); return ['sucess' => false, 'data' => 'Renaming Database tables: failed to remove Database Tables', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $dbHandle->commit(); $dbHandle->autocommit(true); $dbHandle->close(); $this->useHandle->wpcore->maybeUpgradeDatabase(); $this->useHandle->wpcore->maybeRemoveStagingStatus(); return [ 'sucess' => false, 'data' => sprintf('Renaming Database tables was successful: Executed %d tables', $countRenamed), 'saveLog' => true, 'saveLogId' => __METHOD__, 'partialData' => $this->partialData, 'isAppendResponse' => true ]; } private function restoreDropins(): array { $this->partialData->restoreNextPart = self::RESTORE_PART_MU_PLUGINS; if ($this->hasRestoreParts->dropins === self::RESTORE_PART_FALSE) { return ['sucess' => false, 'data' => '', 'partialData' => $this->partialData, 'hasFile' => 0]; } $identifier = WithIdentifier::IDENTIFIER_WPCONTENT; $getPath = $this->getPath($identifier); if ($getPath->src === $identifier || $getPath->dst === $identifier) { $this->kernel->log('Failed to restore Drop-in files: Could not get a valid path', __METHOD__); return ['sucess' => false, 'data' => '', 'partialData' => $this->partialData, 'hasFile' => 0, 'isAborted' => true]; } $getPath->src = $this->kernel->normalizePath($this->extractPath . '/' . $getPath->src); $getPath->dst = $this->kernel->normalizePath($getPath->dst); $this->kernel->mkdir($getPath->dst, __LINE__); $dropinsFile = $this->useHandle->extractor->getDropinsFile(); if (!empty($this->isOverwriteParts->dropins)) { foreach ($dropinsFile as $file) { $dstFile = $getPath->dst . '/' . $file; $this->kernel->unlink($dstFile, __LINE__); } } $countFile = 0; foreach ($dropinsFile as $file) { $srcFile = $getPath->src . '/' . $file; if (!file_exists($srcFile)) { continue; } $dstFile = $getPath->dst . '/' . $file; $this->kernel->unlink($dstFile, __LINE__); if (rename($srcFile, $dstFile)) { $countFile++; } } $text = ''; if ($countFile > 0) { $text = sprintf('Restoring Drop-ins: %d files restored', $countFile); $this->kernel->log($text, __METHOD__); } return ['sucess' => false, 'data' => $text, 'partialData' => $this->partialData, 'hasFile' => $countFile]; } private function restoreMuPlugins(): array { $this->partialData->restoreNextPart = self::RESTORE_PART_DONE; $identifier = WithIdentifier::IDENTIFIER_MUPLUGINS; $getPath = $this->getPath($identifier); $isRemoveOptimizer = false; if ($this->hasRestoreParts->muplugins === self::RESTORE_PART_FALSE) { if ($getPath->dst !== $identifier) { $this->kernel->mkdir($getPath->dst, __LINE__); $isRemoveOptimizer = $this->kernel->unlink($getPath->dst . '/wp-staging-optimizer.php', __LINE__); } return ['sucess' => false, 'data' => '', 'partialData' => $this->partialData]; } if ($getPath->src === $identifier || $getPath->dst === $identifier) { return ['sucess' => false, 'data' => 'Failed to restore Mu-Plugins: Could not get a valid path', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $getPath->src = $this->kernel->normalizePath($this->extractPath . '/' . $getPath->src); $getPath->dst = $this->kernel->normalizePath($getPath->dst); if (!empty($this->isOverwriteParts->muplugins) && !$this->useHandle->file->removeDir($getPath->dst)) { return ['sucess' => false, 'data' => 'Failed to restore Mu-Plugins: Unable to overwrite directory', 'saveLogId' => __METHOD__, 'isAborted' => true]; } if (($countFile = $this->useHandle->file->moveDir($getPath->src, $getPath->dst)) === false) { $countFile = 0; } if ($isRemoveOptimizer) { $this->kernel->log('Counting wp-staging-optimizer.php as a restored Drop-in file. The file will then be installed by the wp-staging plugin', __METHOD__); $countFile += 1; } $text = ''; if ($countFile > 0) { $text = sprintf('Restoring Mu-Plugins: %d files restored', $countFile); $this->kernel->log($text, __METHOD__); } return ['sucess' => false, 'data' => $text, 'partialData' => $this->partialData]; } public function restoreBackup(): array { if (empty($this->meta->dataPost['total-files'])) { return ['success' => false, 'data' => 'Invalid request. Total files not available', 'saveLogId' => __METHOD__, 'isAborted' => true]; } if (empty($this->meta->dataPost['restore-parts']) || !filter_var($this->meta->dataPost['restore-parts'], FILTER_DEFAULT, FILTER_REQUIRE_ARRAY)) { return ['success' => false, 'data' => 'Invalid request. Restore parts not available', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $restorePartData = $this->meta->dataPost['restore-parts']; if (!array_intersect_key($restorePartData, ['plugins' => 1, 'muplugins' => 1, 'themes' => 1, 'uploads' => 1, 'wpcontent' => 1, 'database' => 1, 'lang' => 1, 'dropins' => 1])) { return ['success' => false, 'data' => 'Invalid request. Invalid Restore parts data', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $restorePartData = array_map(function ($data) { return (int)$data; }, $restorePartData); $this->hasRestoreParts = (object)$restorePartData; $overwritePartData = $this->meta->dataPost['overwrite-parts']; if (!array_intersect_key($overwritePartData, ['plugins' => 1, 'muplugins' => 1, 'themes' => 1, 'uploads' => 1, 'wpcontent' => 1, 'database' => 1, 'lang' => 1, 'dropins' => 1])) { return ['success' => false, 'data' => 'Invalid request. Invalid Overwrite parts data', 'saveLogId' => __METHOD__, 'isAborted' => true]; } $overwritePartData = array_map(function ($data) { return (int)$data; }, $overwritePartData); $this->isOverwriteParts = (object)$overwritePartData; if ($this->useHandle->extractor->hasCancelRequest()) { $this->useHandle->wpcore->enableMaintenance(false); return ['success' => false, 'data' => 'The backup restoration was cancelled', 'saveLogId' => __METHOD__, 'isCancelled' => true]; } clearstatcache(); $this->partialData = $this->useHandle->extractor->getPartialDataFromAjaxRequest(); if (!isset($this->partialData->restoreNextPart)) { $this->partialData->restoreNextPart = self::NO_RESTORATION_PROCESS_YET; } if ($this->partialData->restoreNextPart === self::NO_RESTORATION_PROCESS_YET) { $extractRestorePartData = array_filter($restorePartData); if (!empty($this->meta->dataPost['restore-parts-listed']) && (int)$this->meta->dataPost['restore-parts-listed'] === count($extractRestorePartData)) { $extractRestorePartData = null; } $extract = $this->useHandle->extractor->extractBackup(true, $extractRestorePartData); if ($extract['success'] === false) { return $extract; } $this->partialData->status = self::STATUS_DOING_RESTORATION; $this->partialData->restoreNextPart = self::RESTORE_PART_UPLOADS; return ['sucess' => false, 'data' => $extract['data'], 'saveLog' => !empty($extract['saveLog']), 'saveLogId' => !empty($extract['saveLogId']) ? $extract['saveLogId'] : null, 'partialData' => $this->partialData]; } $this->kernel->maxExecutionTime($this->kernel::MAX_TIMEOUT_RESTORE); $this->useHandle->wpcore->enableMaintenance(true); $partResponse = []; switch ($this->partialData->restoreNextPart) { case self::RESTORE_PART_UPLOADS: $partResponse = $this->restoreUploads(); break; case self::RESTORE_PART_PLUGINS: $partResponse = $this->restorePlugins(); break; case self::RESTORE_PART_THEMES: $partResponse = $this->restoreThemes(); break; case self::RESTORE_PART_LANG: $partResponse = $this->restoreLang(); break; case self::RESTORE_PART_WPCONTENT: $partResponse = $this->restoreWpContent(); break; case self::RESTORE_PART_DELAY_DATABASE: $this->partialData->restoreNextPart = self::RESTORE_PART_DATABASE; $text = !empty($this->hasRestoreParts->database) ? 'Restoring Database in progress' : ''; $partResponse = ['sucess' => false, 'data' => $text, 'partialData' => $this->partialData]; break; case self::RESTORE_PART_DATABASE: $partResponse = $this->restoreDatabase(); break; case self::RESTORE_PART_RENAME_TABLES: $partResponse = $this->renameTables(); break; case self::RESTORE_PART_DROPINS: $partResponse = $this->restoreDropins(); if ($partResponse['hasFile']) { $this->useHandle->wpcore->flushObjectCache(); } break; case self::RESTORE_PART_MU_PLUGINS: $partResponse = $this->restoreMuPlugins(); break; default: $partResponse = []; } if (!empty($partResponse)) { if (!empty($partResponse['isCancelled']) || !empty($partResponse['isAborted'])) { $this->useHandle->wpcore->enableMaintenance(false); if (!empty($partResponse['data'])) { $this->kernel->log($partResponse['data'], !empty($partResponse['saveLogId']) ? $partResponse['saveLogId'] : null); } } return $partResponse; } $this->useHandle->wpcore->enableMaintenance(false); $this->useHandle->file->removeDir($this->extractPath); $this->useHandle->wpcore->saveConfig(); return ['sucess' => true, 'data' => 'Restoring backup was successful', 'saveLog' => true, 'saveLogId' => __METHOD__, 'isCompleted' => true]; } }
    final class SearchReplace { private $kernel; private $meta; private $search = []; private $replace = []; private $exclude = []; private $caseSensitive = true; private $currentSearch; private $currentReplace; private $isWpBakeryActive = false; private $smallerReplacement = PHP_INT_MAX; public function __construct(\WPStagingRestorer $kernel) { $this->kernel = $kernel; $this->meta = $this->kernel->getMeta(); } public function getSmallerSearchLength(): int { if ($this->smallerReplacement < PHP_INT_MAX) { return $this->smallerReplacement; } foreach ($this->search as $search) { if (strlen($search) < $this->smallerReplacement) { $this->smallerReplacement = strlen($search); } } return $this->smallerReplacement; } public function replace($data) { if (!$this->search || !$this->replace) { return $data; } $totalSearch = count($this->search); $totalReplace = count($this->replace); if ($totalSearch !== $totalReplace) { $this->kernel->log(sprintf('Can not search and replace. There are %d items to search and %d items to replace', $totalSearch, $totalReplace), __METHOD__); return $data; } for ($i = 0; $i < $totalSearch; $i++) { $this->currentSearch = (string)$this->search[$i]; $this->currentReplace = (string)$this->replace[$i]; $data = $this->walker($data); } return $data; } public function replaceExtended($data): string { if ($this->isWpBakeryActive) { $data = preg_replace_callback('/\[vc_raw_html\](.+?)\[\/vc_raw_html\]/S', [$this, 'replaceWpBakeryValues'], $data); } return $this->replace($data); } public function replaceWpBakeryValues($matched): string { $data = base64_decode($matched[1]); $data = $this->replace($data); return '[vc_raw_html]' . base64_encode($data) . '[/vc_raw_html]'; } public function setSearch(array $search): self { $this->search = $search; return $this; } public function setReplace(array $replace): self { $this->replace = $replace; return $this; } public function setCaseSensitive($caseSensitive): self { $this->caseSensitive = $caseSensitive; return $this; } public function setExclude(array $exclude): self { $this->exclude = $exclude; return $this; } public function setWpBakeryActive($isActive = true): self { $this->isWpBakeryActive = $isActive; return $this; } private function walker($data) { switch (gettype($data)) { case "string": return $this->replaceString($data); case "array": return $this->replaceArray($data); case "object": return $this->replaceObject($data); } return $data; } private function replaceString(string $data) { if (!$this->kernel->isSerialized($data)) { return $this->strReplace($data); } if (strpos($data, 'O:3:"PDO":0:') !== false) { return $data; } if (strpos($data, 'O:8:"DateTime":0:') !== false) { return $data; } if (strpos($data, 'O:') !== false && preg_match_all('@O:\d+:"([^"]+)"@', $data, $match) && !empty($match) && !empty($match[1])) { foreach ($match[1] as $value) { if ($value !== 'stdClass') { return $data; } } unset($match); } $unserialized = false; try { $this->kernel->suppressError(true); $unserialized = unserialize($data); $this->kernel->suppressError(false); } catch (\Throwable $e) { $this->kernel->log( [ 'method' => __METHOD__, 'data' => $data, 'error' => $e ], __METHOD__ ); } if ($unserialized !== false) { return serialize($this->walker($unserialized)); } return $data; } private function replaceArray(array $data): array { foreach ($data as $key => $value) { $data[$key] = $this->walker($value); } return $data; } private function replaceObject($data) { $props = get_object_vars($data); if (!empty($props['__PHP_Incomplete_Class_Name'])) { return $data; } foreach ($props as $key => $value) { if ($key === '' || (isset($key[0]) && ord($key[0]) === 0)) { continue; } $data->{$key} = $this->walker($value); } return $data; } private function strReplace($data = ''): string { $regexExclude = ''; foreach ($this->exclude as $excludeString) { $regexExclude .= $excludeString . '(*SKIP)(*FAIL)|'; } $pattern = '@' . $regexExclude . preg_quote($this->currentSearch, '@') . '@'; if (!$this->caseSensitive) { $pattern .= 'i'; } return preg_replace($pattern, $this->currentReplace, $data); } }
    final class View
    {
        private $kernel;
        private $meta;
        private $useHandle;
        public function __construct(\WPStagingRestorer $kernel)
        {
            $this->kernel    = $kernel;
            $this->meta      = $this->kernel->getMeta();
            $this->useHandle = $this->kernel->getHandle(__CLASS__);
        }
        public function getWpVersion(): array
        {
            $wpver = [];
            $list  = $this->useHandle->extractor->getBackupFiles();
            foreach ($list as $index => $data) {
                $wpver[$data['wpVersion']] = $data['wpVersion'];
            }
            return $wpver;
        }
        public function render(string $page): bool
        {
            if (is_object($this->useHandle) && !isset($this->useHandle->view)) {
                $useHandle         = (array)$this->useHandle;
                $useHandle['view'] = $this;
                $this->useHandle   = (object)$useHandle;
            }
            $methodName = lcfirst(str_replace(' ', '', ucwords(implode(' ', explode('-', $page)))));
            if (!method_exists($this, $methodName)) {
                printf('Item %s is not available', $this->kernel->escapeString($page));
                return false;
            }
            call_user_func([$this, $methodName]);
            return true;
        }
        private function LZWDecompress($binary): string
        {
            $dictionaryCount = 256;
            $bits            = 8;
            $codes           = [];
            $rest            = 0;
            $restLength      = 0;
            for ($i = 0; $i < strlen($binary); $i++) {
                $rest = ($rest << 8) + ord($binary[$i]);
                $restLength += 8;
                if ($restLength >= $bits) {
                    $restLength -= $bits;
                    $codes[] = $rest >> $restLength;
                    $rest &= (1 << $restLength) - 1;
                    $dictionaryCount++;
                    if ($dictionaryCount >> $bits) {
                        $bits++;
                    }
                }
            }
            $dictionary = range("\0", "\xFF");
            $output     = '';
            $word       = ' ';
            $element    = ' ';
            foreach ($codes as $i => $code) {
                $element = isset($dictionary[$code]) ? $dictionary[$code] : $word . $word[0];
                $output .= $element;
                if ($i) {
                    $dictionary[] = $word . $element[0];
                }
                $word = $element;
            }
            return $output;
        }
        private function escapeTooltip($text): string
        {
            return $this->kernel->escapeString($text, ['&#xa;']);
        }
        private function printAppFile()
        {
            echo $this->kernel->escapeString($this->meta->appFile);
        }
        private function printVersion()
        {
            echo $this->kernel->escapeString($this->meta->version);
        }
        private function printLicenseOwner()
        {
            $data = $this->useHandle->activate->getData();
            if (is_object($data) && !empty($data->name) && !empty($data->email)) {
                echo $this->kernel->escapeString($data->name . ' <' . $data->email . '>');
            }
        }
        private function printLicenseType()
        {
            $data = $this->useHandle->activate->getData();
            if (is_object($data) && !empty($data->type)) {
                echo '<a href="https://wp-staging.com" rel="noopener" target="new">' . $this->kernel->escapeString($data->type) . '</a>';
            }
        }
        private function printAssets($name, $isReturn = false)
        {
            $output = $this->meta->appFile . "?wpstg-restorer-file=print-" . $this->kernel->escapeString($name) . "&_=" . $this->kernel->escapeString($this->meta->buildId);
            if ($isReturn) {
                return $output;
            }
            echo $output;
        }
        private function printProcessLoader()
        {
            echo '<img id="wpstg-restorer-spinner" src="' . $this->printAssets('loader', true) . '">';
        }
        private function partSelection($metaData): array
        {
            $sortbyOption     = [];
            $sortbyOption[''] = 'All';
            if ($metaData->isExportingPlugins) {
                $sortbyOption['plugins'] = 'Plugins';
            }
            if ($metaData->isExportingMuPlugins) {
                $sortbyOption['muplugins'] = 'Mu-Plugins';
            }
            if ($metaData->isExportingThemes) {
                $sortbyOption['themes'] = 'Themes';
            }
            if ($metaData->isExportingUploads) {
                $sortbyOption['uploads'] = 'Media Library';
            }
            if ($metaData->isExportingDatabase) {
                $sortbyOption['wpstgsql'] = 'Database';
            }
            if ($metaData->isExportingLang) {
                $sortbyOption['lang'] = 'Languages';
            }
            if ($metaData->isExportingDropins) {
                $sortbyOption['dropins'] = 'Drop-in File';
            }
            if ($metaData->isExportingOtherWpContentFiles) {
                $sortbyOption['wpcontent'] = 'Other Files in wp-content';
            }
            if (count($sortbyOption) - 1 < 2) {
                $sortbyOption = [];
            }
            return $sortbyOption;
        }
        public function backupPaging(string $indexFile, string $databaseFile, &$pagingData = '')
        {
            if (!file_exists($indexFile)) {
                return false;
            }
            $pagingData = [
                'totalIndex'  => 0,
                'totalPage'   => 0,
                'indexPage'   => 0,
                'indexFilter' => '',
                'indexSortby' => ''
            ];
            if (!empty($this->meta->dataRequest['paging-data']) && filter_var($this->meta->dataRequest['paging-data'], FILTER_DEFAULT, FILTER_REQUIRE_ARRAY)) {
                $pagingData = array_merge($pagingData, $this->meta->dataRequest['paging-data']);
                foreach ($pagingData as $key => $value) {
                    if (in_array($key, ['indexFilter', 'indexSortby'])) {
                        $pagingData[$key] = (string)$value;
                        continue;
                    }
                    $pagingData[$key] = abs((int)$value);
                }
            }
            $pagingData = (object)$pagingData;
            try {
                $objectFileIndex = $this->useHandle->file->fileObject($indexFile, 'rb');
                if (version_compare(PHP_VERSION, '8.0.19', '<')) {
                    $objectFileIndex->fgets();
                }
                $perPage = 50;
                if (empty($pagingData->totalPage)) {
                    $totalIndex = $objectFileIndex->totalLines(true);
                    $pagingData->totalPage  = abs((int)($totalIndex / $perPage));
                    $pagingData->totalIndex = $totalIndex;
                }
                if ($pagingData->totalPage > $pagingData->totalIndex) {
                    $pagingData->totalPage  = 0;
                    $pagingData->totalIndex = 0;
                    $pagingData->indexPage  = 0;
                }
                $offset     = $pagingData->indexPage * $perPage;
                $offsetNext = $offset > 0 ? $offset + $perPage : $perPage;
                if ($offsetNext > $pagingData->totalIndex) {
                    $offsetNext = $pagingData->totalIndex;
                }
                $countIndex = 0;
                $isFilter = !empty($pagingData->indexFilter);
                $isSortby = !empty($pagingData->indexSortby);
                if ($isFilter || $isSortby) {
                    $pagingData->indexFilter = htmlspecialchars_decode(urldecode($pagingData->indexFilter));
                    $pagingData->indexSortby = htmlspecialchars_decode(urldecode($pagingData->indexSortby));
                    $countEnd   = $offset > 0 ? $offset + $perPage : $perPage;
                    $countStart = $countEnd - $perPage;
                }
                while ($objectFileIndex->valid()) {
                    $line     = trim($objectFileIndex->fgetsUseParent());
                    $indexKey = $objectFileIndex->key();                    if ($this->useHandle->file->isLineBreak($line)) {
                        continue;
                    }
                    if (!($items = $this->useHandle->extractor->parseFileIndexItem($line))) {
                        continue;
                    }
                    $itemPath = $items[1];
                    $items[0] = $indexKey;
                    $items[1] = WithIdentifier::replaceIdentifierPath($itemPath);
                    $items[3] = (int)$items[3];
                    $items[4] = false;
                    if ($itemPath === $databaseFile) {
                        $items[4] = true;
                    }
                    if (!$isFilter && !$isSortby && ($countIndex >= $offset && $countIndex < $offsetNext)) {
                        yield $items;
                    }
                    if ($isFilter || $isSortby) {
                        if ($isSortby) {
                            $sortbyIdentifier = $this->useHandle->extractor->getIdentifierByPartName($pagingData->indexSortby);
                            if ($pagingData->indexSortby !== 'wpstgsql' && $pagingData->indexSortby !== 'dropins' && !$this->kernel->isStringBeginsWith($itemPath, $sortbyIdentifier)) {
                                continue;
                            }
                            if ($pagingData->indexSortby === 'wpstgsql' && $itemPath !== $databaseFile) {
                                continue;
                            }
                            if ($pagingData->indexSortby === 'dropins' && !in_array(substr($itemPath, strlen(WithIdentifier::IDENTIFIER_WPCONTENT)), $this->useHandle->extractor->getDropinsFile())) {
                                continue;
                            }
                        }
                        if ($isFilter && stripos($items[1], $pagingData->indexFilter) === false) {
                            continue;
                        }
                        if ($countIndex >= $countStart && $countIndex < $countEnd) {
                            yield $items;
                        }
                    }
                    $countIndex++;
                }
                $objectFileIndex = null;
                $pagingData->totalPage  = abs((int)($countIndex / $perPage));
                $pagingData->totalIndex = $countIndex;
            } catch (\Throwable $e) {
                $this->kernel->log($e, __METHOD__);
            }
            return false;
        }
        private function printCss() {echo $this->LZWDecompress("#;gC8e!id0CLDgBi2\r\0`0qAg4\$iD: 8fn')m:A8l2B\nl8b0fsn2IFz`,Z,ApSN3nCa<jF^nӜe7bWQi3E9+#dGP*A\\6f 6@30siHAÅp8LCqf4C>cS.r`Y\"~~`6&4zx:sƅ*9:CK,;H (\n-j\"Mh5SlZ3-䊍@2ʞc\n9#pps0EI2n6cHҬѴ<4Cs]6-j%C\rD6աR>DS8ȴ3DÍd3HE#{% û\\MFL,9P|vHD)t+\$2-\\*tPأ6P:!!E#G,4sJO\r.NlN\"@7'*6.(Zf?//HTU!d=d3׃eC:*\$K7%=YLP44`m3	Ƅ|ɶ.̳lBc~֪Tc}v`%ILC褍քSTőt`5\"(V5ȧChέ2x\nHD\nC}x<Z)#*,r@Fv4ǡ\02.H#XVJT+JZ,)+MAɛ9\0@C aܱ\r㔒~9\r(.Wu'#9vr4MZIRbKxx`05[%7[]7/B&ҡ+/lt YIKE?riW\r(4 ߓ]5%Tch-\rގ#jT蝸2wa5#< YP4@t_\nhWA	S*e.mI\nt bAFg3\"۬@lOg.)&BFxbapm0xra!ь:4pqb\$#=6GǘN䓅K%0f\r9>i#9}-I\r)-Z\$ jI\r(9TQ_&\\\nSo)Ñ0 ް-YoN9KipUA5VHrVvhER*&]Ro:'Pw92dd<?@b,t9`_/иObM	LItm0)Q\$`Dp1@&Hm9b1\$XZ)woH\rMDBpd\0TF3fpM0e\\s0l(P@YYA\r ʚp@uIXXmaL+5XC\$E;眡\$&6Z:\$+jF#`9Xv*+/	%JTKMXO4ҵۺDH[!@dvgiIeV\r絛%i`MMbmmDuNR`^A%%*0}x\$_6(uaN43h\\Ӟ,\r\$FY'd\$\rĩ\\)	Ъ(OI\rI'!r훻C+&`̡'{(82\ncljq%hܖޣvrЂ0 0eŎ3rP;)6Wʹ^1LȳٚϥO[\$Jn,!zM9VCߩTąYX16~צAXl6skAcJЬM3D>tf6~dnsA48PNt=a Dii8mށ\0M'fwr~\$qB|¶@HLc6\"{0\$UjU\n*p!\$FAj}[{N\nлv6	\"ނ~85۹Y]˭sݯ;lu\"K7|\0s_7:!м^&Tޜ~ոJ\"\"C\")5\r?'Zq\rG\"\rܐrs8M+塷O8L>E\$d\r1xVsW瞓w^ìH4g>ߥ\0OxRp\r'`Lp^O\"/()Li\"Fhn~n\0Ex.䞐 *|̽e<Nl#d2 |\r\0dPJ.1\r.f]	&Z)m\$˭v\0dHW-Rr\rh̀:t'|qmeȪ\"zo` `llB諀nWz@NxhQz: `A\nqq!)\0hIkk>g4 GjӠp`prV'BF8]~,<\0\rO.t/*9QQ)1˱11GDB'cZt2&8.JI{(2V\"B.U\r8lo# V1Q)Ia1,(L#q&sqDI#Q%q%Bﱋ\"LRG1LU%Rst\"0sq{#kPIX\$pwug*y8ȴr\r;R!1Ѳng	ūf;//mb;5':/\0(V(,ϴEIirk.kS<kplm'&m\rnf1HFoo6pGq\$tr\$Gr*N`tE'qJm6vHR42	Ɏ\rH9M9't'PU9Y\rK41A.':[\0/ iκnTgm&%u23<T@Pҕ1Tu0@T\$SB'405+n֍>Z4)<3cfHJYE\0f`mECF	GFdDTN F\"ET}E,sGj@JrOIHԞt4Lpl\0K4]KtEF\"ЦH3LzkMnG2JqWI	2\0cKFM=P\0Jd6>Cnzǰޒ\0A =nOcL+WH줠rdҁ&Nu4bHٰ\$|j#T~Y(-RI; 5LtoI]OQcgbñlӇo*qy#0lA}0]r&섫5C?BISfA=Z#>3!	?TsPUM?QZ3R3]2pO\",\"r*}jM\"6&U!\$	=yB5`\$bZ+H!#,@cXo_6y0[B'_^RZbT[Jt/KM._e6Krefvj\rvo`VTSTjg=?O;sUuT4Qc3!:Vv[\r#dKjTP\rl?O?(@0XJAF,ޭ1Vwf/wf\rot-f*&q8c6^9b+Fp[sW\0ab	 >ͲLʘm,Y-qXYv˷hљs%svy%/%W@zw\$f8Gs͵wEmXUj+x'\0hN/,v2~a\0\nU|wMAvxWxϼU[]dlsw-g^H@tG(G|yw}}\":xMYNm7:wZҲ\\)nBv օ'v?=V&tw8Gxv+SYL},}pғ@ 3.,N2@OO]@.\0Cn8Nw(?l{tّ·9\$/\$.h=.F3B(o1͑nD3F!n!P@ZsanpS1z8z+	#0q72ZEz[t+_pZ,H.t\" rج8X}P6u0g1W]jȨHWqk@MIU{d(GV^1\nvz-d8=PKЖWvl0XL7?qc\rqz|.\0+ZM:. `|짜z{ى8ۦF~*wLA&5r:4:əQD\r:\nwYxg#y;vY~u+YAwZytqD:dњtWKFAhˉmL{iv 𱰚٨E]UfX`rݟxf;װ{k94:;jfٹYw:ws[jx\0E[r*e)3l\0\\-\r,bʕC~b-ڗQ+G\0V\0ͼyC\0#/	.&yeeٿϖ\r\\#ׅ-k1qd/[w`\$f<[F՜#wќZw)۵|YJw@A\0;Z<㳌)\r|\"c,]Y`v߹pM݁ 8abUO=нͯрV[Ym}|*+/ⴤ{.p_ɼv}-]6[kUgҗ1=5ii9pKr#/%AُGqVd 7\\){@wv\\Ǉoݳ!j\r\nZQaYggUM]ѥ%0.Sp;s#{Ǩ]qn\rŝ>#Gr`rwݽ e;ezqjAa!uvx~YbJ\r^V\rϻ~g~;ϞҘ#;8nd{>ϿAۚT>徥vܩޫ<x敔n9<A8Y_c~Oе%~\\gd۸_M`^>ȝADp}XS;HgA/^KOM+a\r{2_4n=x\r9I;8%ni\$nC\0=)ނ}s?!nzAO̿˸b 0?@j:ZkA<ٱuyKU\0X7i_~r4*@u+\\ >Ҷ]!3Av~)v	f@+Q>H!L11ƕZ#^\0&!Rh.J+vpO\r`3M+87!<`%U?pX'S;_w\nb\$Y8=].	.m>>װ,3cYF31hNX`.X?P'\$u\\z񠺳+/\"x@ROԀ&~L\0HnB]/	?}B4q2,3;OUF=2|(HohSuuICb#B\rfN\nвiN`)[!0(MC6@M\0007Q\$BO~dJa3DՐFbk\na`\\\nȞ?zADVeYý/|1]E	87&sfݔ3l87 U	\$aoCnƄ'?\r<pچض\rʇ.b\0Þ.aShx\\!RB'n#jŘݐ\r\\b0(%VJ[Oc[JM5s{Dfm1MRg\n\"/{b*8Й4oI}aY{tHe9l+f̅!\"%	!bwxǆ9AՏ͢^Ң*͊DsxrGQvrݣ\$M~jc!Lp<ǦT1.%| ǈg\\QH8ޱ<cȩ\0QH)Y7d-ZRК &!\\dt\\pH^iZ;i&1h`ؑ%'G -	\$_6-.K\\\rJY'`C5\0M@Ek#[\09QKY-8ȰRsF04\nAa[(J\$rnxG,L%	ET:L{x\"J>!;̴*YbX4I][MwB\r)8JL	d@(󳀠F,U#-A[)b1e.R˶\\a8#_ƿ3&/:| \$\rG\rd\r!2Fg.H_%+ KI>p&0˾Sm2)~IKc(̭-ęt oIS8X \0:qZQb×)^o㖒Kvդ#xHYϣ!XƔ7RF\0G5@GShQ\$F\nLu77HIKtfɻI|\$X33\\ыA8	7M윤R1sB\"c55b[\"\n\0g.INvQRԾ%hP\0vdC/yLnuӻ}/iuOvTd ɠ\\r{*bׅ)\nV1VF\"%qJ!vXBeX19uKJVGVH,6Zff4]҇@+	Ppؿj3ʠ/L؞'Jx2si\n\\Mޑ2\nj8ʎ&2Aq@jbVT3C|ܒLRl	C	J6(֠\"/b3e=\$Ph	T\$6bT,\0P<B6N\0ZpHaXPkH	E\r\0dԠۈ]oFQ1`9FѸ6b'Ht1)AcqPYF8)B2z<Z	ePf{pYKqLKn]CPxa9U|	DVl\"hz\n-4'|>/+.eH!-2-\0d.mLl54SIѨSчid\0BJ/(]Uz\$h#neif`aA]>𡡘4Ж\$a͎\0Lah3Q]N?ޗQ\0őPn 4.ZbdYh-0U9T(ZUж>)(ꈜ V(If'V9E|5OFa8Q8;n!榨j'UI*PJ2u=3GRơ(At`}LSh@:oOi`o!BM[/+ZXJ*[&UlOnN4:C\"DL)n*Nzu2Nk4*v`omwS!:ׂ5m%fkڈ%\$#vEBx#\nL7=WւmAzFj٫k]3*b.QvÞ`Wv.7a(XR1P\\Č=\0>");}
        private function printJs() {echo $this->LZWDecompress("\no7\nà\":̢ib#yeGCo3\r!e	E\"Id.2?e\n=\r3n:ΐ``)@hD*i3\n5V\r:FeNW3Dj(L#Ȁe8M8s ;M+FTjR;Ý#peGZ}Md2FS C#9jM0D>## 7G=' 9P\r)<,0\"㙲13~fca㬈Kдm+p0b@J=#hD#6!C5X9I;䟲Mpz=Hz7P2C8XGap0bx7=zL!`Qt0PT<(\n|o\"o4\nMcrn'IJ`\rʨӣt'rX2c8 nB>`:lL<TL	­,󻴑'a3Kr!`tDW42֕)R:FsA`)Dl7L;2{8CƋ	>|z PA\0-B:HH&78w!Jcs\"W\0Te32# ]̘ፃ%j'Pas.F48`Sce@XDIě'\rh9ѳ~7cL&&@;LA\\E2x*	`Hc8hu2/\$4]`pt+]KExw 31{\rv7#̘1Pΰ03wh	GO#l9jRR:&!yàZS`Z\0PM0ɍAj&.\$aaOe^|CS!8=/[sLK<ad27:#\nQvL]tOPA/nd\$ޞqj8CR\riɲ&xKPI*\0e0'rӄq!]DȬ9~CkJ),cVGC'_}LuCa&Lo	VDM;ԸnB]!tD;!Ɓɱu]Dc}GADd@SQFőY\"YcF Le3x&UPX*8q#,s\$삃h\nB\n)>2X%r;Ih&&,Ǚ1f4欹3d游0FVV9vaM1\rNp]r\\9/':/ȳ=QJtqCNGΐTN\$\rFp'S\0ȥf\\lO-唔=8=*܊-go\r0&LsgÝGFHS;+ERdgLWP!:'χ0xAsr3nBKn_C4ZwA_DI1&*Lmp&9da@VAqķα0~Y\\;pZ!T%6,RHGUPYeMJ&\r\090txiZc&o5PFzA@{(,޴S\0` \"s<&˪R\rjLs݁@/& Rc+_'a.%̺r^tlPI:+kmu=j(!-A::BTjuK]u	W'DXrE0TB\"+.'e]spu]Vi@3U2+Ș3:)\r,PX{PD\\<bŋ1H9n;r{ hn4-rUG\"l.ĵX}\rV,݌-:؜kwVN@`ʉ/b#fժ:zBmLZCIg- ]`wj;nOtd\"M\nv睸(VAalB&#]:k\n>CMXe3PDtmECڪj)Pqf\$DRZ)2P\rE:(#Tr۲~Kx,D*{l7elP\\w!@p֌oZ&TնTo`6.r+VnOAvۉ\\t=h'WnT*9X<x#L4\$;DsҵJz\"ԸuiSXBiک60	HW9qҴsbY+-)Tʵ?2cWޱv9fooXD0xӺAغ7Q#گ^BΓ(.J`CD%찫\n`vʶfj)p?Rp60:RVmzKӂJJ	0`@K\0@@D(L@/B+D>P[@3ɔ@L	pD||\"GԚLE(IX6ؤcpc;P\0dP\r̖bGfp\0D<`\0PtRM{uR@S&@Hm`D<fh2DPzRx(eBe`>uOpKXGPrZ,C\rgh+kQΆT	B\n\0g9\"b lѲe#&50:mD` `oZ \0IpZ:Xx\0~Qu{#D6>jLCM6m\\&\nf%[:bj(q\n@hC<eCacl@;@`v -<H!`\n\n}+ȧ/HsEjE\0}GL0S\0\rL}|jL`D6#h\$%3\$\"`bP-4SI*d.m`y 6 %`ZI\r\\ >Ȅ*Z¤\rA91C7%9cj!Dp Zj;8;#\"r!qxm`l W@DBf^1=D@T ~O@SA0-DNױLT?1;D@>1:TL Q+3@U?&A03Ӹ,=HE`KSEkEtCDtcFtjEHS?\nk`VMTtmdЄ(@Q@Ƅ\\L)MNQAT4h@U\nBOqAWB@wCp\rM}\nRNd5AN/HqVLFuOUMOtAp` \nucugQ4V=\n WX5QXRM{\n\n\nUSOOL\n`\n	T5`	 VeS5PE:V6@\"I\n*+\0`!ZI7\"PAYA4x!FbZ	Ds^šRu#btaMVC\nuYld.hm+eu\\7Q_UV71\0s_\"^aaT7Eq kb1@IA Ofom\$Ej+3bjeJbԎFY`m\0 vmAVӽ\"@d<Aqp&mh ^T{;<3<=3QsDE>3>)e/#1n:\0%L>t ZjfHp\0.`dc&&ywl,xy2_y| `z@rY{d`LZ\0jgF\nW\\@X	z	@Pif+d*3k6h&'+9fJb6`1fVPf.B^<&P\"`c<NkAVO\\fɄPJK=hÅXX(j72?Bri6l{X~+HbǘL>BA+#RozbxDL8 0q-Y@.z\rW\0rZ\$FnkP\$@& rN+~#PDZ2Zm'@cGQ0)@f\\t-zelK~_[` da\"l>jᭊ\\1<͕ߕc\0	1gظ*n޹xmDΌ@˔Ӕ\0d׹w>DI {.\$a~\\\0f׮^i`y-x՝Yj\0Р{FEbc `\$x!m\0¶!\rQ~z	Z!bNׁzPjhŌL)s\$Qfb`brcjQ(d\$ҏ37R>a&-E\0|99^ha`ް`R\rZ#`F@h>C\"L3*`Z3>Ծ\"S	4\r8\\~	r@k\0Ǟ<Cxv9X\"f\$K2z\nnef&b/Xoc*0f,c\n;Ǐ^HX.\$d\n`RK3) P>¶#ۗFDwłt<\npDخ`@ۀz\nc!B6M3xr7 c\0ݴ@xLq[bffj!g{)-\"r`mz~bq{<ow7yVg3ţČna=Ȉ\0'e2N\rLP<͘sm76?\r#|c#a<Qp|rK6ܶ͜' μȵ~& ,|͢ph(܇\n	o\0ޗĦoBB`D[Ĥ[w^^سlOQ։]lDmK6P^6#f6wG'Òκ1hu+oHteU.u\0Ge݂ߡ@XTB\0*}\074@pŲHZ>\r~\$`Y+cn\nP];63F\r\nÖ@്,cx\"	5Ӡ\r:m`1ZGf@<,d*\\+يݡ`ޛ/)|b3|B̚|KPicJr\0bh`	\nT0?,r\"DG1NL|->+e_}9'.a`u4յzGYDc[\"'\nwz>&%O}a_r.\n t;,\0QY_o\"'-*Qپ_6=Q}\\Ƥ?PmjjLh\"&~0dߐOM\r0fuQv]|}@i12K\\M\0@DGF	g>S@LzBnUD\\M.\0000.`d/T?@E2D4c<b6w3AZ㠈%O|l*Wu~PZ`'(\n&mvs@'i\0ZdU#֡h8\\#|Z:oy1p@v-@<{.@N`) l@zxU=͠'m\0N\00[pAGSxS \"G|bh`B=ظ]HC/paph*u0s|FCt6KzZ.q@D\rĳbM7д_ ?\rCBHTM]r<-5p0S#p\n&XCq}A\0S.r=6H\"@B&;oÉP0֬IMKb+v؂=dPlL`ZշYDB&FBGvIX6N:T?aDhWsvFQ2DoZR#/Q;aQHnq;+Kb@Zާ}CPQJ_ 1DFt*cƮ(1H/[gb><r#!{=MJyHvV؎\0`cclpyDKd#cr7(\r@Hiڜ!lۊ\\!i~ \"ӋEuc{J9\n\0쀨+\"JLY\$j+d:xCW}Ilh}tg?I\$AlL~@4_<c*&|fs^3\r'U!\0dZHO	\n.GktoahM)G{b\";B{qP=.w{!**\r\"4p͟,D9)Tdbjy	|IBw0|&29EY7\"ri|d-IO@&v<BOK=}-5LeéDbr K)i4WE^ԣ{/|_`\$JX)f+^ Qx/ 7C gp@A@1@OF\$ `i\"KhDa7	\r{v8dw/\"=/LO\nW/Lv۫\$]\">Y\"k`p#Sg5#{\\r9,>IR~`.`RLd'GxΗimAmnT@tQZŐZPB#F`: |%Fa^\"D*@aJ/et')081٠̾0*li`p4I|v8D@Ԑld07	0,Suxb4AAxUSݔ.\")m\",>N\rh5ч,a~ k8k9\0[\\,H\0WB`:ySe>/#ӕj2&s|Ti΄Gɝ\0:CxZ&d-yso'kΒ{eK<ғ^}}=bO`VWl-`[ZA*W[Khkѩhs8:ΐiDZG'LbR8'cMVCSк*2rPxG%hDI52x`u*o'*@Ɍ9ȹ%0\"{Et(S9.vdPrQRAlP1	5<U4Ӈ@DbVd_#>	cqTXհ1C[|ad@ٝ) d :pG]+EQ	+´nѢޖt-`F@?.`	bl=\0(\n*axcn学!ѫT6M0K|A/m;ajS&TQ9RoY\0 YMf&8N]dKyOZf.S-`ǩ=_娨rIݑlH04v@gIB>~~0v-`R6\0:M6'	Sm*`lӯ%TW{i*&;:`(g1'VL\r[E.A`i)\0E	⼇\\ч*P89AE\0[VozHw@\0(\"?(ZޢV]B8h5lò)l'&eY3Yi	d(2[t\nXՐk@(P-7X%ĭYlc[q\r~1`\0\"lEoEq m:kC^։A\"(	d'4+<m[տLkpU'l+de5C0da-gk>u\nr\0\0 ԃI\0^+[W	 Nc.7<E\n-4jdƵ=Yd?1H\n,,vY[-~kB1\"f.8H`\"HH;@5PTˉ(IsxI>YL4|%tNW`ﺌ!n<ϻU4\0=ۃ|[ܡ%>Q5hH{=\0_d(ABL} \n%I0\0P1=!:0Ѵ(\"A=Tj0\0[hcׇ#eZ0XY2j5mZ95\0jC:/iѼI5&(\$H?R~kcQڞB%N|g&.J^s*j(-4dLQW@n4o`A=wdaCi|%)C弧1dZRR@@+\0\0hntj]Yh9X0`kzyS,<UV!TG2H6Qiw<DK5LjЩ@&96#	 UErj.vf鄙DZ>x#3ǜB f-\n\0ج>FM5	hF8|&	41(lW@tr	WTl]6sL8ԗD3+N\$5Fzx@6mpw2046_Lwky1v瘽g@a))k£|0t\0z&*v۹6:7|\\ccmAk4l\nT]n\niI/X,\\4hn-0\0.!C\0AE:K>q5h0\$k{8JF\\\0tJ/@ܨaW	=gbr!ׂ\\*!avPȶ=jP3\\2H#J{<C9y xJfx8ۄg`jg\rYIk\"8M?{QaS¹vO|*elwH\n,UEJH@(1Ѓk*r\\\ne/Q	+\0004<M,cãYf1peMIbq-xrm;P!\0 k%M/mc!E[Ui/fCG}@gTTSaj0CSBpɲ4]?m-ֳ4훕Z\0,xqWT-\rݧ@Qn*!=Oj1\\ǭ٨pUz?yɬ`t?@+).nBm?\"&qL|lT>^n>El9W]\$8VOjJUOX-}nF+QZ􀦣k,> hzC~dpuڊdP\"`kM,&qzqp\0pΩ.URŒ[і/ZDwP^c:}1fWH8NcʳcyuH9U2li_Nl'a#䷨BC`Έ'Ee\063%i 3IztNi-K}\":	h\0p@yN6Xf|-}U95ֹl.ndX@X/bP+\\:g7aXŕ~Q!4n0PtD<tn\0^!	gcܨR;\"nZͷ]>u*ej1=H\\w\0ҥ\nYEa:։\r1Pb5`n_CcQZ@ !\0L.@طu9\0`p*Jj*slmhQ\" ,=8qfzɬ;.[h¼<.!{[p8gA\"ǈcR=C\$\0yԘĦT?	vF#U'*g˥\"	}վu\r^&4M\$\0&M5V*Ev`1%Skb:l[9̼Bj(؅mQs\"Sn|ծ3W\0ţSaӶ\$/L%0@\"b&Dmm4 ed7-j\08\rʚ/ajfX\$?MGn!aá#bWF^|k[eFog7iYNJӰsrcْx\$Nid`+K@Ø#/b[	(NZmC{/=z6Dɷs(EM!\0hSfYǀThSj2k#VËsG6ݸ%	Svߵ7+2 ?j1)KZ0Ef&hNFl %v^vz-CղLavXS0&V>ړ'qDjKIM~]S`X\r:ޘM`.*X~c6{\\4S\0w	H޷h\r9@Z,):\0`E'amXk\0Fa5`>Ia\$H]2Iz@jj`)o	K\"@\rH\0q?e}uop\" pfFCߙWwٍLex7I/R\rNI8,(tw1c8BÀRe0Kx(~Ȫ;ڱhB,AlykM&cE;@E\0=d(&8ͬ)''9#Dm#8P A^>1&U+X籀Ws7gBwRUxPeDJ_Ϙ*;exQqq\n_ėebi#va7mP8^Ifغ(,]\rxέ	a QsAyēs3S̕[\n煽dѰ[i]im/zӶbv4BY?E]\\!^&\$a\\lqz%]a:_\$1рj#~.B`QLj8B\$9KLS895Zl(gd0/t^{\r8Fz#)xYg`~hzk.<iAtkJ\"!lrznם!jQ`~P]3!S|4KkFq0v\$&\\4|lE7vgF#Z89g̢\$a[6= x@DJ!&#}1ʀ},nǬ~,0CG}7_8(/s&؀-T^\r\0ȡC\0\$B\"h<^DuG/@p >Y\n,*RIr.0&2P_kG;;zF-`Ω 7y9d˲i~O4дwf_Ѕ(WLZ,n\"C֠T}|f{HͿ5sc*x^BmFd6(.z^\rٵF>2fN\\iT^.7Xhe4=&l:v FIr~/FІ?LVr]eV.YuB.Vavd0#ḥׄw2p\";̐\\PF;I\$ݡio8߀w;w7`o5\n4m҆.Q\0p\npI\\(F_TBxp~}Kk_\\SօFƾHQ;s;YQt1zȰA6ANd5u.ZSQD생'Y'ϢTv	݂^xu%\nwL!fx`vhio۸[Bph69:\"Ɉ4CrңF~	TA8܁ϒgS_v=7OLo}6쓹_jThXL!#J6gPE\"ҹ\0,K8n6\$xM<@P~jU>̸sS\\WGr(P32,{\"*0A\0B? sM.4'ؘ`n3+\rJ!PAJOf,^㟽ڢ,*<bx4a;;?(֦2cX4<譯k`=SrFaꢃP,l\0\r+?-Fo-,O\$hw?N@(i3>/u;\".n9(6N񑢃:/>{{£ϭ?b*p}N\0tdH(/^f!_;0aoxPYVD(;h(h=2B3\\+.\0+]\0+\r/I=@ Epޘgk?q*t Po2/|*(gAΜ\rA[*jh`xZC(k4A,P.G\"0f#&ŦzHbI0<&\$?\n2=79bE-勿	|&qAHAJ1	+	<&@\"PR)%<*o26zPN\0-r!\"&P;]x<m~!\06zߩ;Kx+kj	:̳04=#όUhj%*	x楈`A2|pN\r@@zgpl4#غp6|ᱏj!pb.hG!t\\+A1E,pΒ1p>	Jn4*D@pj@{ݢ\nl0\nN+Q\ng\0]̎(;`҂墘UC&a*2jn}N\n`X@rT|&fSLI'0A\nĴ!	H2%+̏./\"hoPơ\"ht	}PC	4;)0|݈i\\Nb.8ڸ\rՃV33Xɦ\ri\"Q?r\nCTDIR\$<H\"L\$*tHl:CxH㱲<t*b\\I:xP <A+tHDF!DfKs1?sQHsoME*4bW\"Lb:7vwDT!\nMS\0Bl`2 Ӭ2|+/7Ӟl آ\$\0vwgSezKEm1,[h1qOEK\r+/U	6Dꨠ/}h	;aÎ9B *\$^pÖ_1Q \n\rzVDEdFF%u0\rj`2EaE	bчF 4cqEd^F#bD8h\n<l*=!4`B,{0>\rtjE\"\"m\0\"NCi\"6\0H\$,jH\\h*\0D?QD(2t|oe&)\nv(	|B,ZbQy\0Prr)Uyw0Z)qcfFx	,Y*0hQ[\$苯4rLⳝzuAAG*	91\$Zv)49\"=8惴?iޔwc\nC.07*\n`3\0[\0`kײi/#\0ϧ h:`2`*\ns*2?SiS 3OHahAY I1Rsqu\\hLXs!_3 GZ`赡|jGyH\06p!\0\$\n[HvNĩ\"\r>/,Z=8V	2b/i6H98 \rW4Q^ccV%\"`\rh6%Qr{#p/8H#u	#2(#䇒6#q/N\nTN؇\$!:I`/RD/jW Hs<>krrMA`pVOD1	T%Q7I:k\0u4-\$468p%HfҧPmwȅP;{H{,@ds@~y8'J^I|I5yfMq'|Y½*1QmJ!|ui>Q\r\0GSr+lWytzC1ʌ)\0a %ڔEGcQIDdBC/0/gDtWFql\0X,8<4GȪD3k=>H0&72jBDh`>6w9+@*t.ң~{|bQȿ4HAt\r\$ ).2 DeÈ\$eH1|!!r!ª8⭄-\0S;;P+` ƚ\r /\"4rB,9+\0'pɐA\0\$h*|.IbLlE3(X (	Ql怊yEph+/NށSyGK|\".<2%<kIe c?xr〮.s1loL6ߓHǔ/\"1+3Q0\$1n^z30(^u?!R\rLLɁ;BRJ;\naobK\$x+sLVWPUK/\$÷/TK􀫞!S/K2|p_s|*`AV:c[G(ub5X1x\0 )	2Kp\rоM&L^,\nHBXt.Y̋,p\"Ë,Xz'48`6Gx3q3)D<@i*HmF(0Ǜ+\0ߛo1Ahn\0+ROQZ&S3t)z@\n9Âm	V\n/XI+`)sHQO#;\rꍬ8N-6B&`AT (jͦ&zpL\r*X\r	@YVEߐ<x:PټE\0ӯʅk9CaƔ BCt,\$oSNHtd<M:͐\0Ѐ4i\"hB9,[;:=R^\"/F\0D&uP1,b?@P)d@]\")[m1t,`4a>\r\\;płya28\"qr!KS899GZ;dNdOː#H=\$!-b\"Q/.,O`\nB/z=-^rD\0Hdblυ<H7OO\0@ȸ\0=h\nx>~~ՊI%z/(A#u\n@tt\$@@0Nih 	'C2t~p,\"bIGtwqp>?0B=발BA7л;ot);:5mB!L/Mfu7B/>PCP)xPKFcM8 d2RL\r@bV᭚ NGB>*\0(t2R2 0WAtIjQv;\0*xbXT>,07L > 32?PTXfZL6>r\\~BHd&MBbD0&3,:ED3b'(ѝGt+at)	Oƅ\$	iPDDӘQMɽu~̈>TړH.\0]<9#!PbNt?;b9Pز+!َR!_A78jtMȨo)NL@lHC3^#vD+!QȰEeJPu&DHEܖ33郱.\r<oRQCcD:Ť_+4J(iJԈqxIIXp\0D\r'̛z0QF]fF;'dE,YRP`|ԌK)t-P% 25l3j!Fbz/D֕'sk4Ǡ,8ұLk@H+u7Զ[]+\"TS&JFIN\"3R'K*4Ҡס\n04=#/<K Jq=`u490*R؆c#F\n>\")<.-!xʶ8DJڏą\"34Q\$}4Rk&*Q.XB\0QDE\00Y\05HTY{F %[Izd%PXDD+ELB\n DIV\0H4l2Eafd`	OԤ%	\0%\$%;r9pXgEꂆɀvQ@)\nPUC/E	\n?\$Tb:\"ES@^UE0qcIZp;H\"	\"I;X\0U5@9ZpՇ\\E3C/MPT1%KS	d(0FC<i1VhXXF1QehȚAuE`]\$X-Ywl\0]|\0VlUW#ECS` \0{	a:Mbv7D;Ua\0VԹ'@2P@%nPFbŠb\0ƦpIUum'U\0!Z^<Q@>V.]Dpu\n&AJɼ-iq/\\+멻(fPl#&`X.W	\nJfjGX\0x1ū(,c^ˇVLVn_W(V%<ko5UiMFV+k5MTUzoubH1¶QVB/mTuWB*Hj.Hg2hښg횬L w8\\mp\$*7q*T:\"Y \\+Jo\nڊ2W\nF=tuעSÅ^EZ9[unU\0?\$|>Mo5\\~GD}#bW	bMIeNl\08жbrYe\$Gcm9#@VX19>g`@i`=@T%,?lN%kU3<LD C<pܡe\nH'ăPE\0\r@0Z8b7Dm\0:݅AW: #V8[pHtBXi ; bOv/;\$5xKҢԉhrE`ufYDȶ΄/Tji@\\ \$Ve\"Og16h5aͩ\rZȩd\0vO+XC B wt\\@d.L6H3e\0006ΞW38N8Ld2HPk!rp!Z\"V^[f!`iGBjblB2w8|^T;mu\0ʅŝcA\\hkg`!]Yt͟6tg?gvxh0%~ Yh]VC6'v!uV4b91a)7'΃I 7}JI^\"i\"\"<[8#\n@abͮ8X \"Z:uSm-_(449@6K\0r+TTPBkPJB]Fp\r{kKl[oCgl9-٨ l@6[\rghhTN<m3\0NN1΀ʅ\\Q?a\0FHq\0_iA7HO[[O܁\rg2Z[\r_R\\Kfk2~]DˋQ\"R\n?DL噂H#斸aG%cB.cbk7Pw&M2c'+AݡVB̐Z6Af9!=fpn!*'&Xo;.^A;3XjF@`1tk[OV,!.<%!s_3U*nR%aH;)z.qpPXCz]5Sn2lyLP4@dX-9o9@ֳKc=y|Z@ +貵}5Sb=L\"'LN65l \\ETN\"*Qնp}!2a`s=[lƂ#b14UQu(CQm\rFbLa8Yb;]@]IA.[<]XwL'S]0]o2ce8:X5O;a\$Hǰldq\$졐wivhy\$B3'uX \$lB\nY1]|O<,xd0`HJڴہZ\"r;s\n`ؿ4@IH`\0d.>B;=NkqTB8NpgBT]y4\")}q克w.^Sx⥚\\ca'\0h*eK䛜&yu\ngyWox8/Q2=8=W^=\"^%w&I\nnpDtӘvU<2])/FZ/+\nrӖ4mN7F4#\$\rB	<bKA\"*jp/#\0	A4v&whe\\Ų(P+Q9&qKW%J.7wCʃ[\"-):&5ggPҠt{\ns.=;֓A*k:Q>\0R)?.X0üIW|!yw	90]xwܬ#\nn'x\"b{\0`x\0\nZ#aW\n%N7t|WCQ]UDYnpN'\rnh\"j;\rޡ,񰈉ʳ⷗+>ǖ	&P,.(fW 7 oba\$R&\$)ΜI73\\23s6nK{&ATlx/}A:`QhHi#5w1qe+&h{`0oX9S`rU۠:CV|[E\\U.F{}a9JI*,ù\r\rg&'0\\_M!ąsoj\"2-w<}x`\r*rdm8` 6*\"ΕYdFPV4Y)?FUg=\\;Hkx+b-cX\0i_ݧH*{-];%؀FTTf8{X@x?S؍,	a\r++/	.lI*2uf(RژE4iĴ:L\n-].'L|͠bc@n8L/9`SMCX:ϩ}NRx3StPrf*9WHN-qcJڃ↏⫆}r+S\rwNЪ1:O+-s-b04Dt~ˢǪ*޷3r\nCɲ0tz뷴!43p!+wHjĤ*ԫ6Bΐ8\0{6{.870?-݂\"ކ]y*4P{XкԎq7_cxWu;b9m29jT*N8Dw+rZDL?zDF\$f\rTx66@BODi;v C*3^!I8\rl	v넧=hފ, AL1mL˂X8?a.\ns_6Gr+)k-oK1+'Te+7\$ˀr+W#sG`iTè36B40C\0㜬VʚuCaG20[YNc=y>NqFs\0oN8\rѣj@X:4g8r tQ¸؁Ck-	:;ei\rHƴRe9j[:e/y6eu];,Kt^FYzBqR%#VAt44Xs>@:Ŗ)2td/\08.!8ci\05rqM/d	Ch\rNf>!vd!Fx|#˒fe'^	er\"	\n\\iY\"c&fJ36h\"\0L`eHSk|ȹ&)70ز(r5IS^-Q<1͗\0ϕ\\9^3fHH)y1bo(<`-qg/49&A*	:CJ>	[n{`-ἏrMk-udr`d&C)/T>Ch@iףM-{{`WAp_.鋴g4yg\$b(4k㕞hCߞ<|\r}0DdKpNy`!Ά\r\$v>zƶ!y볒CǜC[\"Q LQygqx|Sc~u {Vy y;9{6h3p8wd;mD>\\Cg_vu{X9hvhyv'8\0kNhr0\rS@!{c*hH%fz;^ɢlj!5&whM<7跢*[׎hѡ&X%Fyz2hzpRga%1(Vf.	yc>؎c](8X\0٤f4X|_dC2<,sK(Z`/\\,h{`zAr9:fzh[)~ci\nƗAXP0b:ldIb:nj\n2Dbɰ`DmݧD0!\"DxW :~`k@!_^wSKvݷ rfM4aۨtFD\nX 7&\$vYUAJ,\rWmػ JZ#zM7jehp`8\"i\0sV\04ٞoc.~HB00B8h\r8}^(F\0hA\$A\n,/A:qy9ϗv#\06}ƠެWA'9bsZB֭\n֭tpzK﫠L/\nȔyY_lȤiXks[[ݘ\\墔:9y-ƤDk%ʄk[!A+Dn>Lٮs?yHg`IC90hoƷ2ηj~\"뇩4bk\$β\n~hkٮPZӮnn#/*fp_V~x)&JEA\\E|̅\0i&>f:,\$?mƵ׉^(9F `D^o؎}\"h==8|yYTJATs2Tx~gcy;Ge~ٴĵi\rkk\0bdQ![@}tM%kQ\rZُ%@{5&46N-!eNïK֯Qm(Zo5)d\rh@~={mX-Z6Z4d~,?:8]`@8d27%NNEz`Pj{;4yaT6\$Db\\ >DXIu\$Y @Qx}\0{VR@E&ѶiQ=.a\n	?	Gd+L݉MŅ3[\\C^K_P02jP/yvP\nAfn'+ BFuіj6AB{c؊m׀]߯R'C.s>Qvax=\\*#Z[=ۆ*Xk8-\0wC\0ID:\r̨)~Qql%|bX t@oKpfL\rm\"*P}BnǽA7,Ak%dbj t]:HV:\rj=n!\$q󔛏40;@ aOÅV/ N.BNxdEq1tSUr˅5o6iQķ-زBnuFKaTE!dM\\F+%M>B-C]!&ˍ!@bk3֥]xd\nɹ_woK̠MePVoCDVћAfx,VACؘ	x~f	,N|c<&`H(\\H*VWynJ\r1\n'P01.\$ˑ*8m׀mdN	*->q\nڎ&oGw!\0ݻBVC	4p\\67^d(4-prV͔l啷{Y~V;P!Yl}\">^s\0XwWram%<{oK0c<uj}vC|ƳSc(v!.y\$m.eu#epZV'd:'gAhT+/RKGYK@&ۛV5^GP!{-+ͥMtr&p\0+oÞ7WԈfxc|,7hIخJt26ep\0r=:_7]CQmm;b# NxAaP-o>&gM{;1\r\\Ɇ	{lLP-M?lAp4tiKOS\npZmM؆Y0A	ܳ;P|^)gHKH,l\nd`o{EHLM rKܳ/-|޴ފO.iĒ/M\\<l*'0(GT>\rHhK{m?!<('pʀ`6 a0&/rBXbnxHW5@0/s,\"Z̉+x<9\r8|k[7,OG G\0;CR񭰇=KԐ-0? 4qɝ#|`0x1IqͺE#vSd\nBVOk<?ܫ	\0l85JeUYա̬~򮬀Cpah\rvËϽ=*JlD6ᐫO؊+g<A\rOD)Iٞ}piKNDEܽPҘ7b<Ҥ\\8\ns%\"߉#|*I\"NŘ32C%0va/U8c9@clC*PslEW@ 9Dֵ<ѝJ0`#Xm۫Qϻ\nSp(?Cv\"vam\nuTkc	WT={ (ULU2\0~\r_/S@e&B\$6Y!-Uf5G+989m̙7΁\0o41Jcu:3\nRo_x]v	O<\0YHNlBZυ?\n|lg^'a\n~(\\G~rx&.So9@rݚwhcĄ8\0a`\r\0צpJih=S%ɓ%#p3@7vLGm\0000mvL@v?n2m]Oom\$3b8yW\0S =K\n	<3Yݙ΅0nrm1D5\ntVGlKbS،vcyu_n\0\0lv7`\0cݸe\$ ;S(sUHlIȮ	<V}sW\"y <\$*xN(?ܷ2}\0ҤwW]x1߷wtw(muL\\^|ǡ8T^]`	~\r\0b|cOw\\;߀f:T`ndWj8!w\\*a]ICm`!T>Upu>0eDLG@\"@#L\rdstMHcu]65[쥍WF	x^ld@3%}MֻUr5lmә&\$\$(1:/B=؊G	rk,NH#U[IX}x%ף{zṈDe^K0ow7xCW\nyXwO~In&0y\\	}:8lԥXADRJƶ:X	1Vc22[zr织_z~ v&A/ݯv\r?6wx\$t]\rހwW\"^=o#\0wu]c~_uwmP\";,:)+8_x\rHɂu:^'zUם`me+yS=@6s!w7蹞K	\nQ(B'E`W`=XxO%yc_^0Wf]}\r|phb~^\nKaZIuMڊ-7XIN#zխg;^r<\0zyMJ)]/Ul[\$KC1 b\\`r! 1w_2A\0pQ/'w*U#瘈<\0;'xh+(8Aɘ9^05ˀ(vȢ+;\n'l\r׺V.5-֗s\07w\nL4Il0d_Oi@pNWVh(XE(\"ˍQ*_0sT9\\׃9W^qo~|cq/?\n|F'v'2Gm1ľ|D?ġM4_\0E_鈌-\r*8Ǻ](nҶ'0&ج{ݞ^!zՑ\0ZpQHu^?f\0{j;%|-y-x>^9jsB:=	@E\0i1\0\$|]#{(p̛H`9aulofaB\$pjľkM=AEyփ2y+}do5[q/v}y\n@жA|Age=iBQ0,-[HL{iz\nG`	-\"{	]yB\n\0cbWQT'p0\r]*=.#}q.\0HEyu6m^M\$Qz%	\0ɦW˂2 h\rߖ>	x\0#Y( )AX{^e9xC}vVc~\"p%S@;OJ1?\\(6~~T(8]󦮞ic.\0\0(!\\*\rOh`	~*Ȉnw-֕ Ț(	pRKzx-&WwwOXͦ.#Rd\0\0q0klVvM]=98AR*\r_4_tMGpHQR|4c3AW(\\׹3c+1T:JtUfpxPs%l0+9vU y!#gg,SUB1cA Ixe!P`-\0_3^JLm_nvfY:[<FlY'ylrőw@ά\r*EzjQ3/Tȿ\0'RXh@pԫ\$FXBm\0\".%υ~⸀P\0%Xt&?o2BЗ+_;nB=F!f+W>@U8g@[jFA\"r<}; H0\"j5A#P8SOIkq`캜C8Rtdi	\\>:E+wEǁWT@'_@@}^B[Jb@MaC#@\r¹)d/.tdQ7j3pa4@.-+!VQwO>:>*`kaXXy6`	*RL)[7L ʻfHͲxBPE'<wdp2\neo`5gvP\"ł`M0%r<H84|זl^L>t	H\$an_]\r_̸8m\0_\$:A\rbO_)@FP'X&#I%aOo=c*\rP=c-*(`ҽJ	>Ȍ,LB֩lHڮ(	:iNTtܾH>\n4q7¶B@+eP<[<Pv[AxĠ0=/R*@㡘C10hናgTtbR\"8-Br fKZ8,Pl& (a\nNU/HZyjՀxu\0005^X~ i_+y6)[\0],çb@@A?!0ǣRe`3/䧙mw\$ɉ@sTB\0X+PpsA\0R%sa	`]91`\r\0T!y<RypRSs>^em2s%q AZ>5WA@ ~88IAe~=*dA=gH8\0\n\0-\ryb,r{i1΄!AH&ŏ˂|3\"`xGP6\nL'2GĜNBerڟRDQBu\nl3!>R\$3\\Pܟs\\M}[l:KSU{X9d*Q{!T1	: .]0ʂ7\0*xA,!a&E\0Y(WPlܟ2\n>ӯl@EA!\\3UKj\r4@jǂڧqyX.BmR^4Qp*kĩ*P9+p֛%S^c̙.l-F)Mxp &FTFf\$I]P{UyАZ0]\0F;BVo~YKN˾NJ@?4 y6x΋A;cTK`Aǵfy;rPF\rþa\"\r{g<}.	12t&?D3nT7T-;#@c.l9²_pB8q^n|Z\\0:sG++|FB&/wdCtuyg.xjAs	52x (ETzMEmaROLs育!+\$I|ʠR4 \n?#AXk^CxKB	-mAYCac\n>yBķA^2zŋkZy*0Q\0`҂eBc%Q\00039\$Oy2.6xQxuc,\"W:  Wd֞\rdV\"h?Ĭᘀ\r2JX{tW*:2'2 &<F0i\n!(7dYK\0js쎐aHl7dL!k4PhN*>q`nhK͊hj@݀+ /?4\n9W`>+rdD;Z<	\nY^F~z]13\nB8,T2ьq/\rL(drEdFBlCT*zW`/Y\0\\1d>&Mw@QC(֛̉D~Rۉ-5Ebw#\n24Jtw#)&h;E	[\n=wfa?*aehJtePa@ !15r@Ѽ!bV#\"L,dለTE'K4C0;AU@57bYER+uHJ\"u.\$<9RxtcXԤLllXRchG3WEzuSDA@3Ad&u+c\00θbIਲ਼/eB@\0r`qg*6\\g6)2h\"C}NCr}RLD,~0]D&27q81.f?B{ns=`v~+\0aE)z#0;\$g04n݇@\"0]шF'^,lgC41@a\n0\0@\"dXpgy˫(\\{79ƱN+,+mcd@ϝU~'LQUEq3Y`dw&dʁz,cq%W\n%S\\=iYd<,|}9;p PF1ckF0D` !r\"-J< Eg82	\0\n0lgH*c>j3(PAZ7r{\nNkc6`	Vj&%f0h35Esah}K|\0<R5C`\0<E/4\0Zq\r#D:T/8CeԳ|\"V6\n (E]\nn#HEF݊{EK3vix|J69R/,(uѨc<jbDfP42D<b٣``A_&\nf4\$h`rCREj܃ q~t3/\0\$D#RTqvc'x/:\"3\npuJt}JhE0iIB^\nAZs[OlJV:3z1ǴG2\0LOq6\"lĤl|nBPb:4w\r \$A\"|2b˗lYNmF;?(ghJ=jqqR<pB\$:ިn正\0j=Tb&@Eg:\0;tƜ:l2RI@Bl)N.`N攻GMe7'ٌwEzw)lT-x&P\\0,RRG\n\rFg䬦t̶x:W\"g>UV>\$ a꤮!cX x1c6\"V?qbTopu0EFGPSuBpArQBl/ǢTU#P\nv'j>ؑ\0*I mMTM5;3\$S*ObGzE0p!Y]N&#8ȀC,s9V6!sIG!4 ?~0!@H{!rB\$Ð\\Xba!@,p8.h`jr#/+P(:U3w0	bޕjCSlK`Fr5\"p>luȗB0Pr\\CG!\nb1PQ {Aȭ%8|B\$THCjqD.:li!L@ˀDWY1ŝ.\n|cAt\09LeBG@,2\0W}3#ZkQ_FDh2Zп(Qˀ9\nFH;k0<5BV\rL>ݫo0Z!%縬Oz\"\rl'\ncI^*EQ{ju(q]IIR)2BWzՙZ/(:d͇VǬoL`!dEXIAD5Gr_gj^gIAE	6*LV>?a<`ldo Rщ]D)(/`󠾓 %%)␗KAvLPr bjE|:L'&*7C塿6d*)W) ƏA1sC0z\r	:]<-\nJ%>Jsݝ`rE8O3o<`YY3#jŀ5Jms=;YőaD&PvN\0ܝ|kQ%	bJT	1A)'`)G\rjKPP]V?f`91Ae\\.HA(lLhS LW=\\\n63Ahڥr\$\$5]fx}#2C*YJt\\/דR|D+:	)hpKK2<JIb0_S̓ЕX0{Y!)v))ӃyBR\$)0!m`@\0b-Hf\0H[yM&rlGĝ\"aSx9Pe.fydI`׀k\$R(Trm(B1I	&4yf2iܻr1\"RVE?c*q85ܘNU,hrIR\0M@5Ri/mHw(w\08J/<=B\nОi043!{'|y!I-d=l2	mN^_/O?\rL7E)p'0J^`_<*Z2K\\8h`䆐L,*Ys5;dBI<^)(W`>-c]B:,\r_\0006@z>̽d%e?U88A^d5\0+cb4Bn&zE`7`An˓z`(\0KddspFP\n)PE<Zպ3}*!APUy@I(Z\$(H4I&&\\NŲJʔNZR ÐܝP@Yu`uS#D%/0)A񹎜Hj]u`Y.i*FYDz\"Ȧ<G~@vsrtԦ\nXN&\\O\0e/W-(L (\\p;R#\\@H|%9hsB]0Vܸ%}[Rg8JِB).NRP1.`aN5 %g*^t	dၵpJh\\\$\nQ.9U:z\n䩔N#.plP0@db]~G<KDu)(8<!mYvҒD\$X(jU\rJ[z\0\rn/.u%Tey@NR(y1>/&#y5_#E LZʅ\0,voDcrɐ	U<XZ)c<ʓ*\0c<bGcd'f4T퀇\"YUS_tFleڲ)1M*Dh3\n4\"\$\\\$AJnۘCԦbñ̄ à\n\n?68sd-\0VYDep'ܪvEIB,.B\0	`hЉpB,)[MLz\r\0 DF#nW3gah]f{\$3sFtV~v@1f0Y	,'AP~\$Uu^`ɞ45|fҿI68h	3VC˳I{A\0]< PY%zMMa,(25jbh\rA4|!`S*%3.AoEG/NRGpe5pek?% bQH8#B`f6Bě\\j*yYr3RkJ`)I02TN\"YlvO3SB!<G6žHP(\nPf1\0zc3eKaC 'h̕Vf\\Iiͯ3.#p`K˒Bm ɞf>既*=ř@gJ@ڐs_\0P.qɦCCb/PlϾ\0w\"dh-`9] K-o݂\0p}d˿rAzsX͸bDq3\0D^L4Q+aAg-S0a)[3'!+?*O԰譗G\0DM0&DE	KSdEI2#\n[H_I]@/Uqq\0!ݍLsFE\0lT-7ikX:q\0\0ZA)v`#J`HcI|4R[U\0.)z\0sua䉄mLμ:!Fh\neu%k@TL73zz%+zݧD^\nM@lB2y#:XN_NsHsYb#rKadl+L;yQGGYNՒ+ýBoDV\$hq nc\rjFNƢwj\r!b;<#1V镂75=f [?\$]%ӦO\n@\0jԦa.ΌאH- 3C턘-DcQ[BiІ:,n`/	\\r,s-\n7L*QB\0bo/:rM\"Rq2GДX5*?^b!,.-Ҟ10	2kzkbKG4G^\$n꿩+[ԡth,5V0Zb~Isf9vRf*@tAyVXANTTyHw.Wj	N	?!>IF%ϙ-ztpχ5X3ћtMSkv0.u+9ļaOGdh(\nQ\rgw\$\\s%SHS*ѵE\"3DG;9^@@M^ʐO.<dRb}~bQRsUD1zTY`5RSRn@fXTC<Re. eL^>)BP\r8;tAgB-:^\0H揔֜6A.DJqP.(Ry]FT,)&Pt*ޒW@7d5p ɽ\"CV\r@4\"Ib&gh<AՆ9HZjp[&n \"w!wh	4jBP,7b%x\\7&T:^H`KYkІB!(J:}kIyT#xM%j'8fX@މdh2H?魪=GOsR,\0Lg\r3h:@	!b\n.\\ӫM\"B3JKx @-PYI(57wЋxO}b'#kn([\nhp+6 \nOeBi05@Bߕ\0\00kB耀**^Sx6<t.\nET#N?\0xP\0Ir({JUa	AnpO%-VlNǄ\0z:b\"2)S>Aux2T8kP)CO:2rQ)IU093MjIEYRR*a\nUtE^l1\nNP&ZPԘgDdbŖ57ME=x JH+*c(Y6ދ.PtcxT+(+=>jkHm\nhĕ##\r׀G}ݸ7+]<OĀuuS9BY)B+(VpRhO\\<M26V@/r.eԌ@j=FJ80\n4x@ц,dȚ6TE*:\0p!(R@&!PipE3=TZp֑:Bi\rimfK;20iXӜ\"\0ρ?1F\0Zސ!ɯT`7%_7T\0\0'\0bD\0YU.t\r7bA'7t\0N\"\rPR7(R@tl	YiG1N\0\\zz=TӟI\$K0c` 4]ORnu;#1PFI'\"J\$ؔ<RJ Mne#AVRJJBSҊQKRGh\"w@Aj4QpSH<XP>v	T\0V.<IR	i\nV4h[p:G. \0(HL\"PR\08b0TӪV	QӖl]h&WۨC(H\$&KºH\\-ppFGtUy#2<*ybhD/>B̛GQgylM6`S*%au&'*Go2l0W[	>%`JlL2ǯ`,p~GMs=U\\`X\nʃA2\"e X41'c┪4xp\"\$pŀ:SyJ9d&d\"S< 3\"40m	#MAeMl*ᶥFEzU&xXQJ`DNPw3 tFq0`HLMfI&Ȳ3EiHi<?a,pR4PWJtކ0#)Y\r\nC|CV@C\0\$N*2nVʣf\\ځlH娢YT`ZP1ւ7n%JKc:!\"+oi8ƈ#|sؽQ)n\08@;N*;\n+2(P}=~@dM?F#Ҏ.uAU`>4P@+%;.AK!2M/\0Tv\$JT(ǣD Vg	jԕy\0Iu>]%7Oc:4EFCOZj\$HL\0A *\r(HB4wH`HYp\r?.r93\\,&DD*j.`4;\"FE6\$J!g/&vmv \n\rv|0EH\"T\0ER<~SzXl9N&QDt>15Jjz\0\$vi7p҈QYGD\rK֡tuj^\"cSZi~*b-X#:x\"Q7݀T_\r5\$Slg6gIteKju6iLKZzeO\0&7Ģ\n t\n\nT\$!}Q0\$;QThXTa#t@/H[\0m/p*8Jt8N\0)SڦPcS+/5¢&(dᇃ7D=T8Cw|CwX<:M'.@2#to@SvjkQқH=,F5\nIKTK'U(]8\n&Q|\0e);)(ItR900. M>S2*& /##aY	Tڕ.-¥m3h9 eNRĶ2{fsޠK,L4JAW\n~*oP\n [%a\rHJ9N\0`Z5tC&DϚH#4#O.Â`k8R&Gp_uܐ-1M)L]SU>~+	JP2OUB:6R/\reגAzQXs?Q.J\09Ā\0Q^RtM@d@8XɱQ\rٜB@SJn:ʄP=0Vz'M_KV6TntՄ]CeJr\"d̫Ϛ*C bՏñ5\0cP04	Κ>U+\"X+e<5ڗ3mI',߫7j=z*f^D\n@z*?Rzq*Mn~[ef>\n5!5GW30!\$(Ώ)|μe\r{9ZpP9-<GZ+E`,!X B\0gˎhk\"@'hQxQln%)F>pbY9Ql5V Qڠ߭2?`[\$2Ft\0ml	eZ%S|U-jgԷ\n&z:>\0\0Lj%F\$]W	\"^%U{7\\,#=MJ?D>ՒcDj:	{:q籖I\$r'T (]DN<,.4<d\0D=تNj-PwTNPE1kUҬ\$aE	F^صҫULW*T>[Z)el7fv1a1W`ؿ6a`3+\0xzDWLəT83xA{WPu5855)\r\"Jx8SqWCLL_\"dQ6cbL5](\0kH4p,	z'SuKFut\rͽ8Xo`S)-7@'M\"m.zt2W^YJ<tF+yTqA`i޷\r(!+\$5(nֺPʑe.&_Pz@Χv21AN%c+a:6E`YݔzHBsI\r\nBS`l=e\"ˣD%Xaۦ]0U[2XA<dhmC N]XM(\ra\rvc0օdys2	p!awA7Xm,8D/ܰX	PJjtAs~iqrcP^]`3\$ɺ!VpKɕ3ASJxv\$FSPHFו]&:p\0H.vp(.:\0pp~#,\$5\$_!|&\0wZ.J4Ԯ,YrD\nz\0h\0'T{\0\$^ lgr\rt`)e;q|0Drl\ncm@~4_胟kd:J(mv-)lCqjdB5d-e\rd!RodXq+Yuh#\\#\nF4nl)6%Y\\Ѡlls\raWiX'>\02e[+(,]\0}}dxYʚT*|Dn9g)2gTLet;p,C*!\"J\\z`HO\$Z\nqET4魶tq7*@H\n˰ũoJ륁V]!J1FHeEznVGm+\r1O	J\r~_u =9\rf5Cŝ٩Pif(Ų6pGٹHe800?@_I͕e\$`!Y)?e+F5Y*^Hʲ1kSUMUqM2P̙rj@g	Ct=*5Q6[v}J'GcF~\nK0,T;1cF( 3GfZ I\r)\"(Ѝ%[Bc@[CZ&T4,P\$Qb%jMvk.,Q	И?gLc5:r7B,YOr,4y[c\"r*<αi@J,)r6)a-N6Q\rB7HR#.ir༩ƼAY0,\\*_cl\$ɼVq,X_Uca#z,ǻ>8M`Y[R( w@-=\0P(`\\qB	z	7CWvai''{՚DފI@Sj)NDԫMvƃ{- \0xխ)P!ݨntUQ\0%QKQAL]\nkPT>\"UqSMFR#HZE!\n5)kqB\nDWQkh%cs͠CZڵzv`֮\0005%[a 'VȘA[%jO6ڴŭTwCTx]*\0ڠlxgMԕirrzI#M{%^=TV\0T0WrɌLjei`TD\"֣gx0î)R[NA|bhÈc\"\rF̐#TVP	3MXBkT\nTum:']j-Q1fF╹Mn8[~BU)LBʺϏ+b2*M\"&ۺ9aq)?Umz͸-[%1z\"N>e.nzkRT#L)Da@;.5ʉS9|!<P4CZ\nKPu%rVԦY/ӯw709Yba`lySƐNTB<	nJX\0P\0N(\"vѤJ65Vޞ}&Ѫʓ+s\08.\$(6m{7\rysw`^&Ovk̾VSZAeeJh_oU..:i6\$IH0F2{F^1i3Z\0fúkrOBvEu*ϲہוb	z8^\r%%:<	\n\nnY4V5!Ia[<\r4Wy/TxfBq1,`d\nB#\\rEMG]\\!x91[653<xvZF!b^yE`/p׺ªL0oVQ˛w0\"FzCb	>'P @cs	9SPY<)0cGB&4ϖ帒[Ƅ]\0s	ƆU#Yu\"e6jQ0Zp]0o9Kb@V2:e|BHe=s<(MA	bLFǕأ;~kVBZͪ*eP,)} q-a夂MB2@̲	HNX)r\$H37qbnwyaNɉo|2ݙƕ ]WF4#nR>X^aWk&7Ѻ\\86+ƙg \"0hJ:|+זd]\0cK@,ݖGi	z)pnأPq&?j`cct/J\nQ^q<5ޗ7GYV4+N\0Q#4Wt\0ȑ {XRI}	;ة2zfĝd-Gw`Z~(cdot\$Sظ@צȑib1ڍWk!\"(ݨ7 !eH;~\"\r.'¶׀PGuJk\r϶wL1!i U\$Ak%*Ivnׄ_L0Q[!tZX	㐆ld]t=d_`OփY{y;t/&-vm(d^`~VfNגv@xWB+*YF<J۫3a_a% 7v46U#!0`7J@@3'㆚zA;,%*IkT2vZ7˰p2¨m=ijb`\r5kB\r-zG@͇l.ؤ8'^\r&;QSB\"+po̓\0̉f{@8^O'{y٭޽H,:>EO\0@2Lw\r\n:~2wAiᣀA )#cw9\0Hp(J'*\$6\0վ-|:R\0x)dK\\cP;&28o`Ɇ(90dfmoSc}|5dI\0&-l!%LU|rd4ɇ4K	\0R4->0/BE)\$\\jh9ALD\0O%(\0 9(&@9ݰ7V0ЁHCD#|~Y7@- /N&,\"@g,)<BSlt ?`@([wZ|iWmGUyl!bQ%[-S	\"qe\$r]g7^~ShE8bm_}~|2ad]T;\\Rm1X!\0Rl@\0=\n\0AB\"\0\n @%:X\0\0!'_\0L\0{`<8_ ䷖840p0`\00@%9{\0w	0`Qhx\0瀐(_Yx0X<^<&E,7Vr`\${\0|S8@V\0KP\0-\0\0\r\0N\0cCslg\nH PS\0\0x@3O%mQ(u`\0.\rHq,\r>iP`\$b`	}f\0LpBOl5`_@\0b&Kx|px%JmB\0HX`hh	p`Ks6\nqpAG(cIlx`6s{8P4bX۸6HU\0DB,	pwW\0u`y{L;,7pK`6d^L8}ـ!@p{`:^[	cn\r\088|\ra\\&0<l7/,Mp\0x6L Ep:\n6	\0KzXy	;x`D>LJp6'H\0	pE`?s\"aU4\n&8Sa=y	5-C^a?&&,(x-^'X80_>Q`8@%4K/T@X!sp	v\"\n0)\0Oo׾xKp`y~/7`=`0`Q\rA<MaNM`|0bm;QAEGf?.~&ja`&lx8aS*m0D`+h'x'RDS͌<7ðFSɴH\r^i\r8*H\rD,LB4`%ㅗ.!a-qL!!~(xG\r/\nU臸T@V@;p0\0AW%pbH\n	&p	}I{pb5XuL߆1^q+f\$.B1ĥ[8B!Q8x(<A&@!x0+%L;x0b!1F	BbJbp[6)\0,E\0PPM_LK1`1o\0\0,Cb1U.,l%bď0<Lؑij<MNʲ Bu^'\0x0`>K\\(LVqD`aߋN(N!q|`mWv*\$w|6+LqXbs% W1_tŃ.lF1iIŝ;\$SZXEq(\0\n	\0L\r:'.\\ʈ?'\"%jiׅ{j0<D_\0Bb\$0/\nG&\"lE@!;[321.b4\$N(8~01(0ŬA`ۿ\"jXW1crI;\ncTӈs.\\k8vƻk]/qN6Cځpl 6&F\07fT\r.\01E	ExT&qcI)Q1f4Zc?5]֖6Ľ4rf7C9	`q(H\n\$Gc߇IԛDPf6Q0(EP\0\0M{H	upڌm\\\\jjp=!WA`\$\\L}r7\r+`,t8tx4ϘB1# tH{4TA\r狞.)b\"dP5Y8FrN}6\0?>\02eVǹ?80GiB/\"A4\$~\0+gBK\"w\"!(Ed/.>,qKCr?`(&\\b6P:1τe}fHzݒ\"g̓(r9dğ~GHn&y#p쑸Z3RA!0;7%vqur]v \0ADFbE_'=L_c}MUMdpZ,҅\$rypHf4\n`\0`䆞\0'ǒYʥ1P)[p@~y ǋS;LhDϳ'Jǘ\0(&QAy\0U4vP=@T\$N\r2`*^?9Ergc>\0wyEݰúk)f52dQ*?kY-^\$\nT9sYo*\\yT\0zUV+Y2C3+fIVrHiR Ǹ['Ȝ(R׬LMl\\\08,qdy~H-Y:2~V\\qrN\0X\093reʅk+:X<aqgeʷJCO@0<A+3#:aǜrZHrfpa\r|\0-\0ʠH#.e\0\0~ClN*5feϗ,X<%nteşt#]p,cY~Id9_fn&+Դ\"!XQ\"\$ܠ,es )gzr\\/w{\\qr_ɵK 졀rc\\!/?ĴX)`Ef0#s	ej\0\n4];&p/6\0Ƹ걠7s1\r#Ջ1^6LًfH5]Z(ݘȨ`5Y'Q<`v`ly`ı\0 ^CVd\"\$sG#?<Ǵd\0Yke~p\0r\0\0@7yQ\0005q'n0y\$x3=I#\0`_;Dl,\r/z-MS4sJW\\C`\0Ekhn͚JVYBN2	/U\0000\0gCRz4TM3]A	Sh<9@3mbp<Pvj\$3cy'4inƀh97Vh\0\r3dxt|)#c(\0d:e&oWB:-@asS9lK\0F(y'Xw()T8`;b`)HH\rp@ҥA}FP/\0c%mQA2FvDƔx\n |Fc3vZj9p`l͙(\0\\u׹UȒwS/:s3a30~uSUBfO<	EL#\03y+gj\r,vKPQ3]H(6w{/Zsf%¹%wkjgXΈJNx+7\\<`zzw@Dt)%5vxٳ\0'D05/	\0\0W1LR`i@\0}ȩِK@f>ԻZbc\0^zLtfGc:3zS,x̳I%\nf!&b\r7Wl:wliZ\nJ3j@KpFox	mZ,8vJ]s\0'ϗJ@c]6:{kI8QG׊2@g2BBTg`Iʂ̼0)Vx';f93fo۟+f `H^3fp4Ut,bE	_Gɘ%0ȞMY/Զٙ42d`\nv=&lqunZa+6Lo%8cЎx\nHgHY	B8!vx\rh>CE~Qq3UwEN,>\r%/T rS+C2mR@P1F-\0+B	L˚\"@>fY7\$hâZ2m&\nHfE-]MtR3M/֍:V`XѪ3G\\X&|.t;hz3.,9k#uXŝFٴEv_G ]:;hϭGd}\r`C{&H<)tlƈi/hZtrD\\	oe?3\$\n4iK%@Α4(>\nC9+' \riIC|5Z0)UaPᡏH&,H}-j`9aN0F\0.ՀF*qݸC@ tGcΘ>E;T|m*t	!Bg\rl^DLI|mfL&!VKt`w1J}W\rMI+2hϽ6\$H0*\0*iY3_'n),4jMA-l8t,?n~d^ZK}ύk&&0؁#cdݧ\0~vLz?ӃO֍qq3U#_EV\$=4еBvlz{9. Bt1gӡo<fBZِ-Y@ ?/5A4K/\0AZ9O%՟[Emt,3GzOޢ\nQΠ /7-\"BvhogDbT]y&+pҺ@;̓4	LMIJ^<Z2^LO	aږZ?q&=Ƒvn^Y lRރ){dI\0Ewn-?ֵAcT:;z?j]ONK4lB&;w\0jaƩR͂ed,եT	PzPtH&UKSw\"Y19'w6r*%%ĎwS&LW\0iW,aաZuIiՙB\0=SZT˪,:ˠ @f88@60Nq=ΪHiN?OvQuuTs2ZŰbk +93RxZԟ~WZg'F͚u4XUYڽ4\$jЕ#Q	\ndbpNdu};P޵]AM'Uv\r@kD;Zn\0ڳh:N\nMm|)Lh,A_?'Hnmbi^80@^ueO]8U@\$5de'sSM[upө)v\$ztUJQKRJ:vV\"~ށk4TTK/3EG,)_}FjGSvN2 +NsZOt	;X]w:\$5I\$uU\0000Zĵfg#NLZpykU{25c+KVWIukV[q>Ŀt>ڠTW\"-^Zm)[f|zϵ֪\0]5Q\\\r7XV8kCZٵ8XQcsIݝe;vf㬻RFI̲JϬ\\}M`!5kC{BY؝_W_-\nz*YաgZ6\rkt'գ'a`4G`؁q':7q~]d2CMb2p4{?=d^-xum@)\0'+L0@\$N\0|`*P\0]@\0\$IH/'p\n*vUO[ 	@&\0I\0eX u\0EHf}{*Vl\0Kp@	i\0_@ }6q\0@-@&\0O9\r:ql\0gʶ[,Eo`\$\0o\0{;@/N0f0\"lҋ#hV͡±\0_\0T<U>1h͢\"@'m\0Acd08C}/&}n&083`upOx^~|tةVsm=slSjq<;K_ڤ\0'Wc=wh%x>\rjcae}zl&e#w~o0`kIhYxzϰwcN`p]Յsl=V-0`gγd[(s\0.fҝK\0\"]]#m0Pv4;h~D&mKk(M;PBL)6ԭM=KcЙmN\rf\nmwn=dmQ#hҵ	 vK֩3X	`	sBk	`,ucmGD&]ed1DApfkvmѲoV-q%Sg\\U\"lT߶Cdm7oTvmؓun)1bf]:6/t֙q  [j[o-C:jػ/q=v;0Ǳ[6z;g=d6y]F#vZķbMәjз]s1HzۺgG-G#ׇԃ^v,=`jxӶԷ6P6eǎ+T-F5Ykc6Ydn\0ܫq\r=^bmcwԆ^m|/_f=}snՁU?ۑP\nHEu&i\rTZ6k}3vmZqԧ[^7`[Fl樊Zmujkm^L\"s;wjnI80lX	O`ے5.OGrN\$IPL=0氽wm[=7fi_p>bm*7^Ӓgonͨ{!ۧSj( ݳ 6kU:oSp{wy0rr~-d{3xư\\;s0Zwr6i3{j=7FJ+W~tnkq.^Vc1k^Vu!3h`кf\0'VpV7 ƹ&3Y0uao)G%	:2h/n]\"b{Qc-ַCYt#\r1wN\"SSX `zMeMu5VnﯫP[7qVN};4o\0{#}~M;#n\\ѿ30oŭ{6-[;:B=oww&5v	UB)a:7\$mEXMvmށs<Tt'z	[`3G#m^\r~5\ra>59ۯp`dq^c;ϵl<۸7KVq\nuJ'u=;}nKݶoO[Pn\\wbGkS-źS7ppSҟw&0ɿ?9[bűP\0:6DneӞom(;:ݛlӲ;t!ud#sv?Sfq{psokhg{Mz;owׄ..;.;MNa)|M%wp%<V*wʶw|=xw`ݻKv};ژHU+s'0\\po~o{xWx/\nwoՍ}Af7b\"س7l78oL_vR=7DTsW/ kZ\$6c|L]^\0\\;(pVլ8'%]=Y6]S;/>\ru5p?wpcfc8 sܥ/}\rL.piܷk[o{x8p/sFn}&6)پs0܂\\q\0{Lt-\"w>k{c!NBde\r_JND[#oA34lݱwM7@;÷W8#Lo7~:iۊ?}Ǹk c׀\$8ln+qG+A]Po2/zN<LxoⷈNm|N+FYˎO|Pv<nW^f\r[ u|^/|z\$tǷG~n;xqVӸG^\\Iq޳z޽MX4r'\\Q-7Lka-~kP@\r\01/ͳ\$\\!]q\$onjZ}&&ŝv5lXZ(c%(.Uաߑx\0&MW'< d4\nrA0?h+{&JZpF7nITr%kOKV&4䷩tZ9rV؟a <;+1/t9۸O\0@v6K2cQMFXqk¸g #>pCjE]wIiͺmF0LFF;uxeLyl*2N]݋Kљ;#g+lwpy-hehx!@,L\0aL:G-\\x9\0ǔ(D8y{Hm:u3+; )\0@\\\"pOƇ|\\t_#/\0wUAk\\trB[m\$jѮ\r)9r\0pL#A./9oӭ,;z~dwɧ\0atp[VV-<qӦk,aq%^cx<WmA|qoGؽ{懧1gɀ'sVܡc~MU6HEjE/>-<ՁO1\r/y`|ja`?7P=yA\0ؕ΢L(`rvh/9⬋/<Dk`mRY^d30vWk*B\0烍_X_<ys嫍C<x:qmRsv:6WB9߅;O9_<nsAΰ=1`ya`C?`{ \\rD>{y>|r(@ 7_b5tl2+?86	͐s~ \ns~WLO\\ҹϭ\$0L.ѫ#?}z\".s˕#5U}@jȧypO>;C@`HRE߇SI:-3cƏyo~N3*o^ś9ظSg,R. 8߬W-_7.27wj^?;7PtO;!@}<Yk~N?{z0oDF^=xtqѯh.9tR)k]F'q|WGn, ct%n,s曐{BLyY/:h@\0ѣTO*〃1_;gPiSIޚTK7@˞aMr@9;YNu}:ot76=A\0004s·H<PI<y*[Ɣ@<z,7uKawkʷ{MNy5z\nƋPޥAAgK(G\\5^FZ@=B#{&<;OP҅[ޔnT-Dtw'^%R:F\"ӭUpXl菟0=8b0G`GKS0\$K{ήC1s/'QNbx\0K.~^:+WAq\$v\ri?/S=7A,;~* o1AXroZ@gSs<N\":ǀZ@9O[7>1yz4b¸( άx' Tg.h9wL/ܺg(ߛK=W5O-4&\nrlk\0{L-c@O?NqreJs0:\r0j!7mh^^oEn[=LI&F6F84\r^IX;`ͽ]y\0󕧈&ݰ5gV(q{`vޖ;m6lىwe~{-v[e℞z]zH+tf^76clٓ/fn;4jYs/W`yz,eRPNd=@}coDH;9`Y{/?n\\YY]:q[&I]9ǲ)H,☀j;lh'/WGMz\r?\nCN<q^{K\0~|6Y-{vYc.)탑㮌t\"\0o%T#g93ghGmi<t'䯖c8|x;PfT:{IO{GF\\V8gă΍}4r,)KɏzLD}1̡GG~FXzt{\rNnl:'b/q}.txhyk]q>{-gh,:!Kal|rvcܣb_h;~UgIwշl!Ń	C\\55?#[\$.8Ckŷf\".Rpʅ|*<Fysw鎷;'dۘw:U!ھ\\#`G?Pt|q1Ҏ#A^-:ktUOF[!B=4H0b0/:>v\0G\\}6^PW~\\ywL(l_k;gz0_miiX|bwSܷlOC=p6EbOI|%er!]Wvw}zkÐO)8u.SK^=;kS~ڊL7B.zp^Tf'0@.˥cn漁we)^_PoqץWCoݎ^#=]1>)c֦_qsTv{+tﳚT>̆,ntgZ_䷰/?.)xO%[OY`{no;R9>\0_T[xq+n\r|C`+sGlћ1hK߫.\r~VvЇ7'bD݋:\r4OwhOO^-^wraV(!I':ɧ3:\n4.;=YH{7ћE=t1#}2XJ/I96z[qёwp0o\"+Gd{߫sWX{,'bc\rw!5]i^)Yk@^-V^Fkb/3mhv1w/͞%wZѨڑx֧T-0uN/PNz7EgopZ	aN<|>֛\\6nvQb[GM܀Ck>[I@`q@Zg]sŋ7+q1ߣu~@1̗N2p*K02攓L;|ҥ~jI?\rVlb\"&A|xyv迧L5yZ}qmEGĝ-ZG5C\$f>N@Q+_(ushjk|a͏/Mxh%߯^ci~8&У@<c\0D W	: .ӯ.?ng@yVeЧRnxY<h'U/@:>zgߧTN:Oorz_,/0N+tp\$fߞU,	y\"CZ\0knD+3nBޏ}e[7ȿ\07z>3Q.yt,Kp\0c{8nWmf:ЮnE-BO3sxıK}'pzSǣąuuNO<+{ppgHޢ\r1/R;y06z	_W!=#(cKT[6\n롇\"{\$\$dyc&UWz ^wX4hiK/-L*s峅Wπ#/&(=\\&K<vh{wqp	cʿ/'}K73R]XyLIwR]܇g`tiLE82cOqWKؿs0Rz-7o4U׭RT->α^3z/WV!a8/Lu+>\n\\\n,[W؆d֏=mG̺ 5^~O2lOlѠ7TJp=*t Gϴ\\Q4X~T\nG)|ķ5cm/OnΆ-'^|6kvɏo4,QKaoOWKt\\ՑM%H\0SM::EiM>7RW;ǕsC9x];nNI;E_F;;{YJ^6{{%26lVw>;tPێu}0t/⤻^;=]aݗsv:=S1Ͼ~ry2{@.kѓIx{UMe@Ɓ߾h{ڡ529Nzw|)޶;zt'߿|3TW_3yC>}/wz|u|ӿy@we/QR`h҇Gzjr#H/,=S=?J/=:	'7Ur0\0e׆}rjaD@b/	޾3bTǎH?l_7ϵ78}=-Όg:X}¿3omzJ`g	u2<q¸cx ApX[㆛o1[9l5K|!Oȧ9s<~	gioxZ?Kc-Ozuío,|%+'0i|uA=m}u/mM}p=7\\ ;~v?ߗi@ɷw]\0=vI1??6ۃ>p|z㒈k^N|eCm׫ܦLo=1o1+on|IrwnF%>\\{t3-7aC'd\n\"76#|bțj;O&DGآfp\0/Wwy8sͯzu\\Kc̣_8;RE<UN3ɀ.ǳf>\r[-6[lucg7e~e/o__\0/gٙ;fM[5\0&mCڛ^וo:U>w7\0>{A	goc>wo]tv+7Ăv	n}Il`IN`،;]W=];ZE]{8e#Ǒ-LAH3Op^8;l/\"@=qΎro:y8Bɸp?miYԸq;͛ym>t^2ܶ;.gY7:D}o(==Cϸr_\0[=lWsܧ|AzQ''ߒ}~Aquu%-{e^!#>oxwx.O>mhG[/}e\nG߸Y>ߝPh_Z0H0O=ӼnKLռɿuohOi=#_~e׏oy;|7׻|H02ID?~\\Ly0W;ouP!^ݻ|Kڗ]Pv:\07x,l4oxq5] _>?X;FO?~f|K=7>߻xry8=?}#KS_UYg}d2@oߟɞ\n>Qܣ/|>\\G廟u^\"͵}TݩwJ\\6rxm;>OKC{oԚ9s??Cw-<W1Sݳg'\\bSi}G}8~>{L(¨B~^[\0Aw0,SX<!φ	6v{c/߇[bwׅS8<?xc_~~a_yO})ɦ2ؿmG׉rG~͚7VotWx%}7{g̋|/E[H<1MPشČ@ZtFXt\0(\0,\0000Lrv^BlE\0X,\0\$P_\0L{\0*\0~O\0\"z~Ҕj&nwv\n3\0xLc:,*22`nb\nr/\0^ \\LX\0\"~jb:Bd4&⽜|b\0(bD@\\>HӼ<J^Vdݖ\"Bo\0Fh2P\n8@z*64zPb:qadP>Mnrw,KX8:=<'tcy\n]) /(z^%(b؟ˬew#EaZp<\n\$:ˬ\0Z>(b^\\pl.fAt@.\\򲡕5e]\0p[\0Ud0b~K\0 9`^TBFJ\0N[\$qtU=bdwۺRh0lգb6xTX\\zF18l=.'\\|xTۺu\r7!k0e6=w22(KLzvª)kh (cjjlj|ш4ª(Ď`)R2RΥ|\0Z*qC\0#@\0@nBUj2&9\0TTܖfP* h*	úT(,\0N7-j.~*I:t^Z4~Yr\0f'dÂ`&VPYd\0YZ1ƵOfY~'\0dzSxJs:)X`\$:%2=&:\nHj4(ь-mpfԄk\$,4UP%\0A\0<r\0DH\$\nђ7d-d1\\ &6\0\0W-`-`k,ߣ\0-B.L!PWr\$.ѐ\\LZ/j_F3?FJ!mLιRgC^f~.<\0`t\0)\0H0(<HɯZZ\0-RZMP2\$*1@\nzjxRVKz!VƠ/^\0ƫ@6A\$m U68\0>\0)\n	\0uw\\x xF%j\0\$6zo#\0:HHKmbL.Մ18Lbnf)h8jR@J<UCg\0s*0E^ c\\hIrn8ג\0\0EG[\0\0A\0\0\0007v\$R`*Ӣ\".ӰRt\$P+\038\0N<LHƚFӽh%\0R\0ӄrdy\0\0Jtr|\"ؐP,`V|\0Jt0vǓzx:x@\\>Lsl.4ɊS.b6h@Rz62bPpr	\n	RȞvpK6~jzmqt(YZ&ơ&Lz~Ɋɦ\r[4tQNDxpNu\0\0a	2\0X\"'	NɊ\0[Y\0vߌl*	#\0f֠l	lN		Q	9im	Lߌ	Z\n3	I\n8L;d#	0 ƖH\"˵\n<	e	7	\n4Rª\"	b\0C\n`b	T\$oX:,~&Fx`Z?\n\n\n~J\"\$r;=W\nߍ\nd!Zt&Z`2T.}\nj-_\nk	/\n np v#	z2KqH>\0ꆵ\\cg	6ctR.S|bڼs8Ϛ2\0,hZQ	,`=n\n	\nO	Q!%\nas\"&+\npP\06˚ˢA(ö\05ȉ	-@`щ*BQSKT&\\?[Kq\"[>eO_1xHP/\rd%%r\rZ\0`=\nMe	<M	9#ypt,(7ɧΫh-^˵go;9SQ\$=@\nt=\"BG\"6('\r\r\n\0X>L\nZ\n\nd_U	|l\rm\rv\0<]+	,C\r	,׆KN,K\rI\r^\r\ni0ۺ~V\nXf	\r.\0	\0ҽn\$v7\rn\"ղ)L:Q\rT%L	\0q\0t<\nZjf6ʤ\$z|ˆ˚˚ˏ	\0	˘˜ˡ6;\rFl3bh&Lp\0	svFX}-\n̆57,j{P\r\rt 104Nx	4\00044ٴIbN֫\rʸ{\0W		e	}*02VB2؋Fdۇ\n\rQ	{o\"'\0Y}\r%^jɋw.t\0\0-\0	Pe\rKlfl;@0'cL,4bx!/zhzpt 8D@p\0TbPty		&Φ,5\$J\ṅ\r2bt`;=e`HRm	{\0\0J:ȉV\0JĆ64xr\"]\"~}\"A[/gu(Arvz~	\r佺<p J\"6&!H!XG:b(p<J&J\nupLdJCk\nLiP!Tp!r!w\0y(q->ܚ()UN}(^'\rW\nbͼ6[\rq#Ű,QkmzI\r<}\rqv&d[;|xΔt\08\$@{X\r*\0@WV'2edT#CPQ+2&6\n^\0	Ɋ\0I\n*l2!2i	X819		?\0xsc\ndQ\n.C{~C\0utz_ok@A\0c\0'_	C\nI\n7	t\n^c#\ng%c	<K	c\r\năT\nё!\n=#Z^\nM	+<\n\0=Prߊ\"\r-N\nac.pdƋb,pSݠR,&\0ZMTP4},2|\\!p\0|L.l16b<!sM@ڥ }M\0@\0-N0Ep+r\03`\0!Ie	&Nj\r~\0!Y:z~%% \0ݞypC	o_'qt\0t\"?\0s!qA|Y\0gn=\0̋*r=8-bE>KOO<Z,.0\r:)	l26,Es{!ҐtkC\rp4jK<4		l|4*<@.D(J>B|Rk\0wgn\r	rzvJ{C-!*!Z\0_	x9?	\0Kp^ךdCv ͦ2,<e8\0\0D\nR\0o\nAw\0\$\"aO\n2̐	ج 0\r>\0\r\n\n؊	9\0#	>ns\0<16TO(#\0dVm6 &&*._.%A\r?9<ӽ\0\rPNHTslYcTEm)\0,<pyq|4F<Ɋ7%.Ǒ?\n4hKyԸj.zh.ZP\0FXM\0r>[\$9(	<PMP(\0m\0L\0J\0\0>Z\$gʬϭgMzn <ޮ8RzμawTzzc\rvZ؎oNǃ\n:\nz\$X)~Ʈ*Lzz\rD\0@#jǃ	~ O}\nѲK:?JXQ.	L[GF{JN<NRþ");}
        private function printLogo() {echo $this->LZWDecompress("(Ph\0I\$!J\0dC\04<Oac,D\0?2NNbK؀΄ix00sBn\"IT<6P+Sctv%r\$PK894821\0.DLc@8Q\0	#bD\nH/d3/T{Uqj%Yjw@Q@(\\Tm#mJRJ(\n&hF@Oc\"4JwH\\X*5h\rYj0G)\rDf;Pv\nXWbQiĸh\0;gtZb<q]Hpq\$TǐnBHv\0c1[f&(%\"AZ_Q*#h zbq)\rЬi	d6\$٤\0\"hr4F'p^Ja2 8NJgpv;i@suai`:dr@)A&Y*\0ZE,rhpT\\UfyXdpnC8I`Aa\r4h8N8\\X~搌JA]	^\")\",Oɪ+\0\\y-aqa1c&\0ȰQx4c\"_:]@T!HnWa`gQpom\$i0x\nќ~`PP\n d5fq::|9n}.~8H\"QXF4\nwSd`r5	É08*FA[._\00023Lyp]&xhj\\g>!A:RFLdb@jq\nj\n#BIZz\0fIf&x`g#'Xl0bQh @t p@#lCU(p\0\\p@\0Ā\0B\0g@1@l\nCQ\0Z\0C\0he5,/0tD\r#{\0]\0(h@H9 <\0\0p\0~	QH0\0#*@Hb\0Xd\\@	D(f\0q\$\"_1 =\nÀnQz)Hncqbضe	P UB(	QTpbOGukz0=\0\0*\0\nt\n`4Axq#(o*\n\0*#XRa;\"QP'c\0qA9Pw\0y\n@{P`@/h4F`d%8t00\0XoBxx\rx!!_\0QRH#\0,@\0#8\02 \"\0@H2\0c9\n4X!`p\0\0\\<a!FXN!v\0kCǀ+aA8Z8Q~Cx\n\0g	1 +6B=a,\"D8`X1`i!:1#C|q0i`f?\08\"aǸ8\r`\nT\"4\n?Ћ\nЄ<\rb<T; \r\0/`@\\\03|\0K@R4	h\r182`t11Q	، \0#\\\0 ;CcP1sx.7rea>5X[!6\r*tz`,x=A\$`P7Ÿ`x\n'\$h\0A&\rW\0hiD@Dw\r:7\n '@`,I7	 X;!lO7E*\0` ^Q8@	`r>(TF \n(=* ,6(C`G	a*(E@0h )	@2=C0H\".X~\0@Bat=Cb_ 3\"\0#13hq\r :T\\6 \\P|4\0(3Q\\4x C`H=\r\0bv(eQm\0\0H\0'@1)cpox\r40f \0(0X,AA(aN\0n\0Dj7\0(<z0JXD`8\0`-En!D/qE`aD1l \$`HX\$@E`DcW\r!pCX`s\0\$\"@PB\"ǈY>	0q\0@qBnn\naf.Apo\0An  !Fzv<\0`aA!0>\rP D\0004\0x\n(\0z\0!pXA^`a j ,\0\0ա	(X\0ATlP\nA~Af\0 \0g|@@b\0\0~\0&!J !`X`!\"a	!@F@Z@H`(\0!\0nP@6\r6a  !.H\0	\0!haa.\"A` X} XFA`jM n`\0!\njB\0(4a\0`!!`D|\0>\"6~lela\0>XTA\nA\$z0``\r!`|\nx&^`:a4	N:@A ^\naR~A!d\0!\0Aa4`pA\r\r|\n\0\0@A@86`\0l:.aj\0F:>jA(e\$A aA`=@A|0\0\0	`\02\n`b@8@@J^Aaa@\r\0A	H\rA\n\0@\ra@n	\\p\n Vd	 8(\0r !d.\0\r DzX``\n\0ax,!\rA\ra\nnF!8A\n xa\\:UA\0`T\"af<F@\"@` Hz`H\nZ ^aD6BZ6aD8j!`,N` f`*	H<rbA@!@! a:\nU`xaz\rA!`8a \0@.A<\00A0!!^ҁa`\0H\0AApaah	b`:Ab3N@RqdV8!4\r\r\0 `aH!D<\0J!\r\0a&A Z\0!0\na8!\\ !ƶ!\$!A@a\0  @ (0\0`\$ 0\rBul`\0r`a\n\0ba\r@ &\0!4cf\0!A@A4@\0> 2b\0@d@`d\n@jx: t\0aj^*\r@!uH st J@A\0AN	A`aL:!6:@*`!*	@0^a @Jx~\rPA\n\0d\0\r\$`\0@a\0ht@D`h\nX4Aj	H\nAh\r \$!Z` \\!xzA\r!^\0\0``	RP!rav\0>@9\0A0\rlVhnaaZ\r^	!<`4G\0B@, 	`Vk\0 `\"xa \r@a&	F\r\"!\na\0~aL\0nCv`D8\r`AA\"\0B(A\nn\n@v\n! !BHax&\n\0B T!B\0\0L@^@\rat!`aV@@p`@à`|@8\nz!\n`:\0d&pp@D`N\r!\0,oN\0\0\0 0\0T!2\r>D`a!*v@\nZ!A@jj*`\r`\$ɡN!v[`\r ~d\08`Rrxj\nJ&\r \0n\0N!n\0\0@a|\\\rA^ \0ZAn` rApF\0	4!\0\0~@\r \ra\r@>~7<`a! x!\0@b`at-aZ\0v\nA2A0<a@j\0j@ zAa(!Z!\0~@6\"\0D `Jl! \r\0\0!rn\0aV\n t\n`^.!k/,\0\0An !c` 	a!l	<@0\$s:8  ,X4\$*p@>.`aB ~	\0\0\0!f4\0D|\0(A`r%`<\0nPS\018	L\r0\0\0\0  t\nf 0\0 Xx@py@z A@}@<(ArL4k*@b\0\0\0 .X:AD;\$@T7A\n\0X`#@\n4xX-AJL@P\n\0r\0 @\$< pi@{0I4\rA`@4!\n`H70{db(,@P`,=@`b@,h@l\0?@	,V\0p/w\n@0`c\0H\0 h6\0 @``:Qh cA*`3\0@v\0p\0\n1p @ox\0@ )\nn\r*h8 0pm	a	G\$\0`-~0@>(%`m\0,@<T\0\0OP3!R}@G0 P@fh/\n\n@L	M@i	\0pr\0\n%\0*@x8 o4:pjL`@nRPb^P*\0l\0&\0N\0=40\\ K;A0t>@\r4 \n	< @\$\0p\"@\07@TE\0y6@8@	\r2\r\n3\0-bs8  #\0>A\0Pd`x]\0\\\0`  H/\0G\$	P!aCd\r`P\0Xa b\0(@At\0DM\07PA0]@8@`!T\0Mh@TP>	@H%.9<Z\nA<@@*H\rC8N\0\$b\n)\0A8P\06t!@!V`H{l`AdKP@\0Mp5`\0x	,t	 	D0@5P`O+|\n}<\0h\0vЁ A:`('\0 ax:\0h\0)\0J !)P!\0v	PAXo@|5\0d\0)Z6X#q \04 Pu\$~< K28\r!@\0\$PV\n 	t /\0p+P ct\rҒXHzp*j`z0:Zh6;I\$\0&efh g\0pp@#A3D@X+h?x&.\0P,A)\0H7p*dh\0hT<\n\0@R1	E\$2R\0\0nX,\rрh9\0IT@:`x?TAJ 4\n`))@t&@&\r\\q8.^h]h-0r?@l8\0	OPb8@@d\0cA^P NWp<P&\0\0\0H`APG\0\0@9@");}
        private function printFaviconIco() {echo $this->LZWDecompress("\0\0\0@\0000\0a\0\0\n\0m\0\0)P7\0B(\0\00\0(A&!\rS8m7NgS}?PhT:%GRiTK}_3ۉn>ysOTQw(}T3wrײ).4#siv.(qwO/jeM=kgt;zD\rb=QQ-C]B|Ms	΁I6+X]⣣Pxx/\"RY^>Znw4WRa+XI6~~摞|%	0z 		8v(Ъtˤ.a읡w\\9hxcǐL@a!\nz@Ѹw7/2n\n +&TI/2pxBKr!  \"\"0;1KQTYFiQPAa1zdQQI~Iq)؞\rJo+}}a#gBxrB6MCiPh-`h;hZ6XXx'B-ǰ>H12|)vqL(fToc705b\"m\\WS\\*#ؖs3gqME	f{&ZZan{E))h?A8G1og>K4|sK[xStn	&FHTp	y#VӇ1NmGըw񎰐E}[BI.{\r_G^ZfPWl[&̈́!)G9@QGIK]I>\\G\rq\"|GdωGǃ#xuFw)f r;E@eV{:gP8B(2A*qiJ~L>\0#6`Y;`)Ez=[A,A@&ǉ0d?B qXN>*~?G( {qut'\"B ]E&r>(.*C[r59:ro#[y*b\$eǛ	u	XܣH\r|r&и{	\"P+}(=5\0\\<\"SNv.\re\\@#zZKinPIcG\r4W\n)@	`5\nX(-\0p5hp@!Px8|y7#\nE0\nL ,SC{B:b\\w+Mi#W	+D|xB<EXBR!T)T\n+U19ݽ'%FJ\$Ś]yUW_VX[\raEV\0'0 G#ŗDO<Cn^5xS\rA)ƘDp\n8EHBf1V2 CV+0s:\"raUEnb\0q\n\0D\0\$\0a6b \n1j\r\0bp\\2.I#E#D(	bd2x{D;1GX*^`=0 hC]comCVr/;xDBL_Q 0B8a\$DP!8Mc\$'C\nagL@.\n80\"`]1v0`0v	pH>8˰7l\\)G#d2vPYS+e2`Y3RY^;D`\rB0huc~,	7䐇2~QyW+圷ray3:F@=?GPIa36ܫj,\0	 nIM,RfAv@s	J=FരtlQj'PɁ\\yZSb<\0:\0Dq1'h=#89PwFIl=+ʠBx\0002;s |H1\$w\r.?Gczt4CθTX1PChE:h`TnP,6b0A25ao}xj^+ [֙'%W<{R=G܀}8{\\lp25ǯ@p;#t^_GCtl\rP.0eB{u,\$` 1giSw{<<H_P|\ra2Ɛ<N(A8lAN!`a!a\rd h@\0xl6`!A	\$ݮ~\"a 0NOV^ 8a6@>!\0DBA`JL!x XTh`\\ZZbAb0Ht0=\r\n@\r*aapn+A@,!,p	P	D!B`LAJ~TaQXbd`V@m\rp ||a	A|!sA!4\n/zn\0004A4pAQIQQY`emu|rovafcB}A:ﺯB!\"a^	a\0Pa<@DaO`SW\0ZXj0AR g\rX!z!BAA!al(a!z@Z`lXvpAdt\0\0lxc#AN)KN۪0oN-53U5d&]5a63e6si6m6q73u7sy7}783%\0\"b\0Z\"f#o5CxՂ*nna\\T}!RA(Kaa, !	2c6	@a!>]T^!'AA^f& 9 \0lD@hCPr\nl!v.!Xa	 @Bla \rz.DHYZ4fC&hBQ@n!z)TA	Ğ|AN@u\n@\0\rі!D!F[DrGQeEaH^`8j	`P`D	1J>&X\" E%.y*AO\"Ouv\nz	ƳXvABn Je\\:r8A\r'5OtU	h*!-*!H@PAbbobycX ^lv!4}za:ZR'mb3^w^Y\nw!a	b@0.@(!VA		@	@ck}JV!H!vZ֞	ban\r x !	\0\n\0\r/falhi>g't<!{vawa-Wf]Өp.4Upi9~~:fl,oja\"uaax*AE'N!* VADaA@~\nX`F\0!6\$@b\0!`k!!\n8\0t\0a|A ,0	ZA2(! 漸a\\a׆!!Z~H\n\0a\\a4(!Tl\r8\0`!-ߌaT\0œCyKR\n9WAO-\"3[t	9=EQYX('@\$(ْW'aA8ә9ٝٛR@!AlaBtۡ*! Z\0A	X	)&a!4_&O\\2Aban\n!᭬a ~!V~Aj<2!6\06@ݏ@(a@jN>paLl V!XX\0dfF̅!Zan:Ꮐ[ AH\$ F!Jx\0VAZFAN@hn\\!'r }(A\\tbv(+^)2AAsD5S:jw)3a3a88\"  \0\$d Saǂ^y>\0~\0ad\020\0.@\0a\0 \0	6\0b\0<:\0\0z\0D\0`\0< \0\0\0Ȁ|A\0\0>\0\0\0ڮ\0\0sN\0*\0@2\0\0\n\0\n,\0\0\06@\0\0 \0\0NP\$lC!\0&\0\0\0\0\0A\0\0\0P\0X\0\0:\0\0\0# 5ˊޡPa\0ȁf=h\0^܄E١ȁ 4\0}}\"af\0 \0W*\0=	\0\0p!\0 ˺*]\0\0꼦*1A!ܮ|Ba@\0\0? \0Ą\0}jaa0/\0m]~^\0\n`\0l\0	\0\0a`\0	Oh\0P\0!a ea!\02~\$\0\0<h\0)v*@\0x\0\0\n]ف5@\0`\0x\0\0F]\"ȍ\0Z 1Sa! <A\n\\!a\0\npL;\0[vȀ&H8Pd`\0@FPM\0,@Y!\0u4`\nx\0\008\r\0`@&T@X\$@)@R\0}0\0d\0H	p`\0D\0006	`R-T\r@n:\0X P=b`@(D	Ph`X1\0D 0@(`7`gt\0<А`A,`\0xA\"&\0@? \0p&XH@`\ri\0	\0P8@\rQ`@\0\n&");}
        private function printFaviconPng32() {echo $this->LZWDecompress("(Ph\0I\$!J\0C\04<z==!C9A,Q1\nDAM@!!z:! t	Dydr N(t<p\$NpXN\0I<t\$Hw(,(eZ*NHaK֨euWqL34%QbT1:\0bO\")\"Ԧn!%J%>ShtHlm(\"abIddpc2l1͌!!fY݋Vn}+(-eL?	H, t	hF`^xh:D0UgF	Jc)a@|c\0 z0mg)@#\"x+YX)#~Y\0Y%~)y&\"@XJňLb@\0q0g|g)|b)p71.d)bFA\nexX:aaxG[CyȾa̦y\0+B'\0	#\$	`(aFOMOᐬL&JK\"pa\r8@aŸVvD\$HJcЦ6&!\0raZ xCqfar!8QY<pFFl0GnAAm\06netlDu s@\n0~fP7@sDzC0=chV&@k p\nv#\0004XsicrJGe)\$0\0rLx<3C\nR\0)J8d@\ncɎYPɠ`0v|%p=DЬa\rFi@&PFqp_#YpF)F`'`2\$)|X/C>oX\")xPgFnldDg1tfK\r'[#x44GCDx/u\$9\03AH\r8|&-p \"\r)t;\r,>9\0a'>;F	7\r	4LX\0ah8HDF@`h0A@0p}';XpP'`j =\0h}*rD4!0*\r!ƘB(f8F	@3\0,7?*\0hE<(\\t	 .Űp66FôZ	\0nK\0DN0 ta\r bD|DHk\0>{\0 >A\rc	\n\0(8/C@		\n`\$q&Ĉ\ra\\=!E\0\0\0/\0xsA@\"Ɛ\rF@c]9EZ\0@(\0Q+`hp<H`RT)h^ 3>Aw`J#Ia\r\0(\0`jbT^EXA\01?F@\\p6@`F%d4!+  y1P)b4!n0jP@Ƙ<?(B|/Fp!pxS\0pKJ\nE p\0#'\0 m\n!\r#\n6:\0\"8bB>cH]q&>F0W#l!8!Dm}!:*@dD\n@	(\n~`L<\0AxA\0b7E<1rMTk<>XD :Q\"E};0 =.+-9\r0W	1-Ÿ\r1bGS\rL&2ȐC(ChCwkD@<6\0+CP\0j\0001(GExr`D aU\$@HjD08R臂PBt:PtÐe⁔\0 `-<NF5@`06C0y0[y?)|\0`;`d<+w9C`e<Ctd <	<`!XX	#\0000Fpc+	&\0@A;\ne1`\0\0J\0uy<5<\0 AA\0");}
        private function printLoader() {echo $this->LZWDecompress("G\${ dxl\$Hr^&r~(TT0[)%\niBPI lسhnڣT~gh22N`3K6i}11ƆKLj39ygB(I j-U+mn5iP\\@QpKy_9\\#2jSp}(9\\fS>NsU{,wJE( DtHΤ)ԝ%:/gcy=ηx';aج1q^wդj{|g}~Ќ\"-5\rÐ=Eđ,MLUől]Bbp*\nb(\\\0`\n\0'`\0\0'\0A\0 @8\$A\0T.BaF%AbQX0G`DdcAQ!.H E %b)D\$! %\0T+FQa\$.02eC9O`@(h&%ЌHGz\\0M0(\0nD)kfm	?\$XVa=	F<\0\$E(\0@a!<B@F\"	J0`h)	c\0cA|c`%QRg0X*p)\0)\0	D6bd{#^7\nIp\0\0 \na%FcY0c!\0\0q\0pZHEnbXip\",h`0 ,!\0A\0iÃb\\\0Z2 N@#\0Z\09D	\"aYp`&(,k &o\"\0\0Ab;`<0@\"-a<80a\0:Uh\0\0\0X` p<&ـ@&*\0\0a0\"<C`8A	 @\0	 Hl'\0z0\\0	0-8AJ\0P*+ȭ-`@B@3+c@oH'P\0,@:@s0TWb\nCk/Gc!\0R8\0&\$A:CJ+\$\0KP-D2K/dLP13,	L'3LMP/53l\rM77NP?93NG;\$OPO=D3OW?ePP_A	4,PgCLQPoE4lQwG!RPI)4!RЇK%1	#?	b0x!@Xp.R@)fh2\nDATm6hV+\r7XW[½n +\r7hUຽL32)\n8Àxf	@q8hNAJ8Kllr9I\0\$a\n84ڈ9pM4'aG0\nDq!lt> CE\r4:G@C+!m\$CQ!hG\$Ĉ\"dTD#0^!d1BA8[-X?\0n2:`Ql/	&Иd-6|M7@[t-a\0]\n/xJ'nF\0a11CPc2F@}C\$A\"FhC8d2\0BhV48fQ~2MC8c`/u|;F c3GQlTiQ5 #o\r8CpQ8G8cr\r9u;Xôr9xCx<czQ=jAh-m?w[[vnݻТR`%+RaLi3֛@+8.`!Iab0 A[%8l<!:GKAf&\0\0ap#q\0EBPg	r \0P\0ch rCP!`EXWÔkPC;!J.ØAaP9\" \0Qt#\01\r \" V=\n0\rD3`P@A*HPB 1*ȴ	!1\n :n16%`f@LR`\"UQX\\0@#h\$gx=`#R\01 lo?H\\z06xAF}HcѶ\r \0Q`|PZ\n4ba!F	 @\0 8`A0AH	\0@ @h\r&i:!\0P4\0TA\nH@AhA`^ \$ @d `@0AH@\n\0\0v\0(\0a\rP,vBta\n a@ D(!Adaff@!\0 \rHqP@FAj\n\0l\"@4@bfA@lr`\n \n\0p	Z\0`@N	Z@@*\r rX\0A<\$`n\r 'P@Z!\"`0!V N@\0ljax@. Z`A2AtAb\n LA L\0L`AH`n `>Xa@`	\n\0	`J8@K!D`P{-@jN \0p\0qn \0nT Z@ :I	F\0\"iZ\0I^\0ib\0f\0j\0n\0r	v)zI~iɎ	)Iiɮ	)Ii	)II\0JJ J(0J8@JHPJX` D D`Dh			\r\n <\0`\rA \r \r`\r`a\r`\rKt\07\0a8@ι@` ˀ|A	V	a@	\r\nbA\ntX@5\05\0x2p2Nh\n``j@x`\ra>@ \0!aT rA4h@f2Aj@b7[a=[aE[aP^&!j&!r&z&!(rhn@	|@!!a̐Ɂ\0~|!0AzP!tldVaz!aPt!LnN!l!V!pab!vaj~r!aaaA!!/a!!Aa!AaaT7TvAArrA!!a!MmW!dADA)}}WBKKLLMMa:`n 6GA*6``!F\0\0\rsavLT\n!<A\rT(`\0a. A`n@!a,:!zdLa a\r4\0002!1-A0Ѧyza\n`.`A@`\n!F\0r\0zR l\no\0Xb\0` l\n@V\r\0XAA\0006@&!bj(v\n\0!(@\" 	N r\0``& jAaJ`a0` T\0!\n^`a>p\0F\0\n`>!&=\\!@Pa6A\r@@TR ـ!\0\0F\0va@@K@<D| \0\nP\"@B<aA\n@2 @Al!\0N`bZ!P@f`<@\"0`.Fa\nA@\"`<` \0!fNaL@@`L! \n Aa!!hdA0tA&	\0D JX@a\r\\@a\0f^j`A la0\r@faAR!	 L@`%*`L\0R@@PdAdp8 P``e |A8l\0.\0 :!L\0! \n@@|Ap|v@`A\0|`\r@f HfI!67t)=7838939:3:;3;<3<=3=>3>?3?i	)Ij\n\n*Jj\"&*.2 t*J>jBFJNR:J^jbBFJN	>,	^0	~4	8	<	@	D\nH\n>L\n^P\n~T\nX\n\\\n`\nd\0\n`@\n\r\0\0\n!\\	`!1D \$@!\$\ra\0A!\r R/! `a\n@\nH4@4@|4@r1j\nVA4!?&\n0\0dAt!\0\rx\n\0a:\$!X0`tAN\0\0\nfnAr =\rǈ@tŁ!(!(!'p4`&!N<4.&!*'!,L? <'|P@A)\0S\0\0LY\0+ \\,0Zt\n>\$#`X	Ux JAvb0h0`~p0Pw80\r@aP1`mY1*vx\0004Ml\rCT\\9\rT@9`p]p;x\0P<y;\"Am x~`|{^ͼ q\"־	LCFMC|ߋ7\nc\0hL0+IA*0\0;A0!P%\\\n@`\0`j\0p	@T`@m`S\n@0f\0p&\0bf@ *\0000*P#l\\\08;A	1 % 1@.(@ X6\\\r`H.n\n&hP%4\0\0Ԏ3\0hI-\07	P\"q \0\r\0-\n`=@T\0t`7<Gp\0000P5p [+0v :@\$\r0J(@,\0ᬂX7\r {.THF`zꀚ.r\\`y=H@(@t`L)\0p7\0sh\r@	I8\0X2Tp|\04\0Pv-\05 A#,\r @nv`/;RP\0Kx5@	4[d@4F(&`EA@I,l@~ |\n\0;ApA(%Ai\"\0NM(0A	A  x\r % I\0 \$ w)T\0V\n0D5\0\$	 \nPQ@@0x#fPK\05l@(@\r o@@>PZ8@_9(  	\n8\$	 &H\rP W\02lO 7p\"sX	i\$8`^@(& w(H/\\@ah\n\0002o%XįLh@IK`,<	`-\0X\r`	Q`h\"@H	\$,0BL\07.\0@P\0.\ng@\nJ\"P\0\$@u@PJV%-pK^&!1Lf8&5Mnx'!9Nv?Py@PKAP&uߊx^\nPÔDE Ix41řyJw/ỳ8,DSX)hP\$O(`8f@- [h;U@\rPiVZ\r@-pZ>oh07_<1@.Jw` Pǲıhx`p\0003De@Y@A2 g0ԥ6p@@\\ ]E\0:HX\09[F*1R-\0 <)3J\0\r>L`'A@\\g )\rpW@@-0PZZV+6`]X>PA0a@&4`/z^0Aa\rXx2\0d\nAP4\0hg0\0` ABD#(	 kL^\nmh+\r[F~/P`R24`s\089Pf6)}Kj<@@9A@t@:v\0./pxR <\rB)t2h<ile@`	PK79a\\3|/k7\0000\0]PtJ\0`h`x	@\$E&`hp	J \0\$n0w	\n\0\n K\"7LG\0 \n@\r\0%`m\0	]t\nP\0\\p9\0Z\r\"W\0\0.\0 c\0#\0n`16\0<L`s\07@J\n0\"(\0b@<X	 ]\$u\r`\"\"\0;H9ð -\0P^0\rD&n\0-aA`fx	J1\0\0@c@P`6P>6\0 G\0j\0RO@;8AphH^\rJ\0``\ru-|H,`D\0@,03\0rP)rI,(\r`t`03\0-@_\0 hĲs\n@b d \0\0rH08-A-\08N\"( \0\0@g\$ 4\0vK@e\0{\" u \0cl1` OD!P2`)40\0d\0h1\r!\0#@%\r,.E(\0C*!~D(@9Pmd\08H:*\r\0@X0 h\0\0q5@	PYX@3@\0^-+B Y\0+\0\n\00+ &H\0p	(`+`-k\\sP2C&X:TeO\r4Qc 1A\n,R\0X/	 &T\$\03p@4A0#9aD`o8A; \r8\0h\ra8!9\0=@@(9\$-uT	Y3\$L86ND3L:>N4\$M<FOD4M1>NO5'h(ڊvgp(!܊\nwBؠx(Aފw&ҡg(aJx2&(n&7ٿpo sȀ@(<\n/(ʀ*<\n0̀8,ND3{ \\p`*t@O&TΔB0@|d\0pm\0\nϢ+s+^(7<\n,0egp4:\r*A\05\0@\r@CP \n``kY@*\\eq\0hh9\0r	ނ\n54tLulQA\nMb\0' P! )U0@-,@Y<h	\$	%lQ[\0V'\n& 	,2a`_AIAIقy=_`XT5̣x34 j)Pf4|ɣgRx#\nJx@(7\n\08\r(9ps,\r;ZT Pqiq0p'k\0x5kH`P7CN&(<MjlPv/H^=r,m`|/|A07>=y_o\0 -(`C@q\0\nAR\r d >i@DH	5 \0H,(\rTp		p9 HT\$`<PA\0  &\0~(1@\0`hAt0uH #D~\n	`)&#]p7\0\r\0qPO\0\0h\$	\$@7@dPJD	!2{,\0] *Dy<5Arh@%@\"\\ *0\0Cp(@\"!P8z\"\04,	!@a\n@\nP&@7p~p@wAjyh\npL\0000@)z0\0`b\0`-PP!x\08APZ\08w`:fR	%\0yA\n\0 t4p*@1Ƽ@F@3&\0qd@p@H 0\r F b\n\0 	\0` [;|#`np`3hE\0@D @@iq,A\n0?B\0A&pT\$p`k`	D`K0*r!\0h@(}9xYD02P	&7`dUATQHL\nY,\0,\n\nu@0v\n`TI\r\0@H	\0\"X@l\0	 h@d\nCh@!\0@4W`j8h(2P9\0\0I\0T	#`\0%-xX\nvQ\0h!Zh`^hPjH`/5vp`v\nlJ \0\$pv\03* jX	pT8D\$ۉ=i&ji*g)Ιtzijh)ޚxii)Z|ij)/'5kbZ/75͂b/G6k͢mNw^Wtݞ!7Mx^?8[s<8BrS\0{@_ 8`C\000\$x,a\0\njVS|1[8X:|z` R'L +@`\\2	H9@P?\0H0 @pA\\W@/3,s0\0aqpB<.\0`<*V0\0`9dp@p;z2@?\0p@ :AunLA\"sT`#A	GF0.u\n VT@iA29:%\0lPI* HȀJ(J(Al	o	n`'`pw7\n 4\r>\03F\n-` 0Fdt01<V x5i*\rm4- ^c8X@8ޓL58A\rtit<A\r\0yDxt6j2 A*E%OClp۫0o ~<-*\0Po\01\rX`Wppi`0.\0001d\0`)\$Lh/A``0y\08@<D@J*@Qhr'O\0-@\rZ)@\0\"\0.l\0\rAN\0w4DCfU&p\0+\0P6D@Vx8AzvH \0\0n`EP7@`4\0,T	H`\0jAW\r@\rM( &X\rpD\"X6rP,4\rhv~\0\0)\0%`\0&h\\2ʴ\0t*xp\0{\n\ru:\0 d4\0<8	\0h\r\0	0`1<ٜ\0)`\0=\nPL3pTDYx?@0Lh@]hp<Up<H qtŉ@0=P\0T`V)q4x\rA@,4\r\04>-i5Q*4``<0(4 n\0'O4\09R`\"\0&P4\r \n0\0000GHA0d\0` 5D`NRD	0,A\0p\rv4\0@m`P:@ u&Xېf>`@Th6;@=P\0\nx39rm \0f&6\00'Q`J\\\0	 TT\0	F\0Πh\r\0q	\r`\0ܓ,Aπ1@o3@^@<`\0@d ;0:h\0SIv0)d'\0s:K:iӎ2`3@uN陧tZ뛾:vnr[;#w˼s7j].vwܧswYݧwwާ{(鵯RjiZngxy'*Zoz/B/b7/\0.@P(`:3*\0`)sc\0_\0,\0\0+8 \n(|\0p\r@\0z	@\"'*}\$i\0)x`-A{'*T\n-D<T\n'9@>@s\0( .8(PRA:PA8*,!p\0EF	J	HAwbӁ ` &\n``0>\r\0CJ\rU\0:\$dP\0Z8\$	 &0@aPyN	b	rp'a8 eb\0fXhId\0.\"7P@<X\\CRj@eW@flstp\\Vp@V w0`bxǐ\0up\rR\0o1 u hw\r6 F썲:몪}*7Z*2jT0`s\\`#\0rA@mAB^b@KH@c\0J}p5@Cb\0`9 Pp%~](Gax\r`[@9\0\n>`T	`\\\0\\VV@j <B(`CH%p@&\n\0`\\\0 CP!xN 3``@Vh\0\0A\0\"`c\0pbb  5l ^\0\0tlhPvL\0002X\0`9\00 6Hs qfa\0\nlB6vX8	]Zg\0006(\08=j8]J [bX!`%86\0`EB7 f88	ft `7`pH\0Y>p\0{p\rr\0 \0@1`\0\\ˏ(B  c:m`pb1.\0R\0l\0?V8 ##Vp@\\`zj( J=fKf=`f\0BH \0#<\0*\"	\rI`I\0l؊7\0@@\0lׁ@%bw+\$H\0A0R\0T\0O	D\n\0F,\0`E^(2tP:N\0`f=,\0\0`4\0`<xy\0004Hop!a\rWY #B9V< \0\0p`s\$[ \r`R\0@0\0t@z\0P\0\rH\n j\0 \"\0`\0`8Ar(	\0\n*P\n0)`\0Hv\0\n@\0X2\0 	P#\0N(cT@@*b`u\n1h\0n@7\0` \n\0`6 b:vܩ5`l%l;scݦbM'Xuug[uv'^\0vvav3wdvcxggP\0\0\"\0R\0MhJ/rrp~p;~*\0\n@5\0p4\r\$<|FTD\0fx bX&a\"h\$nh	j\"B	@oRb0ew\0lYpl āp\\j	!pp\\RpQ\0 8`x U\n=P\0\0Mx-{\0H*t(CD8(?8:E+\"~e~p:\0	i}  <\0(\nN\0m'\\.`\0d\0(/\r:DK( KK3v^F U\0h`\\ ^(%X`o(o8ɠ5(HBx,EX[CJg[`\0tr`_dx2`F :Z8y@gz`o\0xڄ(v+x^,Ch̋\r{i^#LꢟHS @z*J2Rp`e@QV\0\0W:8 yJ\0f\0XMmnwgH]\0`\rn \rj \$\0006L\rl@@T\0R&fW\0b	@[\r`P`@W\0f`\0^\0ڠ@mv[영9@[\0\08K\0} X:2\0a2@A⋰`ez`\0\0h`\09L\0[@X@ !8\0p`BE\0\0h@\0 X\0}6`p)H(\0\08`o.x7\0 2\rHHH#~Jb r\0D)2\0	> `B\0f . X/7l\n)&\0Z`0Dxtp\0a4ܨ&\0\0h\r@p8H@\0p	@d\0C*X!]f*`bCEN@\n*\0\n >`I\0	 \0mG\00!`\0X\rq\0N(\0,\n4`@\0b`v0&#C\0`\0 CС@\0`Kx\0R&ځ`w\0Ը\0p\0I\0!H.:NP\0``=\0(\r\0!8MjE\0H+RH\0c@x0@4RX`s*\0UT\0^ )\0h\0Ř0` \0G(\\N\0`@bphOĦ󤭰\0p1`2 q@\0;\0b\0q.iA\0<Z9#.\0\0@\$jBrX x_ ?\rD\0dPŀA drpp`8@Y2hX\09`GZDڈw)=F\08p\0\r|m	<?u,Q&_R.m.R.rm.ݒ/ݲm/\n/m/HK?\0@\0@<	@ R7 |j}H@/\"/b/c\0@0p-3P=\0004PM5P]\0006Qm7Q\rE-`2\0X 56T\0 @)Fxd@j\0\"mtU@ɰ\0k\0Irf!ɁE`\0&\0ӟ\09	'\0E	-Q`r\r/T\nH0oF@)|9-7\"-EL&K86M;05X\nX@@E0sVemW\"[nj.oB&\0[f	\0[1#@0@(eR\0dh]1hD@ȷꏀCp j&Ljl|\0kjc\0f2qc@vĳf0fp\0fu\0k\0srQ́5@r8v(<Ș@\r`|4{lHBڅ,A,:*Prg>\0w\0F\0p,h`sKP\0zV\0,F\0LT\n~8ꌸ yP\0Hj\0@t\0(@Rh;6h@-\0\0 FfHL\0`jP4,hb\r\0J\0S\0<h\r jP 9\0@U`s\0` YN`F@:;(hd@[\"h >\$ i\0H`Bv_\0,R5\r <]'0hs\0G\0004 F0`p(\0V\0\"x\rv0`l6Qj@B\0x\0007>X\0}\0\"iN&\0K+Ϙ\0q[5`\0`bJ`F\0kh-\0[C`,Hp\0a0DO\0`x EJ!\0\0~UHm/\0@@\0 ~`)\0`2\0,N\0%\\U\0]	 '@N@s`` ZU(P2(+\0VF:@D\00 y\0 \0004H CG@.`<N\0jc\0Ji c(aKtU^`l0P G\0˔ 3B\0`@\"H0\0001	@\rr@wHX@Rl4<\0G܉`C#F3P\n`g\0`i\0004h @hQ * z`'\0\0@\0000\0`	' i(B\0V0\0R\0` @D\0F	r\0r83h@H\0Y<	ї\0DAJ`\0a @\0D0 N!V8@R\" }8H͋\0y8ihH\r`Oi56M\0-\$|Q@7/JMԝj\rםvڝ!؍ݝQMmzUЂޅmЊ޵ВC{7t0DiP,B/Ck.\r@y25\0003E4U\0005e6p\r.b60cv2c84`H,8P@\r\06XJ<\r m00\0MM\08\0@*: H6J)\$k\0U.`/@:`5Q%\0-\n4!'\0| *\nي0TP QftҀr\rh4WN|xǊ\0(6]@+`\r0\nJ\08D|hc9S][`K\$`O!\\C\\[d8^, bA^c@	_ fpb@\0hV`j[gi)R`ly.op fޤ8KhJC\0yp#^\rn5xuhc)jΡGL|yM#2B2A6@@.2#]\0 V`kf@ [Z\0H \r\\.\rml`l.\0d(@QX@<\0aVƀl0`dh\0!O(\r\0'\08 40`uh\0\0V \0.\0x`h\0XK\0\n\0@AXPt8`l 5 c\0=K`A\0Jp~,P\0)#0\r\0b@Y\0: \0v@\0H`\0\0\r`=\0 \0\\H	\0 gT0\0 0\0\0@\r(\0\n \0\0x\0&@\r'A[@6;[\0@{(`\n&Qb`X#>J\0	\0h`f`L\0i:J\r\0@Y@@\\﵀@B\0@d@\0\nLi`eX\0&\0 \$\0\0@\r`\0\n8\$^\0R\0`\03\$\r@AJ}2\0pI\0|5ϠJ\08  \0 \$HmB\$I`\0Rp`r `'B.\0/\0T\$MKJH\0\\\0pP!9p chPimJ9\rJ8@r\0v\n B r *-vVYmltY#@T>nix @\0HR\r\0PJw\0PM\0X	 X\0@H\n`R\r6\0JP\0\\o@\0ap\0L Z6VP`TDkL\0t	l9#g\0~]ITL6w!\0`@JPHN`J@ٰ\0H\0L`9_d@\$e`J8(\0*j\0007;X@\r@~A\$~ATAA)\0\$g}S{7ak|7a/v߅PPߵ/a؃t5qb56\$PC4=C4AD57ؾ 89 :Xx90\r:\$ 1A\r@Z;	#_m`h~[ Ny\"ٷfتȃ,lfTYf\"}6b| \0e;qh\nVZ\0T%\0001 \rPFR>\0?-E`\n#{2h\0I[2q6\00@/Z-{_hm2[+\\.w1\$9`Z(\\\" ^\0 a\"0 }*r\0gh\\@f56lBG\0l eѠhH`x( gjyR`g{`u#\nB+LEV2>_N#:_T\0-r(L8lpH\052`vdT\0\0h@YB@3:#T\0&\0s(|2@\\(\nQ(\r`S\0U\0YDJh0c .үk(`KȀJ\0\0`[\0\n@KXpH\$`W\0`@U	X\0\0tX \0=`9\0\0X@/`L@!h@d\0 3\0\0`4v@v*V 	rh \0\r\0O<0Wh +xx@`\0\0(`;\06`r\0U\0އx@e`a\"h 'jmpp \0 m(9P:E`qY@\0Tl\0\0D\0 g\00 h	@`&\0J\0p !W4\0`N5`#	@) @\0Z&\0{\$\0&E z\0ph\0	@-\"\0\$(@J>1\0 X&h`b\0@8h@MT\0`hhX\0H\0Z\$r=!<.*@@`3P8 :\0006/\0( 0 Cl#\0<\0{8\0TR\0\"\0U\0\$\0\0H8gy=v 0\0004y\$\nW@AY* 	v(\0U\0LTDP#)Izu8 U\0H@Fx{5 :H@\0^`1\0R\0\0\0R\0N\0w\0n`NNP@<\0`j(rP@T(dYI8J 8\04\0h\0n8\0@x`V\\G~j\"*GH\0007GYG-`d~!6\nGHAЅd#_ H+abg\0}V@;Bm!%u!UX}C6t4؉Cu9Xb]'أb7 7\n GX!@XSY\"W#14 YA֢\$jV%l!\$o\0g8@G\0lpV@&DPLph I`TH6!pTa8t[`'\\Ƌf,f̟@vbPK뇕ׁ\03.c\08Ts/	v8\n@rZ,\0r1̉=\r&\0 @8@p<%\0ec FM\0002zx8N`y\n1`:^j.CIXt	``\n`@+\0b\rLgE8{Pg@JY:,g&X`exeZh ii v\n֥<6!E\"n6z@쌡8Y{Z7\r7jO(\0h\0rH 0rH@ @ \0YL\"b	 ~PasS3- O04HP\0rX`8P`\0?G\0T9`\n&tl\0 4Rk`zZ2m#\0x\0\\p60\0~\n@NhP\r 6\0+i(\\`dLh\n`#\0@[\0D\0`U\0сB&`.\r`Wt ;ۥWUO\rL8	C(Q\0xk hPG⁀tP|\\\r@\0`=kռxL( \n#C8&H\0RF\0``e	\n0F\0lQP`\0@r\0 ș@\0:\rH`hp\0`	oÀ,@-\0IY\0\0p@mXؠe=Sl@8\0 d\0\$\n\0; 2\0bP{\0p( \"x(ш (\0pYJ@.DxU`@|`\0oY\0P%ـ3f\r k\0r`X; *_^`6H@(`N`jb\08(\n@\0JWlH}ZY\0aX e]\0Dp`^%j\\mN 1\0-\0` K\0li`nx\0:m;qњyV`\0xJ\0\0/X Vdp\0ԀV0` e\"-My(@v M9dB\0 'Ҷp\nـdn8	\n\0Tv\rt	B\0~etX@XA-tXA]X#AtX+Ag,DtMޤ t0_\03B*gWauPߥP]/k@>T KP`V؜\r\0005bv*D\0@`\r@oX\04\0p@*KR	- !&ɞ3`+ W.Ҁz\r(/\nҀ\rx0@'	nf\$e~H@5TJDX`ZsV`[9 [(\n.[xPd PWab\0Z`]`^ŀbY ~\$n'oX\0i^ih@hZ^[ %|h0d`hBjZH\0kZVki: iEbR bthk BW\rj@fK\\050yF uV%ЁP]9tB5<bBx h z\r6phI!5Xb3խ)7SLP `\0f2F0 j(\0)kx@غc\0w\0V`\065' >\02>R;+\0¸FLP\0\nW H\0\0H^,xJ\0\$\0 *p`7c6 2BjP+\0\$  G]P`kZNSrPo\0(j`c\0M;\0\0X\nT>\0Oo\0KH `dZx\0x	%I\0\"\0Q\0\0007\0@\$B\0},X WP`? [\0JC``V\0c\r 8\0W`1nM@rn*\0b<^D`ћNF\0)\0`e#vd=?%PXp\0Hrb@\0\nK\0//`\n`	\0o@FrhtH\0i +1*yh8xt92 @!Hd@	 2aq9jh\06\0hn\0@@4\0006E \0\0  T@e\0X\0^u&/XP=,2d\0PQSc!Q\0@fL F^qfr8@]H`\$\0B`Q|`\0<`w|	@d\0Y~9\0<[@S 4VM@)2&vC\0@Q	`Fj`\0f\0 m&oL\0F\0006h\r\0\"~݊(\0006hF`&Q@p;Gue9==PFEPReP^OPj7 &7Kyh`P5lzH\0;\0Dd\n\$	 gPh	 o\0P%@=8@Aps QmP	t# '&\0D> `?P6Qr/a &Ɍpf-2hPT\n°%E\nZP|X&r\nfZM\0`8\03`=\0x<,\0@N>RH\0~\$ `(Bh=`E`m<`O`q\0G`QYaH`32Hvz\0?H'ewrw;>U	[⹑P&'n\0K;`M OzQ\0LpTRU,Y_`V\$@Xn!YVƀ\\\0]@`zWz9~(ƀi X`i` j\0b0j8IKHi.āJOOX\\`kn`i2\\;}`@h l]`p¸ej v@`tT]vܧv |chBΧziܯD.)c\$ҺMr\078\0\0\\ \0004\00060lfHݒ\0r 0z]\0-\" @{\0X@y:.p`\0\0\n`2l3\\vKK`@2zz jf\0`r\nF:xI\0\0`ex8@P\0y#q;' \0k\")\0BclkA\0`˕\08X	Ih\08 *H\0`-ԇ@q8 &0	`Q#\n\0dq3\0	tX\0000NQ\0\0`h\0]X\0`M\06i\0pHǮ@f\08%\0q\0	|O MM\0fh>ex\rAiMP `F0؀x0mf۴xp\0J*\n/\n\0#W/u1ŀ 290\0(B@o\0KÕ@\0L\")L`dY\r BCEAJ\0 @` 20R\0`bMW\0+X{w@ t2C\$X5\0x@\n`!\0NJX\0\0|\0@	`<\0BPF( +\0(\0007\0Zh W\0\\x\n@Ol>\rJ\0i\08	@,B@8r`B\0C-È	\0(\0\n 	5\0x r06\0FP\0o`\nj #\0@&(2xl6+x\0  az#eF@b\0\0 h́\n`\0ӓhZ\0phv>\0z\0d\0p@x\" *# :\$ J\$S	@T1\0%Q[r(SH& n@`v3dE|	'ࠧp\n(\$\n8) ()*``*P\n+`@,`,<@@?p.\0\0f.HN `7\0V\0000@p` \n81 ŃJy\0p1N&2J3'tЃF\r(P\rP, Y\r	 qF( 	@p@7n\r69\0z7@܀8\r\n`-,d9 僚x: #=0z;a;@Ȉ;FdeS8 ^vÁz=?Ã^H=p>\0aj`}6T@g@r?@i@!n zs |\$@^6m3Cn\rr^\"QG,ecD#y\0k.!j\0'Y8O~5*]p !\0P\"\0\0005;'\0006@|\0x\0Pp(\0`%P\0;p:&*p&\0000H\0 \"<\0\0@\0\0hE\0\0\0<|@ (@\n\0\0d	a`@15\0\0 \0\0pX	\02\0\0F P`\0B\0\0 \r@@T\0@\0\0  1)9@Z\0-0\0	` @k\0R;O\0\0D\0%`\0000\0+\0}\"5\0@ ?\n\0\0(\0\0F\0,\0\0@R+\0 @`\0|\0L`\r7,`7D5 4\0\0\n\0`\0\0\n\0\0`@:\0(\0D(\rPi)~8\0A P\r\0Rf1m\0\0<\0@\0@;B\0@I	@	\0#\0s\0rGa@+\0\0\09!\0\0\0@ `-\0@.\0\0 /@<\0(\0\0@ 8d@\0Q\0 q`@V1\0\0`\0A| KcP 4@\0dp @*b\0F\0\0\\G`\0(\0\0h\0Z \0r\0Ri8\nl\0:/N\r \0002\0`d	\0@\0\0`*\0t\0y\0\0@6x\0\$P\0V@\0PX\0%\\\0P \0%8D (*n\0`!>T8\0`d\0<f\0 -@v\0Bv\r\0 h\0  \0ۀ,\0001D`\rY\0\0006\0_P`[,\0L\$r4 \0,B\0(<@ 1p\0F_&8%J(?%JFqməƺgqrrnYʛ_o%;sK7<g<oa\0mЙѠnEa:\"`B\0X`\$\"oX̀\01\0elv.\"Z\0;0\nN\0+SA ,\09'L\08\$\0P\nB \0̂LcCg ;u8\"Ѐep	 oZ %5UdX	3\$(S8<x\05FT<|\r(K|l'@*\0Ģ0@Z\0wp\$fƎj>h 0DP6t0\n@[\nl	ÛH2\$[ja: ;M;rp\0\$|X3.\0.8C\$p@H 1\08p1`\$e!f-P\$l4\nҀ*J\0An%:m-`7IL5Jk`<\0c<9P;ߏ(\r9\0v!YN+p\0\"H)M\0I_ֵp!LB\n׎VDeyRGF\0	`5ʅH\0	@8tsƀg!(̉@j\nt		\0\$@.@k;Xh\0 @Cy\0 \r+\0<18рŀ\0\\@\0@ a\0\rO6#\0f, .@=.D\0X@\0,Fou8\0<9P0oC@j\08@`L'\0h6\0\0SP\0\n\08\07<8(@8Dr\0HPPM\0\0Hh	@f\0\0x	0\0awQFU\"@\0Ku,t||!\n>@A:BP&:\0\0\0\0x	\0T!\0ާm\$,\0hC\0+\0@\0IV!h4.\0\"d\0ppJ+\0C(	\00T\0ɐ@P@G;s@15\0002X\0VSU(@8\0;\0\08	\0&\0\0\0\0x[Ā\0c,X\0#k0\0|\0x\$6hE\0&x\n\0000\0p\0H[dSo\0\0\0ZN'\0\0/P%w\0\0R&h8̀\0s\0\0\0 t*F\00\0fVX0\0\nP\0\0(n<\0P '\0!L\0\0\0v\0\\\0\0)\0>@ 'O6\00000@X\0%d>Cc 	&hτ\0\0 2\0*\0~8&,`\0\$\0p\0\0\$\n0e}Qm\0?d\0\0X\0A\0A\0#`\0\r]\0)4``\0\$\0\0H	\0Yk\0sth0@C\0004%hPi\n\0\n (Cr	y\0BS:7Iќ2q\rș:gasrrY/g(39u[3uua#9XWX#:dLH>εBt\0K[0PƀaxEJ63^,V`4i2,fXбXfx\n	  @No\0y( _b\0( \0003\0\0EZ\0000\0\0x7\0lO`;iLb\n><r3\00D`Èk\0002 \0X	0\n\0*8CS٠9x 8q%PȖM\"l\$/(lh	Q%/\n,xP`4\0@z) \"A+`h6^a\0.o\$\n8w 1Z@ ,X017yh.\0#Թ|4\n @6@p`\r,CP\rB5\06Ÿ`[@!@t\00@1\0a\0Q2xp\\p\0`;CN\0\rL\\2b@tjN2.\\ D'<o|ı0с6<\0@u\0004\0J\n`/UZ=p  #	`\0	TU(#L@0+\0^|	\0\"#{\$\00\n@.6'VD\0Y@ ᣀj\np)A\0jp@\r%\0)\0@K\0L\0V,@\n\r\0-	\$@	\0#\0Hd@jM\$H!X`\"(\0^rL\0	(ʆ\0\$Xp\0BiNB\0(W*_<\08\0)@\$\0\0@\0cU ;\0 :[瀐>D  .3\0,	 QZ[ PbE@L\\\0ƓP`u\0L(\0lXuW\0>J\0b\0<p\0\r@(x\0CA/i64H@\"\0tXa4\0F?1@@\0F\0\nN(\np\0\rE\0:\r:\0r\0D\0W!U\0!8\0E!5H@fHl``b-:@\0\0Rx\0 @i\0Z@0n1\0001*\0\0\0&\0G\0\0\0R^\n\r\0vͼ\0\0\0	 @3@!\0@H\0\0\rn\0\0A` &\0i\0l\00\rc@~KrQ]!X@&X\0#W\0X` 	`+}l`\0 \n`%\0F \0H\0\0btw\0P\0a\00\0\0_P8M\\\0@̀H\0y\r\0\0l`\0 O?\0H\0(@\0L\0\\h\0000\"*'\0\0\r\"e\\\0#\0>\0r\0Ŋ(/\05F\0L\0M \0+R`v>7+~A, Bq` q7;XF:&2YCm\"Ф#nBp[ũgr6-kwЌYųXUbŕlX!\0Za\00\$z?d\r@ݪ\0\0xUzM\ri7\0oQ`\\Q@gQn}	m8>`d\0d8И\0ELo8Ш!ĠEK>8\0\n\n5\0H`k*ʥ\0qģx\rPf\0@@'=\0D?+PNh@-t;/\0o\0\n #1AB`-Y9c-H	\nQ7>9xT@#o\\ .^\0\0-gja\0nҐah @]T\\0߀]\0\r\r/\0 42N@vYRص)K✄@ny\"dpנ<@rpB(ODaNS9Cy\0SH\0PL	\0N~\0\0lPCR\0@\0y\0hP@}\0\0@?B> H\\ <M2\0\0@.\0I'K&|4P`:\0N)@\r;\0*T\\@@\0U\0h07@f)\"8(0J\0t@\r @t\0\0hP3BNp(`\0\0P\0q @\03P(gP`0@D{\08\0P`3\"xs` 	Clp@7\0b\0004\09K\0<\0 @JPk0\r#;B\08\r9\0 \00J@.u\0jn @\rG,H0	CӉHDlhp\0\0\0Q :\$\0  q-89 \0vtN C̉64\0^`RD'U\0Px\nn@U&@ 2+\0a*&\0D\0{HY?	;\0h( &x,pH!0Ff^\\p\0ـv\04\0M\0\0qp@e4jD5\0\n\0\0fLKL9\0	G`u\0\0p`[spkӝ\0.v`@<\0t@)_\r@%\0O\0\0\0xq.	`'y?\\.j'qZ@\0\0(\0P\0&JNGN\0\\tl\0Q	0n\0R\$hrD@Q\0\0 0`߼t`	 \0@v\0\0(@\0`\0\0_`i\r\0Q\0(`\0j1a\0@\08\0UpAPȰpGD\rp.X0h ;	f8\raHGy:@`\$E`X)K@\0>\0>|@i;\0`\0\0^W\0\0̎X`_)WYnVuֳr.njus#@\0006\0hx@7\\\08W@ '*Dr0P	\0001@(\0K(h\rP)@QQPHb봱UrSG\0h`\$f`g@>Za@\nu\0\0M rX48PTf>!@Es\0t\0zh8@!k.xi 9&x  !F\08h 1@F`X	\0&2\0H\"R8LP\0RxD	B@%\0h@ \0.IFN9?H@\0Nn	<7>\nu\"Юo>D#8`\0\0PP8&@\0H:Erru>(P &%|9@b#Tt0\0f\0*OeA	`.JR@i>\r0@g\0\0P9@9Ay(Y@Z6Yx\0 .dQ\0aH<~\0Di7o5T~B\r)X8X1mعsV\nk 9C`:mlD`<@qd(hb@z8\\HlaI/`JWB\rJ(}nyD,Gb>2ѿel{B\0\r@SF 6\0\0@\0\0 @Ws\0BL@1z\0(\0԰2\0\$%`.@0(tT0\r	'\0\0XX\0@B\0&\0[y\0@0@\0JJ\0\0  !\0Ji&a\0l34,`@H\0\0lq@.@/\0`3\0@31\0Z/@U\0H\0p\0(H\0'\$\0004\0\0h\n@:@=\0\0\0H@c\0003tq	@\0lޗ\0\0p%@\0v@8	`@)ap\0\0\0vp`	\0\0\\*	P%\0AQ\0\0Hp1;\09@\0\0\0D `d\0\0p\0001`\0 \n\0K\0<\0I`6h`\0	\0\nlXSʀ\0&\0\0@# <\0pg@\0R#;\0\r3|NQ \0/\0\0h\0Hx\0!FUB\0XpT\0&ڃDPҀB\0YP\0>\r\0=@,NPep@\0\0Z\0TB[0\00019\0\0`P\0p_W\0)%\0a\0JnB@\0 Hr\0871ȣ&@\$Gp\n\0]\0pKڀ @:\0K6\08S}`\0rF\0`@Z7\0\\\0c`\"jI\0RRd@\0\0001\0\0Rk1\0P	P	\n;\0d \n@@0 :I*iǕr\0K\0V eL@	+\0S((C4(d@uk\0*,RK#\n\0n5p,\0v&!0`\00i\0`p	G>	\0,\$nSdP@i\0\0648@\0HF@\08m@1Bf\0\0Y5\0\n4\0 ͯ\0\0D8@\n)i1eymWnBMcrUuܵ~+ĺ7\0H \n,\0jh \00\n/3\0l\rb|@*/xA\0'ABNgH\0#\$\0yS27\08,\0 ~*\0OU	/6\0h@h)%ദ;(\0a	 -&a%An)@#(]h-Agc\$BH &@.6=l '6\"@P\0zxP \0qr\0 g\0\0@\$x\0`uL\";h\0~@\n!@ hIJ\"@!h\0-;88#@\"hX.\0\$@#h\0L>@	\n%@\$h\0\\N	J&@%h\0l^	 &F@ \n@)@UcT`P	,@\\\0Gf#`k=nDXa 09D?	L@RFCD}Hc0oBA,,' &\$Γ\0,:\nd\0(2@\\|C}zpMÅwQh\r\03h09`Z̮'hX\r`,os)@/\0£TP@7@h\0:dR\r`/K1\r8d5@x?-	8\0wn;e\0FDCbd)B]3DZ['\r7`9ҧ}MK/(\0J!ưyH@u\0\0@!\0\rYd|*\0f*%hOV\0\\\0\$\0d \0g\0p1\0H#\0\0\0:\00@(P0`\0pj0\0`E|@0\0T@@<!?\0	rCL]\0Gp^`8s\0\0H\0	`.nb  +M(Z\0L3\0@\$D iv{Ln \0000b)4 \0p1+!Z\"\nV8\0`\0p\0A0@8\06ǀ\0>\0P% 5`]\0\0p\n``\0A\0(\0\0p\0\0:x@\0-ә\0uI98\0c@\0\0004\0@ HtR\\\08\0`S@vX\0(ː	`@jb:x \n\0O{hU\0\r@\0(@X\0>\$BX]#ˀ\"@0\0K(\0ч <\0F\0@0\rIP2\0E^pPk\0K  \r7\00a#+i=g\r	\0\0h?0\0R\0h4k`&S'\0 (  -\0\0\0L``\r\0q\0lq\n ?@v \0zaD\0hp ]+\0`x`/ʚW!24Ed`3\0Kj\0F\0\nPD\0\0H4 \rX\"\0\0PH9@g]4>Pk`\0\n	M9\0V%[^I!@L\0Pl@+@\0\0\0Dq\r\0C\n`@\$H\n\0\0#\r\0P;d;X#F3N\0#t!i`0@^\0l\$\r\0\0ffGx@(G\0ibNQX\0{,O\\\0\0\r(b\rd\r@\0003@h\0\0W&1]h!EwuA`2?Xz\$\$?'AAy\0Мy\0`\$A\0XnD8/p	G@&	U.`@`'@s@@;\0Vd`*MR\\P\np`:S\0:j  '\0?<='`4HP\0C,	s%\04M٬gcĠHPXP@#IA\0\0[k{\0~-Z-8@6NSٴ+0o%T @*p{dC\$\0Hu3I k8h`,I={\0B\0P \0,'|l\0\0/<\n\0+ P	 _Lj<P0\n\0n<(%P\0(*\"KgQDp@Ŋ\$\0\"ȢKDM%K3kDމ	A|L,0l	ʣ@'j\0<<	@n\0S\0\00*\0004eGxB2-F\0cpbᅀK60V>4*#9y8y\0r	\n0( )9M@\0/GWppƀ1}Ҍi\01`x\rP4m\0pp 6@n!\07pqF3a\0)Z董աt#\r@Op@<adS`=H:1|(pgRֲ%y\rS^ma|\nfiGr5GQ}ZqdS\n5\0<4@\0!\0kP@@c\0nPI\0c\00`8\0)\0ۊ@2'\0r\0 P@1\0MCa)Wg\0/0\n\0M\0\0 2P\r1H.BH3EhKP -\0[p}X{`Kp5o\0@M\0&\$h0`v\0\00drJ,\r1:]\0p-`3ȀTzJ\n<\0j0\rh8>3E#3~\r@mO(`2&\0o`<M\0\0ds@-[Lշ=\0P@\$Z,\0t \0s\0.`/) b\0mV`\rע\0\0 C)\0|Cqv@b\0\r\\3`\0%1#Pn8\0\0B^`dE0p\0H0 \0003\"`@9@\0	K@	%\\:7h	ph\0G\0\0 P@6X[\01HP`;D\0004\0A\0\$\0X	\0\0\0_8 :\0:\0t\0`\"\"D(P\r\0&\0\0-K@1M\0002Jظ0\0%'ZzPwp\"Ѐ7\0\0x\n`3\0W\0lXGD\$\0|\$\0cd\0V\"6H\n\n@rLE\0 LQtP>(N\0H8s]\0y\0/iP&@.} _ǀ^\0\0,\0`\n8\0xh8\rD\0TL(\0P0`\0]Pm0@\nB-b,\"a.KL`-\0E:ch:XG/h8P: )\0P\0F c\n\0\0\0002Zn\$\0&\08?'GSqss	CPBYuT.nQP֝C%\rT6uPB@u\0x;0\0\0\0j\0 `@\$@Un\0L\0p\r0\0l\08\0n2\n 7\0`Ev-H0\">OOt|XBu\0X@W+	A[l#E UaYG>iр36IL8E@,[\0F(\n	&B\0c^(P\r`4n\0F;1py\0@,BQ\0DB@QD:\"tEh7Dv\$@&\0HL*Hp	@K>d(\n n&R\r-Q\0Sx\n0@,[=:*u\0N|5o 'v\08	p`'\0JQ&A:I{I\n\0e0#BqLs<{\0 H\rF\0'\08,`Pp\0@W\0\0|<dL=\0gϚ+ -Rɾ|\n	jxm\0a@<)@P1DD\rb6fr&X	@'_jPg,@ott|)`A\0VP\\`k <@qU1.bu8m7f*۴؆z~QP@JgVasvux,qf0rx\06eQ(\0E\r,p\r3aBF3 @9\0\0HPd>@)T4@>\0jXp\n`!@4\0dHE\0:9H6\0b\\\n@1iQ\0;p@+@pxh\$<\0V`xPXiJ\0(	@/S\0>\0 {\0\n7(@@/\0)Ne@c\r 5xP)\0z\0x&gU\0\0+\0\\\07x2WX\034\r	@)\0DD\0b` @,\0VS@R\0@`@ZUl@\n *X3^\0J0 8E\0{}ODxđ`Lm /Ϛ\0\r̐0P\0\0HzqO P\"f<b)P\0`4I\0\028@8\0005KPpP	\$3EH7\rV#1VqGʛ1S\$cu\0d(Wexm3E\0\0.Gզ<T	\0\r@]WI |; eSr\0\0*{G`*\$\0\08v/\0d\0\0C\0icR\0<-5 /\0v\05n@+\0H\0lXL\0X\r\0<~ƀ\0)>T\0	\rO\0<8\0\0Ġ\0>kg7\0.\"\rd=x \rV	l\0`K\0000``?\0F[\$%lp	ňL\0\0\0e(\rP\n\0(5\0cNaO\0`\0\r@B\0\0p\\9=@jR@Ge	RJ[rJڨbܲz_m\0V\0`5[\0Zkp=Kf\0W\n\0001QL\n5\0V*+Vu@oz6msq=d\0E'\0Ѽ|l\0rHRh[\0~/;\$6 \0/\0o+x#0^ow1n[N@ 3#	\0+@	@W\0S\0@@5fx\r@=\0p\0a.ـ>\0k(	@N\0\0`Fz\0T\0 \0 Y\0\0t#`\n2QNj\n*ҩ\n&' RX8 OUAZDxF@.\0~?2X`(FC67x ^FTɷҋAt@4gD\r'3?E5.mȀ  \0+T<Z\\ @n(,bbT80AQZ\0\08 \"0L\0Բ`˸	p:\0ZjD 9\0J.\0o2V%%9\0miF3j\0Tƨ\rg 6Cp}Pkx\0t\0\rv.laCdbB4hD,\\<!0Bh	؛y\n@eռx,k@V\0\0v\0e\\\0e`\0p\0\0BQ8KWXx	;0`w1v	T\0\08\0z&\0c@\0N:XP@@>8\\fUD-ce\n@ #\000\0xh	@\0m\0B(!\0Wz J\0=\0W5@\0\0D\0Tp\0k؄5\"Uhrh 71	\0V\\p\r\0C\$\0004p\nR\0\0.\0` \0?\0003Tp`h v \0\0@?@\"l3Vvh\0@\r)*\0p4\0	\0\0r\0D,P7\0\0g\\\0NC@|\0007\0\${ 6`;@T#c\0\0@@\$|vPAF8x\0DqQe#!e_i\\E<\n\08!{NJA`5z`\nlX0j/?@@\0Br \rV \0P%`xP\\^\$ix\n>9@  JZ@L\0w\npVF\0'Hh0 )\0d\0NBIEK-da\0+\0>	T.h\ni:@@UJCR @\0Tt0m*S\0>` `	(f(h(@ώ2\0rR4\0\0 \0@{vL\0`Ij-\0L]`+΀x\nɁ\0L8  -,ˀ3BP;e\r+\0RI\0\0)R\0l}\0}\"cցc6&`7bW+UjP@'\0'>\0+`)\0Ig\$Kr[\0hcu#GMlMW%ŶLeA>Lƨ\$[nm`\r\rUKH_MaRdP\0@#nHI\$5)a\0(`	\0006EH4Rh1@sGd@Hq5J \"jQQ T` 9*U,jC``Bπ^4\\,	\$Y`1%~|q{D4\n~h`dB_\\V9\0i:m& ?\0a( y %@D\0nx\n3C~#9+d		\nE@[Il	бfu\0<+0zpx2=\0\0=b\0Qia\0`\0e>Π:m|0 2b%׫͙n(Ȁ9XS@N\0\rU_ʼ\0fAL7@akY\r1d@p\0z`ik ,\0R9Iqp\0XFh8\r4_`6s0/Ty\"cTQBf+XgMu86Wڅ=޼moV7bD`Xmqt=\0P;\0tP,\n`.\0X1\00\n@@<	hʊ@2R(|I(\0h&So\0z\0Ɗ0	P`\0005%ˮlX@ @\0o\n`\0O\0e\0)I\0f1 \0@I\0:qM\0AW+@D\0p\0886\r \0,@85Y@\r@@P@=b\0>#`!L\0rpBP#Sـ\r\0S\0\0Kk1?\0\0W,6\\p4\0dO/qp\0\0.\0l3B\0\0+@k%7<%Դ7\0h1=+	\$G5\0K&ؠ\0 yL\0D+@\0Nm\0c\0\0a\"P\r5`\$E\"\0z\0Us	 	6R\0LX\0JF(̨\0\"{@>.\09\0j\0*vH\0@s\0\$*aT`+\09\0\\sh'\0^\0 (\0Jqt\0+7P\n a\0	\0\\\n]⦎@\0|\0\0:aD8\0B	\0\0\"d-XS@%\0000D7X\nq^s,\0`8Vq(/r\0`j\0/ 4H 	\0-B,?i`(@\0m( \np@5Et\n@@Hڞz9\"P\0m\0ad#4F\0%5RXf\0	L\0^ 	 \r,NO-̴fP @c\$`)S3@[OEdk۲N\0t\09 \0\0&\n\0_|\0 >'bx\rB)\nFKDk	XQiB85UY`QB\0\0d\r&D[\04\0 ]*`\$ʀ\0\0m~u6+ۂnj6+ۊn2\0007XntsЄE\nN\$tF\0\0HPX-t\0\0n\rP\n\"@)\n *4¬\nl\"b7|`dlXXPa҅~qfiL\ra0F(T\01(\0^S	0i0 ByAV.0piz#X9Ac\$%0iX	 &.6T)3\0+nP %\0-\00rbX\0\0P@'G28\nPm\09nL^ƚ`!.e0QI1\0P@cTr	pI[x 94i*-@H%H=TcI@9s0	.㘡\0K@\n/\0Tg~t(c\0Z@ 9@_pXҒll`h8b\0b\0Dy 3@v\09@p\rRuKd7\\-<7+p@oH6Ð`9\0蛔dY~d%\0\r652\0~S.mpn6#@y\0\0pg\0v#f:`v\\^\0wH\0@*X7UH]@!g\0\0\0p@4\0ZP@1I`\0\0\0\0i\0\00 *\0	\0\0# c\0N<\r@#|\0g\0:dn\0\rYP]5\0000lD@/Tp I[\0I\0\0h@Fl\0p\0<N\"h4OFXw\0\0)@?\0;Ｔ_\0W0\n@L4\0\0@\r	 0|x\n =ͻ6#\0[8@S\0m\rX\0F\0	f\0w\0/40\0a 1\0S]3\0\r@\0k\0\0.`\0H1\0ִh@3@I\0\nU`&'\0pI4\0 5 \0[J(	\0\0e-VԠ,8\00K۪\0&_X	CmpM\0T\0\0\0`\0>hW/@4+Vxe	\0y< C`+X\0NZHF\0D\0)*\0\0#`h\0ϾRP PVRF.,/@T5\0B\0\nP)\0+\"*kR\0\0yd1N\0I s\$8Pfx\0P/z\0,BmXf(\nxӪ@q\0G\0\0y\0<\0\"ww\0PlkP@@\0\0ϐ0%@A7\0#\0\0喀4@ 0Wa[\0\04<\0[΀	\0h=?cG2%	850[P+}@``0郀N`bBP\0,R\npTـ&5\0m\0?n\r_\0(>8KH1p\"R1,T+z-/gNM \0\0(\0N\\<Qc\0*X-amuf	_NV,@\n\0005OlaN\0h\n\0007P\0{[l2^,Ff{,E`XǓkoTI>	C\0(~	q{rʌ3\0҂<\\`\r\08I\0SDø@r-SN;F	 F@<*` @&\0CQcLu@;\0O|a\\	*@Of>P\n\0*Pd\n0\"KJjlD1`<	8Iwhyo\0rTh~\0f\0.(@c\0\0lk \n\n0\0T5'P1`@Ph>%\"\$sO#m\n5@LSe0l	\0'0g\0 H2ǢYEp\r\$#i+.H	U (.\0\nă1\0r\\άq20§\n/<hp<z\0x\0\ra	PdҼN]V	I_&-hK7Po l\0L8\02\0FA\0\n7L^\\[\0H@4[wf(a(3(\0\0h\03/f\0If|0 !\0001*QpL\0f7\0\nB\0h\0 \\\0\0-A^;u%\0㮾\nq\0Qej@\0@\0T\0L5<\rG+0T;\0@	\0q70\0T\0 /\082\0Dd`\r @Y\0M\n`x\0x\0004\0U(	\n@+\0/Q\0000\0dFlґr @9F\0l5@o\0j \0\0=l\09Fʸ	\0ڠ@\0x	\r\0\0&uB\08\0h`\r@\06\0%};@\06\0p\05[H7@R*DN4&Ũ\0\r\0k8p	\0R\0]Vľ\0X@\rFCg\0H U*\09p:M]\08уE\0h<\0	`'퀵t	\r\0@Ƨ0\rQ`{6P!\0e2\0Ilh\nS0M\088@@@)\0|d\0H\rm&n/(5O1\0q\0@m\0BZ7(\0{^\$c\r \r@m,0m1\$Ӱ\0\0` F4\r\0\0Li0\$˲@4\0[.p276P;\0jL	0\$zg\0l\\ (\0`Δ\0000v\0E~p`(0/#IB.ud8\r\0e|pPx<Q ذ`ǥ\0F\n@6,Z;@: }UzY^h#W]Y`(W+97\$!y!p\$ŘP!:QI\0n\0,	\0&\06af@R\\^Ȁb	54\0CTi\rSj@o`e\0`f\0!\0A&n\0f\0\0vc3rS\n'Т@@qJàL@`ENf(*Ġi\0U	e(;o`07s\0{3.\".\0MR9\030-M\0]\0X`*߀W)\0j7&ܕ) '&@\0UH\n7踸?\nc\"1\0y(i@\0/ЉRhtBIIQ\r!ز\r:	80Ѯ.mPiZv\\\rP0J\0607\0_Ȣzc /yJT.`0VNC `4@y\0:j|<\08n8UǺt:ƭbJu)#r)]Ch?7\",&r5&1%\0]\0W`Vg)tP`;uxd` @QDװ\0`u\0zȗ!I\0\0d\0s\0	i0\0џJH\n@B6\0004\0\0i@\n\0\0lXWk2Zu\0-@\ny*8`6Z[\0t\0(@	ۉ6\0`ޠ[\0E\0 \0\0Xp\0(\r\0t	`Hk  \r\09*s\0 f\0\0Տ0;\0R\0006@\0@\0007U?-\0|\$?\0\$\0Q2P7mu4T\0P\nw	@VG\0t`\r ʀ?\0@\0u\r'\0\0006@S\0\00068pyDպ.\0M`Q3\0|\0'9oYc&\0=C̀eo@ZhcؘHg\0LΐFe;,s(@3\0:m8\0i\06B0 @A\0d\003\0\0pBn >\0+O3e\0007'p\0@aXp	ujdPQ \0L\0\r\0	+\0\0\\@xEp\0h{Q@ԣD=P0\0\0e\0\0\$\0q&,x	@:@l\0rJ\"<i@ \0000Hf8U\0 @I\0Xt@!g\0002JRt )N\0\0:\$2(\0.ʗYt\\}9M\n\0Ye[rE\\`l\0DHõ\0@\0p0\0'\0E4\$P6\$jX\08JG@{Z\0O;\0000jzj	\0mY`7Z(p,v\0d\ry_p\r\0\0m\$:/nc\$X	4n6cs x9ŐBTYVj\r\0*\0ͱ_#@\n \rA\0:HEV\0kGP\0DQ8@A0`\0!\0(:.AX)C&g1 I\n~9K@\0s\0j\0F\09[(\$ݶs20	-Rr_,eqv\$P0\0007@YPN/UN  ,@,\0i\"@ @&;\0Jd\nWQxGϪ07>f>+]+SL1V|aÀ-t1b8Ā<CF%wpበ9_\08(H3>k(:'\0@_@2\0DV^xp!\0OL\$ Ϡ01-xx:.<&C	6x,wi`Q~\05\rP:\0f:ppY{z\0y\0ʍހ<ͼ=qƠH]t.`H\r+Ϡ٧Sҽg:nVw	NQ6\0\r\0	V\nu5C\0;@QHHK0H(`@qH(\n)\0.yhj[ S\0lHd	\0P	URl\0W\$\0\08X \"q\0v\0\0d͠)\0ZaXǮ\0`e',-5\00\0\0]\0@`;\0sdh\0RF\0 ANi\0;\0	L-fA\0x @\0E\0KtL=M{@x\0K\0p\0-4Cp;fr!q\0\0X\0#G@\0004Bh\0n\01\r/d\0p@r+\"`D`.HVh\0006+?5\0;\0\0OA\0ibpw}ZhxU06Kr\0\0 H!bE(\0c\0\0P@0FG6E]rh@72\0& 	-J\0Bm} 18@#zB8`L2\0{\0 `\rD`Y\0\00003AK*kXpE\0\"z{=r\0s@@U\0Bit\0+\0C28\0`.\0005R\0\0V\0<X\0d9D\07=t\068\0XPWɃ29\0}PNo@)@xw	%8+ *`C\0n`s'QOAW>Z&@:\0d  @32Bnqy1=\\G p%0p/\0\0`(\04!zhP68&C\0ld`\r851Ttds	۪,0lT71Q*|E71V2|A/:nz!eۢ7<%P@9D\0	6Op	&lA_\00*\0QcAUvן'	\r:m%D\0|\"˨J\0&B\\\00\nc\0*x/y8KjZ_j@\r5*޳!@<~0!\$oHc\$(\\\"\0l\0\0`\n\0S\0y\0\nю04]\0|ϼErQ:dUkXPD\n+#@|\nC-7q-Ua\0es\r!\0i\rt6\0j@xzds&`|;\0X 3w\02,\n\0{.?(hb\0/\$ڞ^Ju\n(6@_\0VRx .@vw&Te9l kb5j`\nZ҅a	A.t꩜sU\0D x)ԕ34;fidYLl~\0=ʀ)L31Y-ijVE<uZt8\0\0%\0T@T\0000\0Z\0NY0M\0\rwB)QP:\0,hx@#H+\\46^XZXH \0bZ\0#;` \r\0\0c0t?Z|[\0ER0;ھiL2\$\0ۉqg`-*\rX8\n\0]P&\0X\0`\n@a\0\0@r4f`6\0aotp`e\0004VK\0\"J\0\0d:@!@k1\0\02D >#T!\0,\0\0p\0H\n`%@D\000@̡4I \0t]\0\r,\0MyW2\0\0[1)\0\0&2p<vPxFQX\0001oB(p@0g\0xH8\0SɁ*@\"	\0f\0L`#@6D;@Bu0+ZC\0v  \0\0\0-\0\0C),8jW\0-Ν\0@p\06\r	i\0^\0E\0(p@4\0:ϐ@H_6\08og,\0h	:\0Hvn@\n0@,@a}(@u\0l \0zq\0\0D8 \n\$+*NqDSH9*F\0H(P \0Y\0vkj4饀GHHl\n@+@g\0X`\08T !\0o~5`^XH \0	<PbE\0##Fэ\0007\0\$h\nВwQ\0-.W\0	[T\0=>W݈\r	YjC\ncഖaEtXJ`0yt\0)\0\$L\0~\0(8p;<l``s\rG8q\\s?9r\\s-_9s\\s=:t\\sMΟG:u\\s]ο;v\\smߝ;w\\s}<x\\s\0\0");}
        private function printSvgDatabase($isReturn = false) {$svg = '<svg id="printSvgDatabase" class="svg-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><rect x="0" fill="none" width="20" height="20"/><g><path d="M10 6c3.9 0 7-.9 7-2s-3.1-2-7-2-7 .9-7 2 3.1 2 7 2zm0 9c-3.9 0-7-.9-7-2v3c0 1.1 3.1 2 7 2s7-.9 7-2v-3c0 1.1-3.1 2-7 2zm0-4c-3.9 0-7-.9-7-2v3c0 1.1 3.1 2 7 2s7-.9 7-2V9c0 1.1-3.1 2-7 2zm0-4c-3.9 0-7-.9-7-2v3c0 1.1 3.1 2 7 2s7-.9 7-2V5c0 1.1-3.1 2-7 2z"/></g></svg>'; if ($isReturn) {return $svg;} echo $svg; }
        private function printSvgMuplugin($isReturn = false) {$svg = '<svg id="printSvgMuplugin" class="svg-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><rect x="0" fill="none" width="20" height="20"/><g><path d="M13.6 5.1l-3.1 3.1 1.8 1.8 3.1-3.1c.3-.3.2-1-.3-1.5s-1.1-.6-1.5-.3zm.3-4.8c-.7-.4-9.8 7.3-9.8 7.3S.6 5.5.1 5.9c-.5.4 4 5 4 5S14.6.6 13.9.3zm5.5 9.3c-.5-.5-1.2-.6-1.5-.3l-3.1 3.1 1.8 1.8 3.1-3.2c.3-.2.2-.9-.3-1.4zm-11.7-1c-.7.7-1.1 2.7-1.1 3.8v3.8l-1.2 1.2c-.6.6-.6 1.5 0 2.1s1.5.6 2.1 0l1.2-1.2h3.8c1.2 0 3-.4 3.7-1.1l1.2-.8-8.9-8.9-.8 1.1z"/></g></svg>'; if ($isReturn) {return $svg;} echo $svg; }
        private function printSvgPlugin($isReturn = false) {$svg = '<svg id="printSvgPlugin" class="svg-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><rect x="0" fill="none" width="20" height="20"/><g><path d="M13.11 4.36L9.87 7.6 8 5.73l3.24-3.24c.35-.34 1.05-.2 1.56.32.52.51.66 1.21.31 1.55zm-8 1.77l.91-1.12 9.01 9.01-1.19.84c-.71.71-2.63 1.16-3.82 1.16H6.14L4.9 17.26c-.59.59-1.54.59-2.12 0-.59-.58-.59-1.53 0-2.12l1.24-1.24v-3.88c0-1.13.4-3.19 1.09-3.89zm7.26 3.97l3.24-3.24c.34-.35 1.04-.21 1.55.31.52.51.66 1.21.31 1.55l-3.24 3.25z"/></g></svg>'; if ($isReturn) {return $svg;} echo $svg; }
        private function printSvgTheme($isReturn = false) {$svg = '<svg id="printSvgTheme" class="svg-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><rect x="0" fill="none" width="20" height="20"/><g><path d="M2 2h5v11H2V2zm6 0h5v5H8V2zm6 0h4v16h-4V2zM8 8h5v5H8V8zm-6 6h11v4H2v-4z"/></g></svg>'; if ($isReturn) {return $svg;} echo $svg; }
        private function printSvgUpload($isReturn = false) {$svg = '<svg id="printSvgUpload" class="svg-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><rect x="0" fill="none" width="20" height="20"/><g><path d="M4 15v-3H2V2h12v3h2v3h2v10H6v-3H4zm7-12c-1.1 0-2 .9-2 2h4c0-1.1-.89-2-2-2zm-7 8V6H3v5h1zm7-3h4c0-1.1-.89-2-2-2-1.1 0-2 .9-2 2zm-5 6V9H5v5h1zm9-1c1.1 0 2-.89 2-2 0-1.1-.9-2-2-2s-2 .9-2 2c0 1.11.9 2 2 2zm2 4v-2c-5 0-5-3-10-3v5h10z"/></g></svg>'; if ($isReturn) {return $svg;} echo $svg; }
        private function printSvgWpcontent($isReturn = false) {$svg = '<svg id="printSvgWpcontent" class="svg-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><rect x="0" fill="none" width="20" height="20"/><g><path d="M18 12h-2.18c-.17.7-.44 1.35-.81 1.93l1.54 1.54-2.1 2.1-1.54-1.54c-.58.36-1.23.63-1.91.79V19H8v-2.18c-.68-.16-1.33-.43-1.91-.79l-1.54 1.54-2.12-2.12 1.54-1.54c-.36-.58-.63-1.23-.79-1.91H1V9.03h2.17c.16-.7.44-1.35.8-1.94L2.43 5.55l2.1-2.1 1.54 1.54c.58-.37 1.24-.64 1.93-.81V2h3v2.18c.68.16 1.33.43 1.91.79l1.54-1.54 2.12 2.12-1.54 1.54c.36.59.64 1.24.8 1.94H18V12zm-8.5 1.5c1.66 0 3-1.34 3-3s-1.34-3-3-3-3 1.34-3 3 1.34 3 3 3z"/></g></svg>'; if ($isReturn) {return $svg;} echo $svg; }
        private function printSvgInfoOutline($isReturn = false) {$svg = '<svg id="printSvgInfoOutline" class="svg-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><rect x="0" fill="none" width="20" height="20"/><g><path d="M9 15h2V9H9v6zm1-10c-.5 0-1 .5-1 1s.5 1 1 1 1-.5 1-1-.5-1-1-1zm0-4c-5 0-9 4-9 9s4 9 9 9 9-4 9-9-4-9-9-9zm0 16c-3.9 0-7-3.1-7-7s3.1-7 7-7 7 3.1 7 7-3.1 7-7 7z"/></g></svg>'; if ($isReturn) {return $svg;} echo $svg; }
        private function printSvgUpdateAlt($isReturn = false) {$svg = '<svg id="printSvgUpdateAlt" class="svg-icon" xmlns="http://www.w3.org/2000/svg" width="1em" height="1em" viewBox="0 0 20 20"><path fill="currentColor" d="M5.7 9c.4-2 2.2-3.5 4.3-3.5c1.5 0 2.7.7 3.5 1.8l1.7-2C14 3.9 12.1 3 10 3C6.5 3 3.6 5.6 3.1 9H1l3.5 4L8 9zm9.8-2L12 11h2.3c-.5 2-2.2 3.5-4.3 3.5c-1.5 0-2.7-.7-3.5-1.8l-1.7 1.9C6 16.1 7.9 17 10 17c3.5 0 6.4-2.6 6.9-6H19z"/></svg>'; if ($isReturn) {return $svg;} echo $svg; }
        private function pageActivate() {
?>
<?php
$data = $this->useHandle->activate->getData(); ?>
<h2>Restorer Security</h2>
<p>
    Request activation for this site using the license key to use the WP Staging Restore
</p>
<button class="action" data-action="activate-license">Activate License</button>
<?php $this->useHandle->view->printProcessLoader();?>
<div id="wpstg-restorer-console"></div>
<?php }
        private function pageBackupContent() {
?>
<?php
$backupIndex = $this->meta->dataPost['backupIndex']; $data = $this->useHandle->extractor->getBackupFiles($backupIndex); $metaData = (object)$this->useHandle->extractor->readBackupMetaDataFile($data['metaFile']); $extractPath = $this->useHandle->extractor->getDefaultExtractPath(); $totalFiles = !empty($metaData->totalFiles) ? (int)$metaData->totalFiles : 0; $sortbyOption = $this->useHandle->view->partSelection($metaData); if (empty($metaData->databaseFile)) { $metaData->databaseFile = ''; } ?>
<div id="backup-extract">
    <ul class="breadcrumb">
        <li><a href="<?php $this->useHandle->view->printAppFile();?>" data-action="page-main">Home</a></li>
        <li><a href="<?php $this->useHandle->view->printAppFile();?>" data-page="extract" data-index="<?php echo $this->kernel->escapeString($backupIndex);?>">Extract Backup</a></li>
        <li>View Backup</li>
        <?php if (!$data['isMultisite']) : ?>
        <li><a href="<?php $this->useHandle->view->printAppFile();?>" data-page="restore" data-index="<?php echo $this->kernel->escapeString($backupIndex);?>">Restore Backup</a></li>
        <?php endif;?>
    </ul>
    <h3>View Backup</h3>
    <div id="extract-info" class="info-block">
        <div>
            <label>Backup Name</label>
            <span><?php echo $this->kernel->escapeString(basename($metaData->name));?></span>
        </div>
        <div>
            <label>Backup File</label>
            <span><?php echo $this->kernel->escapeString(basename($data['path']));?></span>
        </div>
        <div>
            <label>Backup Size</label>
            <span><?php echo $this->kernel->escapeString($this->kernel->sizeFormat($metaData->backupSize));?></span>
            <span> ( <?php echo $this->kernel->escapeString($totalFiles) . " " . ( $totalFiles > 1 ? "Files" : "File");?> )</span>
        </div>
        <div id="root-path">
            <label>Root Path</label>
            <span><?php echo $this->kernel->escapeString($this->kernel->normalizePath($this->meta->rootPath));?></span>
        </div>
    </div>
    <div id="extract-block" class="action-block hide">
        <h3 id="extract-to-path">Extract to Directory Path</h3>
        <input type="text" name="extract-path" id="extract-path" value="<?php echo $this->kernel->escapeString($extractPath);?>" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
        <label for="extract-path-overwrite" class="checkbox">
            <input type="checkbox" id="extract-path-overwrite" name="extract-path-overwrite" value="1">
            <span>Overwrite directory
                <span data-tooltip="Check this option to completely remove the extract directory before the extraction process"><?php $this->useHandle->view->printSvgInfoOutline();?></span>
            </span>
        </label>
        <?php if ($metaData->isExportingDatabase) :?>
        <label id="normalize-db-file-block" for="normalize-db-file" class="checkbox">
            <input type="checkbox" id="normalize-db-file" name="normalize-db-file" value="1" checked>
            <span>
                Normalize database file
                <span data-tooltip="Check this option to normalize the Database file by replacing the WPSTG tag with actual data">
                    <?php $this->useHandle->view->printSvgInfoOutline();?>
                </span>
            </span>
        </label>
        <?php endif;?>
        <input type="hidden" name="backupfile-path" value="<?php echo $this->kernel->escapeString($data['path']);?>">
        <input type="hidden" name="total-files" value="<?php echo (int)$totalFiles;?>">
        <button class="action" data-action="extract">Extract</button>
        <button class="action-close" data-action="extract-block-close">Close</button>
        <button class="action-cancel hide" data-action="extract-cancel">Cancel</button>
        <button class="action-green hide" data-action="extract-retry">Retry</button>
        <?php $this->useHandle->view->printProcessLoader();?>
    </div>
    <div id="wpstg-restorer-console"></div>
    <div id="extract-list" class="action-block">
        <?php if ($totalFiles > 1) :?>
        <div id="paging">
            <div>
                <input type="text" id="index-filter" name="index-filter" value="" placeholder="filename">
                <button class="action" data-action="filter">Search</button>
                <button class="action-cancel hide" data-action="filter-reset">Reset</button>
            </div>
            <div>
                <?php if (!empty($sortbyOption)) :?>
                <select id="index-sortby" name="index-sortby" data-action="sortby">
                    <?php foreach ($sortbyOption as $sortbyId => $sortbyName) :?>
                    <option value="<?php echo $this->kernel->escapeString($sortbyId);?>"><?php echo $this->kernel->escapeString($sortbyName);?></option>
                    <?php endforeach;?>
                </select>
                <?php endif;?>
                <button class="action" data-action="paging-prev" data-value="0" disabled>Prev</button>
                <button class="action" data-action="paging-next" data-value="1">Next</button>
                <input type="hidden" id="index-total" name="index-total" value="<?php echo (int)$totalFiles;?>">
                <input type="hidden" id="index-page-total" name="index-page-total" value="0">
            </div>
        </div>
        <div id="paging-bottom">
            <div class="left"></div>
            <div class="right"></div>
        </div>
        <?php endif;?>
        <div id="paging-table">
            <table>
                <thead>
                    <tr>
                        <th><input type="checkbox" value=""></th>
                        <th>File</th>
                        <th>Size</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
$pagingData = ''; foreach ($this->useHandle->view->backupPaging($data['indexFile'], $metaData->databaseFile, $pagingData) as $data) : $hasSqlFile = !empty($data[4]) ? " data-is-sqlfile=1" : ""; ?>
                    <tr>
                        <td>
                            <input type="checkbox" name="item[]" value="<?php echo (int)$data[0];?>" <?php echo $this->kernel->escapeString($hasSqlFile);?>>
                        </td>
                        <td>
                            <p title="<?php echo $this->kernel->escapeString($data[1]);?>"><?php echo $this->kernel->escapeString($data[1]);?></p>
                        </td>
                        <td>
                            <p title="<?php echo $this->kernel->sizeFormat($data[3]);?>"><?php echo $this->kernel->sizeFormat($data[3]);?></p>
                        </td>
                    </tr>
                    <?php endforeach;?>
                </tbody>
            </table>
            <?php
if (is_object($pagingData)) { $code = '<script id="paging-data" type="application/json">'; $code .= json_encode($pagingData); $code .= '</script>'; echo $code; } ?>
        </div>
    </div>
</div>
<?php }
        private function pageBackupExtract() {
?>
<?php
$backupIndex = $this->meta->dataPost['backupIndex']; $data = $this->useHandle->extractor->getBackupFiles($backupIndex); $metaData = (object)$this->useHandle->extractor->readBackupMetaDataFile($data['metaFile']); $extractPath = $this->useHandle->extractor->getDefaultExtractPath(); $totalFiles = !empty($metaData->totalFiles) ? $metaData->totalFiles : 0; $sortbyOption = $this->useHandle->view->partSelection($metaData); if (empty($metaData->databaseFile)) { $metaData->databaseFile = ''; } $extractData = [ 'total-files' => $totalFiles, 'backupfile-path' => $data['path'], 'dbfile-path' => $metaData->databaseFile ]; ?>
<div id="backup-extract">
    <ul class="breadcrumb">
        <li><a href="<?php $this->useHandle->view->printAppFile();?>" data-action="page-main">Home</a></li>
        <li>Extract Backup</li>
        <li><a href="<?php $this->useHandle->view->printAppFile();?>" data-page="content" data-index="<?php echo $this->kernel->escapeString($backupIndex);?>">View Backup</a></li>
        <?php if (!$data['isMultisite']) : ?>
        <li><a href="<?php $this->useHandle->view->printAppFile();?>" data-page="restore" data-index="<?php echo $this->kernel->escapeString($backupIndex);?>">Restore Backup</a></li>
        <?php endif;?>
    </ul>
    <h3>Extract Backup</h3>
    <div class="info-block">
        <div>
            <label>Backup Name</label>
            <span><?php echo $this->kernel->escapeString(basename($metaData->name));?></span>
        </div>
        <div>
            <label>Backup File</label>
            <span><?php echo $this->kernel->escapeString(basename($data['path']));?></span>
            <?php foreach ($extractData as $key => $value) :?>
            <input type="hidden" name="<?php echo $this->kernel->escapeString($key);?>" id="<?php echo $this->kernel->escapeString($key);?>" value="<?php echo $this->kernel->escapeString($value);?>">
            <?php endforeach;?>
        </div>
        <div>
            <label>Backup Size</label>
            <span><?php echo $this->kernel->escapeString($this->kernel->sizeFormat($metaData->backupSize));?></span>
            <span> ( <?php echo $this->kernel->escapeString($totalFiles) . " " . ( $totalFiles > 1 ? "Files" : "File");?> )</span>
        </div>
        <div>
            <label>Root Path</label>
            <span><?php echo $this->kernel->escapeString($this->kernel->normalizePath($this->meta->rootPath));?></span>
        </div>
    </div>
    <div class="action-block">
        <h3>Extract to Directory Path</h3>
        <div id="action-option">
            <?php if (!empty($sortbyOption)) :?>
            <select id="index-sortby" name="index-sortby">
                <?php foreach ($sortbyOption as $sortbyId => $sortbyName) :?>
                <option value="<?php echo $this->kernel->escapeString($sortbyId);?>"><?php echo $this->kernel->escapeString($sortbyName);?></option>
                <?php endforeach;?>
            </select>
            <?php endif;?>
            <input type="text" name="extract-path" id="extract-path" value="<?php echo $this->kernel->escapeString($extractPath);?>" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
        </div>
        <label for="extract-path-overwrite" class="checkbox">
            <input type="checkbox" id="extract-path-overwrite" name="extract-path-overwrite" value="1">
            <span>
                Overwrite directory
                <span data-tooltip="Check this option to completely remove the extract directory before the extraction process">
                    <?php $this->useHandle->view->printSvgInfoOutline();?>
                </span>
            </span>
        </label>
        <?php if ($metaData->isExportingDatabase) :?>
        <label id="normalize-db-file-block" for="normalize-db-file" class="checkbox">
            <input type="checkbox" id="normalize-db-file" name="normalize-db-file" value="1" checked>
            <span>
                Normalize database file
                <span data-tooltip="Check this option to normalize the Database file by replacing the WPSTG tag with actual data">
                    <?php $this->useHandle->view->printSvgInfoOutline();?>
                </span>
            </span>
        </label>
        <?php endif;?>
        <button class="action" data-action="extract">Extract</button>
        <button class="action-close" data-action="page-main">Close</button>
        <button class="action-cancel hide" data-action="extract-cancel">Cancel</button>
        <button class="action-green hide" data-action="extract-retry">Retry</button>
        <?php $this->useHandle->view->printProcessLoader();?>
    </div>
</div>
<div id="wpstg-restorer-console"></div>
<?php }
        private function pageBackupList() {
?>
<h2 class="backup-list-header">
    Available Backups <span data-tooltip="Rescan available backups" data-action="reload-backup-list"><?php $this->useHandle->view->printSvgUpdateAlt();?></span>
</h2>
<?php
$listBackup = $this->useHandle->extractor->getBackupFiles(); if (empty($listBackup)) : ?>
<p>No backup available</p>
    <?php
endif; foreach ($listBackup as $fileIndex => $arrData) : if (!$arrData['isValid']) { continue; } $metaData = (object)$this->useHandle->extractor->readBackupMetaDataFile($arrData['metaFile']); if (!isset($metaData->id)) { continue; } $backupTypeTitle = !empty($arrData['isMultisite']) ? 'Multi Site' : 'Single Site'; if (!empty($arrData['backupType'])) { switch ($arrData['backupType']) { case 'multi': $backupTypeTitle = 'Entire Network'; break; case 'main-network-site': $backupTypeTitle = 'Main Network Site'; break; case 'network-subsite': $backupTypeTitle = 'Network Subsite'; break; case 'single': $backupTypeTitle = 'Single Site'; break; default: $backupTypeTitle = 'Unknown Backup Type'; } } ?>
<div class="backuplist" data-backup-id="<?php echo $this->kernel->escapeString($metaData->id);?>">
    <main>
        <div>
            <label>Name</label>
            <span class="name">
                <?php echo $this->kernel->escapeString($metaData->name);?>
            </span>
        </div>
        <div>
            <label>Type</label>
            <span>
                <?php echo $this->kernel->escapeString($backupTypeTitle);?>
            </span>
        </div>
        <div>
            <label>Created On</label>
            <span>
                <?php
$dateCreated = (new \DateTime())->setTimestamp($metaData->dateCreated); echo $this->kernel->escapeString($this->kernel->setDateTime($dateCreated)); ?>
            </span>
        </div>
        <div>
            <label>Size</label>
            <span>
                <?php echo $this->kernel->escapeString($this->kernel->sizeFormat($metaData->backupSize));?>
            </span>
        </div>
        <div>
            <label>Contains</label>
            <span class="backup-list-tooltip">
                <?php
if ($metaData->isExportingDatabase) { $sqlSize = !empty($metaData->indexPartSize['sqlSize']) ? $metaData->indexPartSize['sqlSize'] : 0; $toolTip = 'Database' . ($sqlSize ? '&#xa;Size: ' . $this->kernel->sizeFormat($sqlSize) : ''); echo '<div data-tooltip="' . $this->useHandle->view->escapeTooltip($toolTip) . '">' . $this->useHandle->view->printSvgDatabase(true) . '</div>'; } if ($metaData->isExportingPlugins) { $pluginsSize = !empty($metaData->indexPartSize['pluginsSize']) ? $metaData->indexPartSize['pluginsSize'] : 0; $toolTip = 'Plugins' . ($pluginsSize ? '&#xa;Size: ' . $this->kernel->sizeFormat($pluginsSize) : ''); echo '<div data-tooltip="' . $this->useHandle->view->escapeTooltip($toolTip) . '">' . $this->useHandle->view->printSvgPlugin(true) . '</div>'; } if ($metaData->isExportingMuPlugins) { $muPluginsSize = !empty($metaData->indexPartSize['mupluginsSize']) ? $metaData->indexPartSize['mupluginsSize'] : 0; $toolTip = 'Must-Use Plugins' . ($muPluginsSize ? '&#xa;Size: ' . $this->kernel->sizeFormat($muPluginsSize) : ''); echo '<div data-tooltip="' . $this->useHandle->view->escapeTooltip($toolTip) . '">' . $this->useHandle->view->printSvgMuplugin(true) . '</div>'; } if ($metaData->isExportingThemes) { $themesSize = !empty($metaData->indexPartSize['themesSize']) ? $metaData->indexPartSize['themesSize'] : 0; $toolTip = 'Themes' . ($themesSize ? '&#xa;Size: ' . $this->kernel->sizeFormat($themesSize) : ''); echo '<div data-tooltip="' . $this->useHandle->view->escapeTooltip($toolTip) . '">' . $this->useHandle->view->printSvgTheme(true) . '</div>'; } if ($metaData->isExportingUploads) { $uploadsSize = !empty($metaData->indexPartSize['uploadsSize']) ? $metaData->indexPartSize['uploadsSize'] : 0; $toolTip = 'Uploads' . ($uploadsSize ? '&#xa;Size: ' . $this->kernel->sizeFormat($uploadsSize) : ''); echo '<div data-tooltip="' . $this->useHandle->view->escapeTooltip($toolTip) . '">' . $this->useHandle->view->printSvgUpload(true) . '</div>'; } if ($metaData->isExportingOtherWpContentFiles) { $wpcontentSize = !empty($metaData->indexPartSize['wpcontentSize']) ? $metaData->indexPartSize['wpcontentSize'] : 0; $toolTip = 'Other files in wp-content' . ($wpcontentSize ? '&#xa;Size: ' . $this->kernel->sizeFormat($wpcontentSize) : ''); echo '<div data-tooltip="' . $this->useHandle->view->escapeTooltip($toolTip) . '">' . $this->useHandle->view->printSvgWpcontent(true) . '</div>'; } ?>
            </span>
        </div>
    </main>
    <aside>
        <div>
            <button class="action" data-page="extract" data-index="<?php echo $this->kernel->escapeString($fileIndex);?>">Extract Backup</button>
            <?php if ($arrData['isMultisite']) : ?>
            <button class="action-disabled" data-tooltip="The restorer does not support Restore for Multisite backups">Restore Backup</button>
            <?php else : ?>
            <button class="action" data-page="restore" data-index="<?php echo $this->kernel->escapeString($fileIndex);?>">Restore Backup</button>
            <?php endif; ?>
        </div>
    </aside>
</div>
<?php endforeach;?>
<?php }
        private function pageBackupRestore() {
?>
<?php
$backupIndex = $this->meta->dataPost['backupIndex']; $data = $this->useHandle->extractor->getBackupFiles($backupIndex); $metaData = (object)$this->useHandle->extractor->readBackupMetaDataFile($data['metaFile']); if ($this->useHandle->wpcore->isWpMultisite() || $data['isMultisite']) { $this->kernel->addBootupError('wpmultiste', 'The restorer does not yet support restoring backups for WordPress Multisites.'); $this->useHandle->view->render('page-bootup-error'); return; } if (empty($metaData->databaseFile)) { $metaData->databaseFile = ''; } $extractPath = $this->useHandle->extractor->getDefaultExtractPath(); $totalFiles = !empty($metaData->totalFiles) ? (int)$metaData->totalFiles : 0; $wpcoreConfig = (object)$this->useHandle->wpcore->getConfig(); $wpBakeryActive = !empty($metaData->wpBakeryActive) ? 1 : 0; $hasExportParts = count(array_filter([ $metaData->isExportingPlugins, $metaData->isExportingMuPlugins, $metaData->isExportingThemes, $metaData->isExportingUploads, $metaData->isExportingOtherWpContentFiles, $metaData->isExportingDatabase, $metaData->isExportingLang, $metaData->isExportingDropins ])); $restoreList = [ 'Media Library' => [ 'name' => 'uploads', 'status' => (int)$metaData->isExportingUploads, 'path' => $wpcoreConfig->uploads, 'restore' => 1, 'overwrite' => 1, ], 'Themes' => [ 'name' => 'themes', 'status' => (int)$metaData->isExportingThemes, 'path' => $wpcoreConfig->themes, 'restore' => 1, 'overwrite' => 1, ], 'Plugins' => [ 'name' => 'plugins', 'status' => (int)$metaData->isExportingPlugins, 'path' => $wpcoreConfig->plugins, 'restore' => 1, 'overwrite' => 1, ], 'Mu-Plugins' => [ 'name' => 'muplugins', 'status' => (int)$metaData->isExportingMuPlugins, 'path' => $wpcoreConfig->muplugins, 'restore' => 1, 'overwrite' => 1, ], 'Languages' => [ 'name' => 'lang', 'status' => (int)$metaData->isExportingLang, 'path' => $wpcoreConfig->lang, 'restore' => 1, 'overwrite' => 1, ], 'Drop-in File' => [ 'name' => 'dropins', 'status' => (int)$metaData->isExportingDropins, 'path' => $wpcoreConfig->wpcontent, 'restore' => 1, 'overwrite' => 1, ], 'Other Files in wp-content' => [ 'name' => 'wpcontent', 'status' => (int)$metaData->isExportingOtherWpContentFiles, 'path' => $wpcoreConfig->wpcontent, 'restore' => 1, 'overwrite' => 1, ], 'Database File' => [ 'name' => 'database', 'status' => (int)$metaData->isExportingDatabase, 'path' => $metaData->isExportingDatabase && !empty($metaData->databaseFile) ? dirname($this->meta->rootPath . '/' . $this->useHandle->withIdentifier::replaceIdentifierPath($metaData->databaseFile)) : '', 'restore' => 1, 'overwrite' => 2, ] ]; $restoreData = [ 'total-files' => $totalFiles, 'wp-version' => $metaData->wpVersion, 'backupfile-path' => $data['path'], 'sqlfile-path' => $metaData->databaseFile, 'searchreplace-backupsiteurl' => $metaData->siteUrl, 'searchreplace-backuphomeurl' => $metaData->homeUrl, 'searchreplace-backupwpbakeryactive' => (int)$wpBakeryActive, 'searchreplace-siteurl' => $wpcoreConfig->siteurl, 'searchreplace-homeurl' => $wpcoreConfig->homeurl, ]; ?>
<div id="backup-extract">
    <ul class="breadcrumb">
        <li><a href="<?php $this->useHandle->view->printAppFile();?>" data-action="page-main">Home</a></li>
        <li><a href="<?php $this->useHandle->view->printAppFile();?>" data-page="extract" data-index="<?php echo $this->kernel->escapeString($backupIndex);?>">Extract Backup</a></li>
        <li><a href="<?php $this->useHandle->view->printAppFile();?>" data-page="content" data-index="<?php echo $this->kernel->escapeString($backupIndex);?>">View Backup</a></li>
        <li>Restore Backup</li>
    </ul>
    <h3>Restore Backup</h3>
    <div class="info-block">
        <div>
            <label>Backup Name</label>
            <span><?php echo $this->kernel->escapeString(basename($metaData->name));?></span>
        </div>
        <div>
            <label>Backup File</label>
            <span><?php echo $this->kernel->escapeString(basename($data['path']));?></span>
            <?php foreach ($restoreData as $key => $value) :?>
            <input type="hidden" name="<?php echo $this->kernel->escapeString($key);?>" id="<?php echo $this->kernel->escapeString($key);?>" value="<?php echo $this->kernel->escapeString($value);?>">
            <?php endforeach;?>
        </div>
        <div>
            <label>Backup Size</label>
            <span><?php echo $this->kernel->escapeString($this->kernel->sizeFormat($metaData->backupSize));?></span>
            <span> ( <?php echo $this->kernel->escapeString($totalFiles) . " " . ( $totalFiles > 1 ? "Files" : "File");?> )</span>
        </div>
        <div>
            <label>Root Path</label>
            <span><?php echo $this->kernel->escapeString($this->kernel->normalizePath($this->meta->rootPath));?></span>
        </div>
    </div>
    <div class="action-block">
        <?php if ($hasExportParts) : ?>
        <h3>Available Contents</h3>
        <div id="restore-table">
            <table>
                <thead>
                    <tr>
                        <th>Parts</th>
                        <th>Path</th>
                        <th>Restore</th>
                        <th>Overwrite</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($restoreList as $type => $data) : if (!$data['status']) : continue; endif; ?>
                    <tr>
                        <td>
                            <p title="<?php echo $this->kernel->escapeString($type);?>"><?php echo $this->kernel->escapeString($type);?></p>
                        </td>
                        <td>
                            <p title="<?php echo $this->kernel->escapeString($data['path']);?>"><?php echo $this->kernel->escapeString($this->kernel->normalizePath($data['path']));?></p>
                        </td>
                        <td>
                            <?php if ($data['restore'] === 1) : ?>
                            <input type="checkbox" name="restore-<?php echo $this->kernel->escapeString($data['name']);?>" value="1" checked>
                            <?php elseif ($data['overwrite'] === 2) : ?>
                            <input type="checkbox" name="restore-bydefault-<?php echo $this->kernel->escapeString($data['name']);?>" value="1" checked disabled>
                            <?php else : ?>
                            [X]
                            <?php endif;?>
                        </td>
                        <td>
                            <?php if ($data['overwrite'] === 1) : ?>
                            <input type="checkbox" name="overwrite-<?php echo $this->kernel->escapeString($data['name']);?>" value="1" checked>
                            <?php elseif ($data['overwrite'] === 2) : ?>
                            <input type="checkbox" name="overwrite-bydefault-<?php echo $this->kernel->escapeString($data['name']);?>"" value=" 1" checked disabled>
                            <?php else : ?>
                            [X]
                            <?php endif;?>
                        </td>
                    </tr>
                    <?php endforeach;?>
                </tbody>
            </table>
        </div>
        <button class="action" data-action="restore">Restore</button>
        <button class="action-close" data-action="page-main">Close</button>
        <button class="action-cancel hide" data-action="restore-cancel">Cancel</button>
        <button class="action-green hide" data-action="restore-retry">Retry</button>
            <?php $this->useHandle->view->printProcessLoader();?>
        <?php else : ?>
        <p>No contents available to restore</p>
        <div class="action-block">
            <button class="action-close" data-action="page-main">Close</button>
        </div>
        <?php endif; ?>
    </div>
</div>
<div id="wpstg-restorer-console"></div>
<?php }
        private function pageBootupError() {
?>
<h2>Restorer Error</h2>
<p>
    WP Staging Restore could not continue for a reason:
</p>
<ul>
    <?php foreach ($this->kernel->getBootupError() as $id => $text) :?>
    <li><?php echo $this->kernel->escapeString($text); ?></li>
    <?php endforeach;?>
</ul>
<button class="action" data-action="page-main">Reload Page</button>
<?php }
        private function pageHash() {
?>
<script id="wpstg-restorer-page"></script>
<?php }
        private function pageLogin() {
?>
<h2>Restorer Security</h2>
<p>
    Please enter the name of the backup file.
</p>
<input class="action" type="text" name="backupfile" id="backupfile" value="" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false" autofocus>
<button class="action" data-action="verify-backup-filename">Verify</button>
<?php $this->useHandle->view->printProcessLoader();?>
<div id="wpstg-restorer-console"></div>
<?php }
        private function pageLogout() {
?>
<h2>Restorer Security</h2>
<p>
    WP Staging Restore session will terminate
</p>
<div id="logout" class="action-block">
    <label class="checkbox">
        <input type="checkbox" name="remove-app-file" id="remove-app-file" value="1">
        <span>
            Remove wpstg-restore.php
            <span data-tooltip="Check this option to remove the wpstg-restore.php file">
                <?php $this->useHandle->view->printSvgInfoOutline();?>
            </span>
        </span>
    </label>
</div>
<button class="action" data-action="access-terminate">Logout</button>
<button class="action-close" data-action="page-main">Cancel</button>
<?php }
        private function pageMain() {
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <meta name="robots" content="noindex, nofollow">
    <meta name="access-token" content="<?php echo $this->useHandle->access->getInitialToken();?>">
    <meta name="app-file" content="<?php $this->useHandle->view->printAppFile();?>">
    <link rel="icon" type="image/png" sizes="32x32" href="<?php $this->useHandle->view->printAssets('favicon-png32');?>">
    <link rel="shortcut icon" type="image/png" sizes="32x32" href="<?php $this->useHandle->view->printAssets('favicon-ico');?>">
    <title>WP Staging | Restore v<?php $this->useHandle->view->printVersion();?></title>
    <link rel="stylesheet" type="text/css" media="all" href="<?php $this->useHandle->view->printAssets('css');?>">
    <script type="text/javascript" src="<?php $this->useHandle->view->printAssets('js');?>"></script>
</head>
<body>
    <div id="wpstg-restorer">
        <?php
$activateIsActive = $this->useHandle->activate->isActive(); $accesshasSession = $this->useHandle->access->hasSession(); ?>
        <header>
            <div class="header-left">
                <img src="<?php $this->useHandle->view->printAssets('logo');?>">
            </div>
            <div class="header-right">
                <table>
                    <tr>
                        <td>
                            <span class="app-name">Restorer</span>
                            <span class="app-version">v<?php $this->useHandle->view->printVersion();?></span>
                            <?php if ($activateIsActive && $accesshasSession) :?>
                            <span class="app-license-type"><?php $this->useHandle->view->printLicenseType();?></span>
                            <?php endif;?>
                        </td>
                    </tr>
                    <tr>
                        <td>
                            <?php if ($activateIsActive && $accesshasSession) :?>
                            <span class="app-license-owner">
                                <?php $this->useHandle->view->printLicenseOwner();?>
                            </span>
                            <?php endif;?>
                        </td>
                    </tr>
                </table>
                <?php if ($accesshasSession) :?>
                <span class="app-logout">
                    <a href="<?php $this->useHandle->view->printAppFile();?>" data-action="page-logout">Logout</a>
                </span>
                <?php endif;?>
            </div>
        </header>
        <div class="content">
            <?php
if (!empty($this->kernel->getBootupError())) { $this->useHandle->view->render('page-bootup-error'); } elseif (!$accesshasSession) { $this->useHandle->view->render('page-login'); } elseif (!$activateIsActive) { $this->useHandle->view->render('page-activate'); } elseif (!$this->useHandle->wpcore->isAvailable()) { $this->useHandle->view->render('page-wpcore-install'); } elseif (!$this->useHandle->wpcore->isReady()) { if (!$this->useHandle->wpcore->isDbConnect()['success']) { $this->useHandle->wpcore->resetDbConfig(); $this->useHandle->view->render('page-wpcore-setup-db'); } elseif (!$this->useHandle->wpcore->isDbInstalled()) { $this->useHandle->view->render('page-wpcore-setup-site'); } else { $this->useHandle->view->render('page-wpcore-setup-complete'); } } else { $this->useHandle->view->render('page-hash'); }?>
        </div>
    </div>
</body>
</html>
<?php }
        private function pageWpcoreInstall() {
?>
<h2>Restorer Setups</h2>
<p>
    The WordPress Core is not installed.
</p>
<div id="wpcore" class="form-block">
    <div>
        <label>WordPress Version</label>
        <select id="wpversion" name="wpversion">
            <option value="latest">Latest</option>
            <?php foreach ($this->getWpVersion() as $id => $wpVersion) : ?>
            <option value="<?php echo $this->kernel->escapeString($wpVersion);?>"><?php echo $this->kernel->escapeString($wpVersion);?></option>
            <?php endforeach; ?>
        </select>
    </div>
</div>
<button class="action" data-action="wpcore-install">Install WordPress Core</button>
<?php $this->useHandle->view->printProcessLoader();?>
<div id="wpstg-restorer-console"></div>
<?php }
        private function pageWpcoreSetupComplete() {
?>
<h2>Restorer Setups</h2>
<p>
    WP Staging Restore detects your site has a Database installed. No further action is required.
</p>
<p>
    Click the Finish button to complete the WordPress installation.
</p>
<button class="action" data-action="wpcore-setup-complete">Finish</button>
<?php $this->useHandle->view->printProcessLoader();?>
<div id="wpstg-restorer-console"></div>
<?php }
        private function pageWpcoreSetupDb() {
?>
<?php
$wpPrefix = $this->useHandle->cache->get('wpprefix', 'setup'); $wpPrefix = !empty($wpPrefix) ? $wpPrefix : ''; ?>
<h2>Restorer Setups</h2>
<p>
    Please enter the database settings.
</p>
<div id="dbconfig" class="form-block">
    <div>
        <label>Database Server</label>
        <input type="text" name="dbhost" id="dbhost" value="localhost" placeholder="localhost" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false" autofocus>
    </div>
    <div>
        <label>Database Name</label>
        <input type="text" name="dbname" id="dbname" value="" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
    </div>
    <div>
        <label>Database User</label>
        <input type="text" name="dbuser" id="dbuser" value="" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
    </div>
    <div>
        <label>Database Password</label>
        <input type="password" name="dbpass" id="dbpass" value="" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
    </div>
    <div>
        <label>Table Prefix</label>
        <input type="text" name="dbprefix" id="dbprefix" value="<?php echo $this->kernel->escapeString($wpPrefix);?>" placeholder="wp_" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
    </div>
    <div>
        <label>Custom Port</label>
        <input type="text" name="dbport" id="dbport" value="" placeholder="" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
    </div>
    <div>
        <label>Enable SSL</label>
        <input type="checkbox" name="dbssl" id="dbssl" value="1">
    </div>
</div>
<button class="action" data-action="wpcore-setup-db">Submit</button>
<?php $this->useHandle->view->printProcessLoader();?>
<div id="wpstg-restorer-console"></div>
<?php }
        private function pageWpcoreSetupSite() {
?>
<h2>Restorer Setups</h2>
<p>
    Click submit to complete the WordPress installation.
</p>
<div id="siteconfig" class="form-block">
    <div>
        <label>Site Title</label>
        <input type="text" name="sitetitle" id="sitetitle" value="WP Staging | Restore" placeholder="Site Title" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false" autofocus>
    </div>
    <div>
        <label>Admin User</label>
        <input type="text" name="siteuser" id="siteuser" value="wpstg-restore" placeholder="Enter Admin User" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
    </div>
    <div>
        <label>Admin Email</label>
        <input type="email" name="siteemail" id="siteemail" value="" placeholder="Enter Admin Email Address" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
    </div>
    <div>
        <label>Admin Password</label>
        <input type="password" name="sitepass" id="sitepass" value="" placeholder="Enter Admin Password" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
    </div>
</div>
<button class="action" data-action="wpcore-setup-site">Submit</button>
<?php $this->useHandle->view->printProcessLoader();?>
<button class="action-green hide" data-action="page-main">Continue</button>
<div id="wpstg-restorer-console"></div>
<?php }
    }
    final class WithIdentifier { private $kernel; private $meta; private $useHandle; const IDENTIFIER_ABSPATH = 'wpstg_a_'; const IDENTIFIER_WPCONTENT = 'wpstg_c_'; const IDENTIFIER_PLUGINS = 'wpstg_p_'; const IDENTIFIER_THEMES = 'wpstg_t_'; const IDENTIFIER_MUPLUGINS = 'wpstg_m_'; const IDENTIFIER_UPLOADS = 'wpstg_u_'; const IDENTIFIER_LANG = 'wpstg_l_'; public function __construct(\WPStagingRestorer $kernel) { $this->kernel = $kernel; $this->useHandle = $this->kernel->getHandle(__CLASS__, ['wpcore']); $this->meta = $this->kernel->getMeta(); } public static function matchWith(string $string, string $identifier): bool { return $identifier === substr($string, 0, strlen($identifier)); } public static function replaceIdentifierPath(string $string): string { $key = substr($string, 0, 8); $path = self::getRelativePath($key); if (!empty($path) && is_string($path)) { return substr_replace($string, $path, 0, 8); } return $string; } public static function getRelativePath(string $identifier): string { static $cache = []; if (!empty($cache) && !empty($identifier) && !empty($cache[$identifier])) { return $cache[$identifier]; } $path = [ self::IDENTIFIER_ABSPATH => '', self::IDENTIFIER_WPCONTENT => 'wp-content/', self::IDENTIFIER_PLUGINS => 'wp-content/plugins/', self::IDENTIFIER_THEMES => 'wp-content/themes/', self::IDENTIFIER_MUPLUGINS => 'wp-content/mu-plugins/', self::IDENTIFIER_UPLOADS => 'wp-content/uploads/', self::IDENTIFIER_LANG => 'wp-content/languages/', ]; if (!empty($identifier) && !empty($path[$identifier])) { $cache[$identifier] = $path[$identifier]; return $cache[$identifier]; } trigger_error(sprintf('Could not find a path for the placeholder: %s', filter_var($identifier, FILTER_SANITIZE_SPECIAL_CHARS))); return $identifier; } public function getAbsolutePath(string $identifier): string { static $cache = []; if (!empty($cache) && !empty($identifier) && !empty($cache[$identifier])) { return $cache[$identifier]; } $wpcorePath = (object)$this->useHandle->wpcore->getConfig(); $path = [ self::IDENTIFIER_ABSPATH => $wpcorePath->abspath, self::IDENTIFIER_WPCONTENT => $wpcorePath->wpcontent, self::IDENTIFIER_PLUGINS => $wpcorePath->plugins, self::IDENTIFIER_THEMES => $wpcorePath->themes, self::IDENTIFIER_MUPLUGINS => $wpcorePath->muplugins, self::IDENTIFIER_UPLOADS => $wpcorePath->uploads, self::IDENTIFIER_LANG => $wpcorePath->lang, ]; if (!empty($identifier) && !empty($path[$identifier])) { $cache[$identifier] = $path[$identifier]; return $cache[$identifier]; } trigger_error(sprintf('Could not find a path for the placeholder: %s', filter_var($identifier, FILTER_SANITIZE_SPECIAL_CHARS))); return $identifier; } }
    final class WpCore { private $kernel; private $meta; private $useHandle; private $taskFile; private $dbConfigFile; private $wpConfigFile; private $downloadUrl; private $maintenanceFile; const WPCORE_INSTALL_FAILURE = 0; const WPCORE_INSTALL_SUCCESS = 1; const WPCORE_INSTALL_DONE = 2; const IS_STAGING_KEY = 'wpstg_is_staging_site'; const STAGING_FILE = '.wp-staging'; public function __construct(\WPStagingRestorer $kernel) { $this->kernel = $kernel; $this->meta = $this->kernel->getMeta(); $this->useHandle = $this->kernel->getHandle(__CLASS__, ['file', 'cache']); $this->taskFile = $this->meta->tmpPath . '/wpstg-task-wpcore.php'; $this->dbConfigFile = $this->meta->tmpPath . '/wpstg-dbconfig.php'; $this->wpConfigFile = $this->locateWpConfigFile(); $this->downloadUrl = 'https://wordpress.org'; $this->maintenanceFile = $this->meta->rootPath . '/.maintenance'; } private function loadLibrary(): bool { static $isLoaded = false; if ($isLoaded) { return true; } if (!$this->isAvailable()) { return false; } if (!$this->isReady()) { return false; } $isMaintenance = $this->isMaintenance(); if ($isMaintenance) { $this->enableMaintenance(false); } try { define('SHORTINIT', true); require_once __DIR__ . '/wp-load.php'; wp_plugin_directory_constants(); require_once ABSPATH . WPINC . '/class-wp-textdomain-registry.php'; if (!isset($GLOBALS['wp_textdomain_registry']) || !($GLOBALS['wp_textdomain_registry'] instanceof \WP_Textdomain_Registry)) { $GLOBALS['wp_textdomain_registry'] = new \WP_Textdomain_Registry(); } foreach ( [ 'l10n.php', 'class-wp-user.php', 'class-wp-roles.php', 'class-wp-role.php', 'class-wp-session-tokens.php', 'class-wp-user-meta-session-tokens.php', 'http.php', 'formatting.php', 'capabilities.php', 'user.php', 'link-template.php' ] as $file ) { require_once ABSPATH . WPINC . '/' . $file; } wp_cookie_constants(); foreach ( [ 'vars.php', 'kses.php', 'cron.php', 'rest-api.php', 'pluggable.php', 'theme.php' ] as $file ) { require_once ABSPATH . WPINC . '/' . $file; } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); return false; } if ($isMaintenance) { $this->enableMaintenance(true); } $isLoaded = true; return true; } public function maybeUpgradeDatabase(): bool { if (!$this->loadLibrary()) { return false; } try { if (file_exists(trailingslashit(ABSPATH) . 'wp-admin/includes/upgrade.php')) { global $wpdb, $wp_db_version, $wp_current_db_version; wp_templating_constants(); require_once ABSPATH . WPINC . '/class-wp-theme.php'; require_once ABSPATH . WPINC . '/class-wp-walker.php'; require_once ABSPATH . 'wp-admin/includes/upgrade.php'; $wp_current_db_version = (int)__get_option('db_version'); if (!empty($wp_current_db_version) && !empty($wp_db_version) && $wp_db_version !== $wp_current_db_version) { $wpdb->suppress_errors(); wp_upgrade(); $this->kernel->log(sprintf('WordPress database upgraded successfully. Old version: %s, New version: %s', $wp_current_db_version, $wp_db_version), __METHOD__); return true; } } else { $this->kernel->log('Could not upgrade WordPress database version as the wp-admin/includes/upgrade.php file does not exist', __METHOD__); } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); } return false; } public function maybeRemoveStagingStatus(): bool { if (!$this->loadLibrary()) { return false; } if (defined('WPSTAGING_DEV_SITE') && (bool)WPSTAGING_DEV_SITE === true) { return false; } if (file_exists(ABSPATH . self::STAGING_FILE)) { return false; } if (get_option(self::IS_STAGING_KEY) === "true") { return delete_option(self::IS_STAGING_KEY); } return false; } public function flushObjectCache(): bool { if (!$this->loadLibrary()) { return false; } $dropInFile = wp_normalize_path(WP_CONTENT_DIR) . '/object-cache.php'; clearstatcache(true, $dropInFile); if (!file_exists($dropInFile) || !function_exists('wp_cache_flush')) { return true; } try { wp_cache_flush(); } catch (\Throwable $e) { $this->kernel->log('Failed to flush object cache', __METHOD__); $this->kernel->log($e, __METHOD__); return false; } return true; } public function getConfig(bool $force = false) { $data = $this->useHandle->cache->get('wpcoreconfig', 'config'); if (!$force && $data !== null) { return $data; } return $this->saveConfig(); } public function getBackupPath(): string { $backupPath = $this->meta->backupPath; $config = $this->getConfig(false); if (empty($config) || empty($config['uploads'])) { return $backupPath; } $uploadPath = $config['uploads']; if ($uploadPath !== $this->meta->uploadPath) { $backupPath = $uploadPath . '/' . $this->meta->backupDir; $this->kernel->log($backupPath); } return $backupPath; } public function saveConfig() { if (!$this->loadLibrary()) { return false; } list( $host, $port, $socket, $isIPv6 ) = $this->parseDbHost(DB_HOST); $siteUrl = get_option('siteurl'); $homeUrl = get_option('home'); $guessUrl = wp_guess_url(); if ($guessUrl !== $siteUrl) { $siteUrl = $guessUrl; $homeUrl = $guessUrl; } $keys = [ 'abspath' => ABSPATH, 'uploads' => wp_normalize_path(wp_upload_dir(null, false, false)['basedir']), 'plugins' => wp_normalize_path(WP_PLUGIN_DIR), 'muplugins' => wp_normalize_path(WPMU_PLUGIN_DIR), 'themes' => wp_normalize_path(get_theme_root(get_template())), 'wpcontent' => wp_normalize_path(WP_CONTENT_DIR), 'lang' => wp_normalize_path(WP_LANG_DIR), 'dbname' => DB_NAME, 'dbuser' => DB_USER, 'dbpass' => DB_PASSWORD, 'dbhost' => $host, 'dbport' => $port, 'dbssl' => defined('MYSQL_CLIENT_FLAGS') ? 1 : 0, 'dbprefix' => isset($GLOBALS['table_prefix']) ? $GLOBALS['table_prefix'] : 'wp_', 'dbcharset' => DB_CHARSET, 'dbcollate' => DB_COLLATE, 'siteurl' => $siteUrl, 'homeurl' => $homeUrl, 'multisite' => defined('WP_ALLOW_MULTISITE') && WP_ALLOW_MULTISITE && defined('MULTISITE') && MULTISITE ? 1 : 0 ]; if ($this->useHandle->cache->put('wpcoreconfig', $keys, 'config')) { return $keys; } return false; } public function isMaintenance(): bool { clearstatcache(); return file_exists($this->maintenanceFile); } public function enableMaintenance(bool $isMaintenance): bool { if ($isMaintenance && !$this->isMaintenance()) { file_put_contents($this->maintenanceFile, '<?php $upgrading = time() ?>', LOCK_EX); $this->kernel->chmod($this->maintenanceFile, false, __LINE__); return true; } if (!$isMaintenance && $this->isMaintenance()) { $this->kernel->unlink($this->maintenanceFile, __LINE__); return true; } return false; } public function isAvailable(): bool { clearstatcache(); return file_exists($this->meta->rootPath . '/wp-load.php') && file_exists($this->meta->rootPath . '/wp-blog-header.php') && file_exists($this->meta->rootPath . '/wp-settings.php') && file_exists($this->meta->rootPath . '/wp-includes/load.php') && file_exists($this->meta->rootPath . '/wp-admin/admin.php') && is_dir($this->meta->rootPath . '/wp-content'); } public function isReady(): bool { if (!file_exists($this->wpConfigFile)) { return false; } if (!$this->isWpIndex()) { return false; } return true; } public function isWpIndex(): bool { $wpIndex = $this->meta->rootPath . '/index.php'; if (!file_exists($wpIndex)) { return false; } $content = file_get_contents($wpIndex, false, null, 0, 8 * 1024); if (empty($content) || strpos($content, '/wp-blog-header.php') === false) { return false; } $wpIndexSetup = $this->meta->rootPath . '/index-wp.php'; if (file_exists($wpIndexSetup)) { $this->kernel->unlink($wpIndexSetup, __LINE__); } return true; } public function isWpMultisite(): bool { if (!file_exists($this->wpConfigFile)) { return false; } $content = file_get_contents($this->wpConfigFile, false, null, 0, 8 * 1024); if (empty($content)) { return false; } if (!preg_match('@define\(\s+(\'|")WP_ALLOW_MULTISITE(\'|"),\s+(true|1)\s+\)\;@', $content) && !preg_match('@define\(\s+(\'|")MULTISITE(\'|"),\s+(true|1)\s+\)\;@', $content)) { return false; } return true; } private function setTaskStatus($status, $text, $callback = false): bool { $data = $this->getTaskStatus(); if (empty($data) || !is_array($data)) { $data[0] = [ 'status' => $status, 'text' => $text, 'callback' => $callback, ]; } else { $lastData = !empty($data[0]) && count($data) > 0 ? $data[count($data) - 1] : $data; if ($lastData['status'] !== self::WPCORE_INSTALL_DONE) { $data[] = [ 'status' => $status, 'text' => $text, 'callback' => $callback, ]; } } $this->kernel->log($text, __METHOD__); return $this->useHandle->cache->put('wpcoretask', $data); } public function getTaskStatus(): array { $data = $this->useHandle->cache->get('wpcoretask'); return is_array($data) ? $data : []; } public function resetTaskStatus(): bool { return $this->useHandle->cache->remove('wpcoretask'); } private function tempName(string $input): string { return substr(md5($input), 0, 12); } public function downloadStatus(string $savePath): int { $fileName = $this->tempName(basename($savePath)) . '.txt'; $filePath = $this->meta->tmpPath . '/download-status-' . $fileName; if (!file_exists($filePath)) { return 0; } $data = file_get_contents($filePath); if (empty($data)) { return 0; } $data = strtok($data, '|'); return (int)$data; } private function downloadFile(string $fileUrl, string $savePath, bool $refresh = false): bool { if ($refresh && file_exists($savePath)) { unlink($savePath); } $saveName = basename($savePath); $this->setTaskStatus(self::WPCORE_INSTALL_SUCCESS, sprintf('Downloading %s as %s', $fileUrl, $saveName), ['downloadStatus', $savePath]); if (!($fileHandle = fopen($savePath, 'wb'))) { $this->setTaskStatus(self::WPCORE_INSTALL_FAILURE, sprintf('Failed to create %s', $saveName)); return false; } $curlHandle = curl_init($fileUrl); $fileName = $this->tempName($saveName) . '.txt'; curl_setopt_array($curlHandle, [ CURLOPT_USERAGENT => $this->kernel->userAgent(), CURLOPT_SSL_VERIFYHOST => 0, CURLOPT_SSL_VERIFYPEER => 0, CURLOPT_HEADER => false, CURLOPT_FOLLOWLOCATION => true, CURLOPT_BINARYTRANSFER => true, CURLOPT_NOPROGRESS => false, CURLOPT_FORBID_REUSE => true, CURLOPT_FRESH_CONNECT => true, CURLOPT_TIMEOUT => 180, CURLOPT_FILE => $fileHandle, CURLOPT_PROGRESSFUNCTION => function ($resource, $downloadSize, $downloaded, $uploadSize, $uploaded) use ($fileName, $fileUrl) { if (!empty($downloadSize)) { $percentage = ($downloaded / $downloadSize) * 100; file_put_contents($this->meta->tmpPath . '/download-status-' . $fileName, $percentage . '|' . $fileUrl, LOCK_EX); } }, ]); if (!($status = curl_exec($curlHandle))) { $this->setTaskStatus(self::WPCORE_INSTALL_FAILURE, sprintf('Failed to download %s: %s', $fileUrl, curl_error($curlHandle))); } curl_close($curlHandle); fclose($fileHandle); return $status ? true : false; } private function checksum(string $zipFile, string $md5File): bool { return trim(file_get_contents($md5File)) === md5_file($zipFile); } private function extractFile(string $zipFile): bool { $zipFileName = basename($zipFile); $this->setTaskStatus(self::WPCORE_INSTALL_SUCCESS, sprintf('Extracting %s', $zipFileName), ['wpcorestatusextract']); try { $zip = new \ZipArchive(); if ($zip->open($zipFile) && $zip->extractTo($this->meta->tmpPath)) { $this->useHandle->cache->put('wpcorestatusextract', '<!--{{taskCallbackDone}}-->'); } $zip->close(); } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); $this->setTaskStatus(self::WPCORE_INSTALL_FAILURE, sprintf('Failed to extract %s: %s', $zipFileName, $e->getMessage())); return false; } return true; } private function copyToRootPath(): bool { $this->setTaskStatus(self::WPCORE_INSTALL_SUCCESS, 'Copying WordPress files to the root path', ['wpcorestatuscopy']); $dstPath = $this->meta->rootPath; $srcPath = $this->meta->tmpPath . '/wordpress'; if (!is_dir($dstPath) || $dstPath === '/' || !is_dir($srcPath) || $srcPath === '/') { $this->setTaskStatus(self::WPCORE_INSTALL_FAILURE, 'Failed to copy WordPress files to the root path'); return false; } try { $this->useHandle->cache->remove('wpcorestatuscopy'); $dirIterator = new \RecursiveIteratorIterator( new \RecursiveDirectoryIterator($srcPath, \RecursiveDirectoryIterator::SKIP_DOTS), \RecursiveIteratorIterator::SELF_FIRST ); foreach ($dirIterator as $item) { $filePath = $this->kernel->normalizePath($dstPath . '/' . $dirIterator->getSubPathname()); if ($item->isDir()) { $this->kernel->mkdir($filePath, __LINE__); } else { if ($filePath === $this->meta->rootPath . '/index.php') { $filePath = $dstPath . '/index-wp.php'; } $itemCopy = $this->kernel->normalizePath($item->getPathname()); $this->kernel->mkdir(dirname($filePath), __LINE__); if (!rename($itemCopy, $filePath)) { $this->setTaskStatus(self::WPCORE_INSTALL_FAILURE, sprintf('Failed to copy WordPress file: %s', $this->kernel->stripRootPath($filePath))); return false; } $this->useHandle->cache->put('wpcorestatuscopy', $this->kernel->stripRootPath($filePath)); } } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); $this->setTaskStatus(self::WPCORE_INSTALL_FAILURE, sprintf('Failed to copy WordPress files to the root path: %s', $e->getMessage())); return false; } $this->useHandle->cache->put('wpcorestatuscopy', '<!--{{taskCallbackDone}}-->'); $this->useHandle->file->removeDir($srcPath); return true; } public function getTaskResponse(): array { $data = $this->getTaskStatus(); if (empty($data) || !is_array($data)) { return [ 'success' => true, 'data' => [ 'status' => self::WPCORE_INSTALL_SUCCESS, 'content' => 'Checking.. please wait.', ] ]; } $lastData = !empty($data[0]) && count($data) > 0 ? $data[count($data) - 1] : $data; $content = ''; foreach ($data as $k => $arr) { $text = $arr['text']; if (!empty($arr['callback']) && is_array($arr['callback'])) { if ($arr['callback'][0] === 'downloadStatus') { $percent = $this->downloadStatus($arr['callback'][1]); if ($percent > 0 && $percent < 100) { $text .= '.. ' . $percent . "%\n"; } } elseif (substr($arr['callback'][0], 0, 12) === 'wpcorestatus') { $status = $this->useHandle->cache->get($arr['callback'][0]); if (!empty($status)) { $text .= $status === '<!--{{taskCallbackDone}}-->' ? ' was successful' : ': ' . $status; } } } if (!empty($text)) { $content .= $text . "\n"; } } return [ 'success' => true, 'data' => [ 'status' => $lastData['status'], 'content' => $content ] ]; } public function runTask(): bool { $this->kernel->maxExecutionTime(240); $version = !empty($this->meta->dataPost['wpcore-version']) ? $this->meta->dataPost['wpcore-version'] : 'latest'; $zipFileName = 'wordpress-' . $version . '.zip'; $zipUrl = $this->downloadUrl . '/' . $zipFileName; $md5Url = $zipUrl . '.md5'; $zipFile = $this->meta->tmpPath . '/' . $zipFileName; $md5File = $this->meta->tmpPath . '/' . $zipFileName . '.md5'; clearstatcache(); if (!file_exists($zipFile) && !$this->downloadFile($zipUrl, $zipFile)) { $this->setTaskStatus(self::WPCORE_INSTALL_FAILURE, sprintf('Failed to download %s', $zipUrl)); unlink($zipFile); return false; } if (!file_exists($md5File) && !$this->downloadFile($md5Url, $md5File)) { $this->setTaskStatus(self::WPCORE_INSTALL_FAILURE, sprintf('Failed to download %s', $md5Url)); unlink($md5File); return false; } $this->setTaskStatus(self::WPCORE_INSTALL_SUCCESS, sprintf('Validating checksum %s', $zipFileName), ['wpcorestatuschecksum']); if (!$this->checksum($zipFile, $md5File)) { unlink($zipFile); unlink($md5File); $this->setTaskStatus(self::WPCORE_INSTALL_FAILURE, sprintf('Invalid checksum for %s', $zipFileName)); return false; } $this->useHandle->cache->put('wpcorestatuschecksum', '<!--{{taskCallbackDone}}-->'); if (!$this->extractFile($zipFile)) { return false; } if (!$this->copyToRootPath()) { return false; } $this->setTaskStatus(self::WPCORE_INSTALL_DONE, 'Installing WordPress was successful'); return true; } private function randomNumber($min = null, $max = null): int { static $rndValue; $maxRandomNumber = 3000000000 === 2147483647 ? (float) '4294967295' : 4294967295; if ($min === null) { $min = 0; } if ($max === null) { $max = $maxRandomNumber; } $min = (int) $min; $max = (int) $max; static $useRandomIntFunctionality = true; if ($useRandomIntFunctionality) { try { $smax = max($min, $max); $smin = min($min, $max); $val = random_int($smin, $smax); if ($val !== false) { return abs((int) $val); } else { $useRandomIntFunctionality = false; } } catch (\Throwable $e) { $useRandomIntFunctionality = false; } } if (strlen($rndValue) < 8) { static $seed = ''; $rndValue = md5(uniqid(microtime() . mt_rand(), true) . $seed); $rndValue .= sha1($rndValue); $rndValue .= sha1($rndValue . $seed); $seed = md5($seed . $rndValue); } $value = substr($rndValue, 0, 8); $rndValue = substr($rndValue, 8); $value = abs(hexdec($value)); $value = $min + ( $max - $min + 1 ) * $value / ( $maxRandomNumber + 1 ); return abs((int) $value); } private function generateSaltKey(int $length = 12, bool $specialChars = true, bool $extraSpecialChars = false): string { $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'; if ($specialChars) { $chars .= '!@#$%^&*()'; } if ($extraSpecialChars) { $chars .= '-_ []{}<>~`+=,.;:/?|'; } $saltkey = ''; for ($i = 0; $i < $length; $i++) { $saltkey .= substr($chars, $this->randomNumber(0, strlen($chars) - 1), 1); } return $saltkey; } private function parseDbHost(string $host) { $socket = null; $isIPv6 = false; $socketPos = strpos($host, ':/'); if ($socketPos !== false) { $socket = substr($host, $socketPos + 1); $host = substr($host, 0, $socketPos); } if (substr_count($host, ':') > 1) { $pattern = '#^(?:\[)?(?P<host>[0-9a-fA-F:]+)(?:\]:(?P<port>[\d]+))?#'; $isIPv6 = true; } else { $pattern = '#^(?P<host>[^:/]*)(?::(?P<port>[\d]+))?#'; } $matches = []; $result = preg_match($pattern, $host, $matches); if ($result !== 1) { return false; } $host = !empty($matches['host']) ? $matches['host'] : ''; $port = !empty($matches['port']) ? abs((int) $matches['port']) : null; return [$host, $port, $socket, $isIPv6]; } public function dbHandle(): Database { $dbData = $this->getDbConfig(); if (empty($dbData)) { throw new \BadMethodCallException('Failed to read database config'); } return new Database($this->kernel, $dbData); } public function isDbConnect(): array { clearstatcache(); if (!file_exists($this->dbConfigFile)) { $dbData = $this->parseWpConfigForDb(); $isSaveDbConfig = false; if (!empty($dbData) && is_array($dbData)) { $this->meta->dataPost['db-data'] = $dbData; $isSaveDbConfig = $this->saveDbConfig()['success'] === true; unset($this->meta->dataPost['db-data']); } if (!$isSaveDbConfig) { return ['success' => false, 'data' => 'Configuration not found']; } } try { $dbHandle = $this->dbHandle(); if ($dbHandle->connect() === false) { return ['success' => false, 'data' => $dbHandle->response]; } } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); return ['success' => false, 'data' => sprintf('Error: %s', $e->getMessage())]; } $text = sprintf("Connection Success: %s\nServer Info: %s\n", $dbHandle->clientInfo(), $dbHandle->serverInfo()); $dbHandle->close(); return ['success' => true, 'data' => $text]; } public function isDbInstalled(): bool { $dbHandle = $this->dbHandle(); if ($dbHandle->connect() === false) { return false; } $dbPrefix = $dbHandle->dbPrefix; $result = $dbHandle->query('SHOW TABLES LIKE "' . $dbPrefix . '%"'); if (! ($result instanceof \mysqli_result) || (int)$result->num_rows === 0) { return false; } $wpTables = [ $dbPrefix . 'commentmeta' => 1, $dbPrefix . 'comments' => 1, $dbPrefix . 'links' => 1, $dbPrefix . 'options' => 1, $dbPrefix . 'postmeta' => 1, $dbPrefix . 'posts' => 1, $dbPrefix . 'term_relationships' => 1, $dbPrefix . 'term_taxonomy' => 1, $dbPrefix . 'termmeta' => 1, $dbPrefix . 'terms' => 1, $dbPrefix . 'usermeta' => 1, $dbPrefix . 'users' => 1, ]; $tableTotal = (int)$result->num_rows; $tableFound = 0; while ($row = $result->fetch_row()) { if (isset($wpTables[$row[0]])) { $tableFound++; } } if ($tableFound !== count($wpTables)) { return false; } $result = $dbHandle->query('SELECT ID from `' . $dbPrefix . 'users` LIMIT 1'); if (! ($result instanceof \mysqli_result) || (int)$result->num_rows === 0) { return false; } $result = $dbHandle->query('SELECT option_id from `' . $dbPrefix . 'options` LIMIT 1'); if (! ($result instanceof \mysqli_result) || (int)$result->num_rows === 0) { return false; } return true; } public function getDbConfig() { $this->maybeCreateDbConfig(); try { $dbData = include $this->dbConfigFile; } catch (\Throwable $e) { $this->kernel->log($e, __METHOD__); return false; } return $dbData; } private function maybeCreateDbConfig(): bool { clearstatcache(); if (!file_exists($this->wpConfigFile) || (file_exists($this->dbConfigFile) && filemtime($this->dbConfigFile) > filemtime($this->wpConfigFile))) { return false; } $config = $this->getConfig(true); if (empty($config) || !is_array($config)) { return false; } $config = (object)$config; $this->meta->dataPost['db-data'] = [ 'dbname' => $config->dbname, 'dbuser' => $config->dbuser, 'dbpass' => $config->dbpass, 'dbhost' => $config->dbhost, 'dbport' => $config->dbport, 'dbssl' => $config->dbssl, 'dbprefix' => $config->dbprefix, 'dbcharset' => $config->dbcharset, 'dbcollate' => $config->dbcollate, ]; if ($this->saveDbConfig()['success'] === false) { return false; } return true; } public function saveDbConfig(): array { if (empty($this->meta->dataPost['db-data'])) { return ['success' => false, 'data' => 'Please enter database setting!']; } $dbData = []; foreach ($this->meta->dataPost['db-data'] as $key => $value) { if ($key === 'dbpass') { $value = htmlspecialchars_decode($value); } $dbData[$key] = $value; } $errorData = []; foreach (['dbhost','dbname', 'dbuser', 'dbpass', 'dbprefix', 'dbport', 'dbssl', 'dbipv6', 'dbcharset', 'dbcollate'] as $key) { if (empty($dbData[$key])) { switch ($key) { case 'dbhost': $dbData[$key] = 'localhost'; break; case 'dbname': $errorData[$key] = 'Please enter Database Name!'; break; case 'dbuser': $errorData[$key] = 'Please enter Database User!'; break; case 'dbpass': $errorData[$key] = 'Please enter Database Password!'; break; case 'dbprefix': $dbData[$key] = 'wp_'; break; case 'dbssl': $dbData[$key] = 0; break; case 'dbipv6': $dbData[$key] = 0; break; case 'dbport': $dbData[$key] = null; break; case 'dbcharset': $dbData[$key] = 'utf8'; break; case 'dbcollate': $dbData[$key] = ''; break; } } } if (!empty($errorData)) { return ['success' => false, 'data' => implode("\n", $errorData)]; } $this->useHandle->file->opcacheFlush($this->dbConfigFile); $hostData = $this->parseDbHost($dbData['dbhost']); if ($hostData) { list( $host, $port, $socket, $isIPv6 ) = $hostData; $dbData['dbipv6'] = $isIPv6 ? 1 : 0; } $code = '<?php return ' . var_export($dbData, true) . ';'; if (!file_put_contents($this->dbConfigFile, $code, LOCK_EX)) { return ['success' => false, 'data' => 'Failed to save database setting']; } return $this->isDbConnect(); } public function resetDbConfig(): bool { if ($this->kernel->unlink($this->dbConfigFile)) { if ($this->isWpIndex()) { rename($this->meta->rootPath . '/index.php', $this->meta->rootPath . '/index-wp.php'); } return true; } return false; } private function writeWpConfig(): array { if (($dbData = $this->getDbConfig()) === false) { return ['success' => false, 'data' => 'Failed to get Database configuration']; } $dbData = (object)$dbData; $dbHost = $dbData->dbhost . ( !empty($dbData->dbport) ? ':' . $dbData->dbport : ''); $dbPass = addslashes($dbData->dbpass); $code = '<?php ' . PHP_EOL; $code .= '// Generated by WP Staging Restore: ' . date('M j, Y H:i:s') . ' UTC' . PHP_EOL; $code .= "define('WP_CACHE', false);" . PHP_EOL; $code .= "define('WP_REDIS_DISABLED', true);" . PHP_EOL; foreach ( [ 'DB_NAME' => $dbData->dbname, 'DB_USER' => $dbData->dbuser, 'DB_PASSWORD' => $dbPass , 'DB_HOST' => $dbHost, 'DB_CHARSET' => 'utf8', 'DB_COLLATE' => '', 'AUTH_KEY' => $this->generateSaltKey(64, true, true), 'SECURE_AUTH_KEY' => $this->generateSaltKey(64, true, true), 'LOGGED_IN_KEY' => $this->generateSaltKey(64, true, true), 'NONCE_KEY' => $this->generateSaltKey(64, true, true), 'AUTH_SALT' => $this->generateSaltKey(64, true, true), 'SECURE_AUTH_SALT' => $this->generateSaltKey(64, true, true), 'LOGGED_IN_SALT' => $this->generateSaltKey(64, true, true), 'NONCE_SALT' => $this->generateSaltKey(64, true, true) ] as $name => $value ) { $code .= "define('" . $name . "', '" . $value . "');" . PHP_EOL; } if ($dbData->dbssl) { $code .= "define('MYSQL_CLIENT_FLAGS', MYSQLI_CLIENT_SSL | MYSQLI_CLIENT_SSL_DONT_VERIFY_SERVER_CERT);" . PHP_EOL; } $code .= "\$table_prefix = '" . $dbData->dbprefix . "';" . PHP_EOL; $code .= "define('WP_DEBUG', false);" . PHP_EOL; $code .= "if (!defined('WP_DISABLE_FATAL_ERROR_HANDLER')) { define( 'WP_DISABLE_FATAL_ERROR_HANDLER', true ); }" . PHP_EOL; $code .= "if (!defined('WP_HTTP_BLOCK_EXTERNAL')) { define( 'WP_HTTP_BLOCK_EXTERNAL', false ); }" . PHP_EOL; $code .= "if (!defined('WP_ACCESSIBLE_HOSTS')) { define( 'WP_ACCESSIBLE_HOSTS', 'analytics.local, analytics.wp-staging.com' ); }" . PHP_EOL; $code .= "if (!defined('ABSPATH')) { define( 'ABSPATH', __DIR__ . '/' ); }" . PHP_EOL; $code .= "require_once ABSPATH . 'wp-settings.php';" . PHP_EOL; if (!file_put_contents($this->wpConfigFile, $code, LOCK_EX)) { return ['success' => false, 'data' => 'Failed to create wp-config.php']; } $this->kernel->chmod($this->wpConfigFile, false, __LINE__); return ['success' => true, 'data' => 'Creating wp-config.php succesful']; } public function installSite(): array { if (empty($this->meta->dataPost['site-data'])) { return ['success' => false, 'data' => 'Please enter Site setting!']; } $siteData = []; foreach ($this->meta->dataPost['site-data'] as $key => $value) { if ($key === 'sitepass') { $value = htmlspecialchars_decode($value); } $siteData[$key] = $value; } $errorData = []; foreach (['sitetitle','siteuser', 'siteemail', 'sitepass'] as $key) { if (empty($siteData[$key])) { switch ($key) { case 'sitetitle': $errorData[$key] = 'Please enter Site Title!'; break; case 'siteuser': $errorData[$key] = 'Please enter Admin User!'; break; case 'siteemail': $errorData[$key] = 'Please enter Admin Email!'; break; case 'sitepass': $errorData[$key] = 'Please enter Admin Password!'; break; } } else { switch ($key) { case 'siteemail': if (!filter_var($siteData[$key], FILTER_VALIDATE_EMAIL)) { $errorData[$key] = sprintf('Invalid email address %s!', $siteData[$key]); } break; } } } if (!empty($errorData)) { return ['success' => false, 'data' => implode("\n", $errorData)]; } $isWriteWpconfig = $this->writeWpConfig(); if ($isWriteWpconfig['success'] === false) { return $isWriteWpconfig; } $isUserExists = false; try { global $wpdb; define('WP_INSTALLING', true); require_once __DIR__ . '/wp-load.php'; require_once ABSPATH . 'wp-admin/includes/upgrade.php'; require_once ABSPATH . WPINC . '/class-wpdb.php'; $appFile = $this->meta->appFile; $siteUrl = str_replace('/' . $appFile, '/', wp_guess_url()); define('WP_SITEURL', $siteUrl); $wpdb->suppress_errors(true); $isUserExists = username_exists($siteData['siteuser']); $wpdb->suppress_errors(false); ignore_user_abort(true); wp_install($siteData['sitetitle'], $siteData['siteuser'], $siteData['siteemail'], false, null, $siteData['sitepass']); $isInstallComplete = $this->installComplete(); if ($isInstallComplete['success'] === false) { return $isInstallComplete; } } catch (\Throwable $e) { return ['success' => false, 'data' => $e->getMessage(), 'saveLog' => $e, 'saveLogId' => __METHOD__]; } $text = 'WordPress installation was successful'; if ($isUserExists !== false) { $text .= "\nUser already exists. Password inherited."; return ['success' => true, 'data' => $text, 'isprompt' => 1, 'saveLog' => str_replace("\n", ". ", $text)]; } return ['success' => true, 'data' => $text, 'saveLog' => true, 'saveLogId' => __METHOD__]; } public function installComplete(): array { $isWriteWpconfig = $this->writeWpConfig(); if ($isWriteWpconfig['success'] === false) { return $isWriteWpconfig; } $rootPath = $this->meta->rootPath; $this->useHandle->file->opcacheFlush($rootPath . '/index.php'); if (file_exists($rootPath . '/index-wp.php') && !rename($rootPath . '/index-wp.php', $rootPath . '/index.php')) { return ['success' => false, 'data' => 'Failed to complete WordPress installation', 'saveLog' => true, 'saveLogId' => __METHOD__]; } if (!$this->isWpIndex()) { return ['success' => false, 'data' => 'Something went wrong, missing index.php']; } return ['success' => true, 'data' => 'WordPress installation was successful', 'saveLog' => true, 'saveLogId' => __METHOD__]; } private function locateWpConfigFile() { $upperPath = dirname($this->meta->rootPath); if (file_exists($upperPath . '/wp-config.php') && !file_exists($upperPath . '/wp-settings.php')) { return $upperPath . '/wp-config.php'; } return $this->meta->rootPath . '/wp-config.php'; } private function parseWpConfigForDb() { if (!file_exists($this->wpConfigFile)) { return false; } $content = file_get_contents($this->wpConfigFile, false, null, 0, 8 * 1024); if (empty($content) || strpos($content, 'DB_') === false) { return false; } $pattern = 'define\(\s?(\'|")(DB_(HOST|NAME|USER|PASSWORD))(\'|")\s?,\s?(\'|")(.*?)(\'|")\s?\)\;'; $pattern .= '|define\(\s?(\'|")(MYSQL_CLIENT_FLAGS)(\'|")\s?,\s?(.*?)\s?\)\;'; $pattern .= '|\$(table_prefix)\s?=\s?(\'|")(.*?)(\'|")\;'; if (!preg_match_all('@' . $pattern . '@m', $content, $matches, PREG_SET_ORDER)) { return false; } $dbData = [ 'dbhost' => '', 'dbname' => '', 'dbuser' => '', 'dbpass' => '', 'dbprefix' => 'wp_', 'dbssl' => 0, ]; foreach ($matches as $match) { switch ($match[2]) { case 'DB_HOST': list( $host, $port, $socket, $isIPv6 ) = $this->parseDbHost($match[6]); $dbData['dbhost'] = $host; $dbData['dbport'] = $port; break; case 'DB_NAME': $dbData['dbname'] = $match[6]; break; case 'DB_USER': $dbData['dbuser'] = $match[6]; break; case 'DB_PASSWORD': $dbData['dbpass'] = $match[6]; break; } if (isset($match[9]) && isset($match[11]) && $match[9] === 'MYSQL_CLIENT_FLAGS' && strpos($match[11], 'MYSQLI_CLIENT_SSL') !== false) { $dbData['dbssl'] = 1; } if (isset($match[12]) && !empty($match[14]) && $match[12] === 'table_prefix') { $dbData['dbprefix'] = $match[14]; } } return $dbData; } }
}
namespace {
    if (!getenv('wpstg-restorer-as-library')) {
        (new \WPStagingRestorer())->run();
    }
}
